/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.maven.client.api.model;

import static java.lang.String.join;
import static java.util.Collections.emptyList;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.maven.client.api.util.Preconditions.checkState;

import java.io.File;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;

/**
 * Immutable implementation for {@link MavenConfiguration}.
 *
 * @since 1.0
 */
class ImmutableMavenConfiguration implements MavenConfiguration {

  private final List<RemoteRepository> remoteRepositories;
  private final File localMavenRepositoryLocation;
  private final Optional<File> userSettingsFileOptional;
  private final Optional<File> globalSettingsFileOptional;
  private boolean forcePolicyUpdateNever;
  private boolean forcePolicyUpdateAlways;
  private boolean offlineMode;
  private boolean ignoreArtifactDescriptorRepositories;
  // Should not use Optional type here, as this field was added on later versions and will be null
  // when deserializing a previous version of this instance.
  private final File settingsSecurityFile;
  private final List<String> activeProfiles;
  private final List<String> inactiveProfiles;
  private final Properties userProperties;
  private String globalChecksumPolicy;

  /**
   * Creates an {@link ImmutableMavenConfiguration}
   *
   * @param localMavenRepositoryLocation the maven local repository location.
   * @param remoteRepositories list of remote maven repositories to use for discovering artifacts.
   * @param userSettingsFileOptional the user settings.xml file location.
   * @param globalSettingsFileOptional the global settings.xml file location.
   * @param forcePolicyUpdateNever if {@code true}, regardless of the settings.xml configuration, for all repositories, the update policy
   *        will be configured to never update artifact.
   * @param forcePolicyUpdateAlways if {@code true}, regardless of the settings.xml configuration, for all repositories, the update policy
   *        will be configured to always update artifact.
   * @param offlineMode if {@code true} , regardless of the remote repositories configured it will only use local repository for resolutions.
   * @param ignoreArtifactDescriptorRepositories {@code true} to ignore additional repositories from artifact descriptors,
   *        {@code false} to merge those with the originally specified remote repositories.
   * @param settingsSecurityFile the settings-security.xml file location.
   * @param activeProfiles identifiers of those profiles that should be activated by explicit demand.
   * @param inactiveProfiles identifiers of those profiles that should be deactivated by explicit demand.
   * @param userProperties user properties to use for interpolation and profile activation. The user properties have been
   *        configured directly by the user on his discretion.
   * @param globalChecksumPolicy a checksum policy to be applied to all the remote repositories regardless of their checksum policy defined.
   *
   */
  public ImmutableMavenConfiguration(File localMavenRepositoryLocation, List<RemoteRepository> remoteRepositories,
                                     Optional<File> userSettingsFileOptional, Optional<File> globalSettingsFileOptional,
                                     boolean forcePolicyUpdateNever, boolean forcePolicyUpdateAlways, boolean offlineMode,
                                     boolean ignoreArtifactDescriptorRepositories, File settingsSecurityFile,
                                     List<String> activeProfiles, List<String> inactiveProfiles,
                                     Properties userProperties, String globalChecksumPolicy) {
    Set<String> uniqueIdList = remoteRepositories.stream().map(RemoteRepository::getId).collect(toSet());
    checkState(uniqueIdList.size() == remoteRepositories.size(),
               "The configured set of remote repositories do not have unique IDs");
    this.localMavenRepositoryLocation = localMavenRepositoryLocation;
    this.remoteRepositories = remoteRepositories;
    this.userSettingsFileOptional = userSettingsFileOptional;
    this.globalSettingsFileOptional = globalSettingsFileOptional;
    this.forcePolicyUpdateNever = forcePolicyUpdateNever;
    this.forcePolicyUpdateAlways = forcePolicyUpdateAlways;
    this.offlineMode = offlineMode;
    this.ignoreArtifactDescriptorRepositories = ignoreArtifactDescriptorRepositories;
    this.settingsSecurityFile = settingsSecurityFile;
    this.activeProfiles = activeProfiles != null ? activeProfiles : emptyList();
    this.inactiveProfiles = inactiveProfiles != null ? inactiveProfiles : emptyList();
    this.userProperties = userProperties;
    this.globalChecksumPolicy = globalChecksumPolicy;
  }

  @Override
  public File getLocalMavenRepositoryLocation() {
    return localMavenRepositoryLocation;
  }

  @Override
  public List<RemoteRepository> getMavenRemoteRepositories() {
    return remoteRepositories;
  }

  @Override
  public Optional<File> getUserSettingsLocation() {
    return userSettingsFileOptional;
  }

  @Override
  public Optional<File> getGlobalSettingsLocation() {
    return globalSettingsFileOptional;
  }

  @Override
  public Optional<File> getSettingsSecurityLocation() {
    return ofNullable(settingsSecurityFile);
  }

  @Override
  public boolean getForcePolicyUpdateNever() {
    return forcePolicyUpdateNever;
  }

  @Override
  public boolean getForcePolicyUpdateAlways() {
    return forcePolicyUpdateAlways;
  }

  @Override
  public String getGlobalChecksumPolicy() {
    return globalChecksumPolicy;
  }

  @Override
  public boolean getOfflineMode() {
    return offlineMode;
  }

  @Override
  public boolean getIgnoreArtifactDescriptorRepositories() {
    return ignoreArtifactDescriptorRepositories;
  }

  @Override
  public Optional<Properties> getUserProperties() {
    return ofNullable(userProperties);
  }

  @Override
  public Optional<List<String>> getActiveProfiles() {
    return ofNullable(activeProfiles);
  }

  @Override
  public Optional<List<String>> getInactiveProfiles() {
    return ofNullable(inactiveProfiles);
  }

  @Override
  public String toString() {
    return "ImmutableMavenConfiguration{" +
        "remoteRepositories=" + repositoriesToString() +
        ", localMavenRepositoryLocation=" + localMavenRepositoryLocation +
        ", userSettingsFileOptional=" + userSettingsFileOptional +
        (settingsSecurityFile != null ? ", settingsSecurityFile=" + settingsSecurityFile : "") +
        ", globalSettingsFileOptional=" + globalSettingsFileOptional +
        ", forcePolicyUpdateNever=" + forcePolicyUpdateNever +
        ", forcePolicyUpdateAlways=" + forcePolicyUpdateAlways +
        ", offlineMode=" + offlineMode +
        ", ignoreArtifactDescriptorRepositories=" + ignoreArtifactDescriptorRepositories +
        ", activeProfiles=" + activeProfiles +
        ", inactiveProfiles=" + inactiveProfiles +
        ", globalChecksumPolicy=" + globalChecksumPolicy +
        '}';
  }

  private String repositoriesToString() {
    return join(",\n", remoteRepositories.stream().map(RemoteRepository::toString).collect(toList()));
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    ImmutableMavenConfiguration that = (ImmutableMavenConfiguration) o;

    if (forcePolicyUpdateNever != that.forcePolicyUpdateNever) {
      return false;
    }
    if (forcePolicyUpdateAlways != that.forcePolicyUpdateAlways) {
      return false;
    }
    if (globalChecksumPolicy != that.globalChecksumPolicy) {
      return false;
    }
    if (offlineMode != that.offlineMode) {
      return false;
    }
    if (ignoreArtifactDescriptorRepositories != that.ignoreArtifactDescriptorRepositories) {
      return false;
    }
    if (!remoteRepositories.equals(that.remoteRepositories)) {
      return false;
    }
    if (!localMavenRepositoryLocation.equals(that.localMavenRepositoryLocation)) {
      return false;
    }
    if (!userSettingsFileOptional.equals(that.userSettingsFileOptional)) {
      return false;
    }
    if (!globalSettingsFileOptional.equals(that.globalSettingsFileOptional)) {
      return false;
    }
    if (!Objects.equals(settingsSecurityFile, that.settingsSecurityFile)) {
      return false;
    }
    if (!Objects.equals(activeProfiles, that.activeProfiles)) {
      return false;
    }
    if (!Objects.equals(inactiveProfiles, that.inactiveProfiles)) {
      return false;
    }
    return Objects.equals(userProperties, that.userProperties);
  }

  @Override
  public int hashCode() {
    int result = remoteRepositories.hashCode();
    result = 31 * result + localMavenRepositoryLocation.hashCode();
    result = 31 * result + userSettingsFileOptional.hashCode();
    result = 31 * result + globalSettingsFileOptional.hashCode();
    result = 31 * result + (forcePolicyUpdateNever ? 1 : 0);
    result = 31 * result + (forcePolicyUpdateAlways ? 1 : 0);
    result = 31 * result + (offlineMode ? 1 : 0);
    result = 31 * result + (ignoreArtifactDescriptorRepositories ? 1 : 0);
    result = 31 * result + (settingsSecurityFile != null ? settingsSecurityFile.hashCode() : 0);
    result = 31 * result + (activeProfiles != null ? activeProfiles.hashCode() : 0);
    result = 31 * result + (inactiveProfiles != null ? inactiveProfiles.hashCode() : 0);
    result = 31 * result + (userProperties != null ? userProperties.hashCode() : 0);
    result = 31 * result + (globalChecksumPolicy != null ? globalChecksumPolicy.hashCode() : 0);
    return result;
  }
}
