/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.maven.client.api.model;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

import org.mule.api.annotation.NoImplement;
import org.mule.maven.client.api.MavenClient;

import java.io.File;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

/**
 * Configuration settings for a {@link MavenClient}.
 *
 * @since 1.0
 */
@NoImplement
public interface MavenConfiguration {

  /**
   * @return {@link File} with the location of the local maven repository. It has precedence over the Maven {@code settings}
   * {@code localRepository}.
   */
  File getLocalMavenRepositoryLocation();

  /**
   * @return a list of remote repository to lookup for artifacts. May be empty. It has precedence over the Maven {@code settings}
   * {@code remoteRepositories}. If a remote repository is defined within authentication the authentication credentials from this remote
   * repository will be considered and ignored the ones (if any) on Maven {@code settings} file.
   * <p/>
   * On the other hand if the remote repository does not define authentication credentials and Maven {@code settings} has a {@code server}
   * entry with authentication for the same repository id, the one from {@code settings} file will be considered.
   * <p/>
   * If a Maven {@code settings} file is defined with proxies and mirrors for repositories those will also be applied to the remote
   * repositories declared here. This will allow to use proxy even with default repositories declared programatically.
   */
  List<RemoteRepository> getMavenRemoteRepositories();

  /**
   * @return the user settings.xml file location. Any server settings defined on settings will override the authentication defined for an
   * explicit remote repository declared as {{@link #getMavenRemoteRepositories()}}.
   */
  Optional<File> getUserSettingsLocation();

  /**
   * @return the global settings.xml file location. Any server settings defined on settings will override the authentication defined for an
   * explicit remote repository declared as {{@link #getMavenRemoteRepositories()}}.
   */
  Optional<File> getGlobalSettingsLocation();

  /**
   * @return the global {@code secure-settings.xml} file location to decrypt the password for severs defined on settings.xml.
   */
  Optional<File> getSettingsSecurityLocation();

  /**
   * @return if set to {@code true} it will set a global update policy for remote repositories to UPDATE_POLICY_NEVER. By default
   *         it is set to {@code false}. This is the equivalent to {@code -nsu} option for Maven CLI.
   */
  boolean getForcePolicyUpdateNever();

  /**
   * @return if set to {@code true} it will set a global update policy for remote repositories to UPDATE_POLICY_ALWAYS. By default
   *         it is set to {@code false}. This is the equivalent to {@code -U} option for Maven CLI.
   */
  boolean getForcePolicyUpdateAlways();

  /**
   * @return a global checksum policy to be applied to all the remote repositories. If not set, checksum policy declared on each repository will be taken into account.
   * @see RepositoryPolicy
   */
  String getGlobalChecksumPolicy();

  /**
   * @return {@code true} to ignore additional repositories from artifact descriptors, {@code false} to merge those with the originally specified remote repositories.
   * Default value is {@code true}.
   */
  boolean getIgnoreArtifactDescriptorRepositories();

  /**
   * @return true if resolution should work in off line mode, meaning that no remote repositories would be accessed and only local repository.
   */
  boolean getOfflineMode();

  /**
   * Gets the identifiers of those profiles that should be activated by explicit demand.
   *
   * @return The identifiers of those profiles to activate, never {@code null}.
   */
  Optional<List<String>> getActiveProfiles();

  /**
   * Gets the identifiers of those profiles that should be deactivated by explicit demand.
   *
   * @return The identifiers of those profiles to deactivate, never {@code null}.
   */
  Optional<List<String>> getInactiveProfiles();

  /**
   * Gets the user properties to use for interpolation and profile activation. The user properties have been
   * configured directly by the user on his discretion.
   *
   * @return The user properties, never {@code null}.
   */
  Optional<Properties> getUserProperties();

  /**
   * @return a {@link MavenConfigurationBuilder} to build a {@link MavenConfiguration}.
   */
  static MavenConfigurationBuilder newMavenConfigurationBuilder() {
    return new MavenConfigurationBuilder();
  }

  class MavenConfigurationBuilder {

    private File localMavenRepository;
    private final LinkedList<RemoteRepository> remoteRepositories = new LinkedList<>();
    private File userSettingsFile;
    private File globalSettingsFile;
    private File settingsSecurityFile;
    private boolean forcePolicyUpdateNever = false;
    private boolean forcePolicyUpdateAlways = false;
    private boolean offlineMode = false;
    private boolean ignoreArtifactDescriptorRepositories = true;
    private List<String> activeProfiles;
    private List<String> inactiveProfiles;
    private Properties userProperties;
    private String globalChecksumPolicy;

    /**
     * @param localMavenRepository the local maven repository location.
     * @return this
     */
    public MavenConfigurationBuilder localMavenRepositoryLocation(File localMavenRepository) {
      requireNonNull(localMavenRepository, "localMavenRepository cannot be null");
      this.localMavenRepository = localMavenRepository;
      return this;
    }

    /**
     * Adds a new remote repository to use for discovering artifacts. The order in which the remote repositories are added it's
     * going to be the same order in which they will be used to discover artifacts.
     *
     * @param remoteRepository a remote maven repository
     * @return this
     */
    public MavenConfigurationBuilder remoteRepository(RemoteRepository remoteRepository) {
      requireNonNull(remoteRepository, "remoteRepository cannot be null");
      this.remoteRepositories.addLast(remoteRepository);
      return this;
    }

    /**
     * @param userSettingsFile the user settings.xml file, usually located in USER_HOME/.m2/settings.xml
     * @return this
     */
    public MavenConfigurationBuilder userSettingsLocation(File userSettingsFile) {
      requireNonNull(userSettingsFile, "userSettingsFile cannot be null");
      this.userSettingsFile = userSettingsFile;
      return this;
    }

    /**
     * @param globalSettingsFile the user settings.xml file, usually located in MAVEN_HOME/conf/settings.xml
     * @return this
     */
    public MavenConfigurationBuilder globalSettingsLocation(File globalSettingsFile) {
      requireNonNull(globalSettingsFile, "globalSettingsFile cannot be null");
      this.globalSettingsFile = globalSettingsFile;
      return this;
    }

    /**
     * @param settingsSecurityFile the user settings-security.xml file, usually located in ~/.m2/settings-security.xml
     * @return this
     */
    public MavenConfigurationBuilder settingsSecurityLocation(File settingsSecurityFile) {
      requireNonNull(settingsSecurityFile, "settingsSecurityFile cannot be null");
      this.settingsSecurityFile = settingsSecurityFile;
      return this;
    }

    /**
     * @param forcePolicyUpdateNever if true, regardless of the configuration in the settings.xml for the repositories, the update
     *        policies for artifacts will be to never update them
     * @return this
     */
    public MavenConfigurationBuilder forcePolicyUpdateNever(boolean forcePolicyUpdateNever) {
      requireNonNull(forcePolicyUpdateNever, "forcePolicyUpdateNever cannot be null");
      this.forcePolicyUpdateNever = forcePolicyUpdateNever;
      return this;
    }

    /**
     * @param globalChecksumPolicy a global checksum policy to be applied to all the remote repositories. If not set, checksum policy declared on each repository will be taken into account.
     * @return this
     * @see {@link RepositoryPolicy}
     */
    public MavenConfigurationBuilder globalChecksumPolicy(String globalChecksumPolicy) {
      requireNonNull(globalChecksumPolicy, "globalChecksumPolicy cannot be null");
      this.globalChecksumPolicy = globalChecksumPolicy;
      return this;
    }

    /**
     * @param forcePolicyUpdateAlways if true, regardless of the configuration in the settings.xml for the repositories or remote repositories
     *                                declared through this API, the update policies for artifacts will be to always update them.
     * @return this
     */
    public MavenConfigurationBuilder forcePolicyUpdateAlways(boolean forcePolicyUpdateAlways) {
      requireNonNull(forcePolicyUpdateAlways, "forcePolicyUpdateAlways cannot be null");
      this.forcePolicyUpdateAlways = forcePolicyUpdateAlways;
      return this;
    }

    /**
     * @param offlineMode if true, regardless of the remote repositories configured it will only use local repository for resolutions.
     * @return this
     */
    public MavenConfigurationBuilder offlineMode(boolean offlineMode) {
      requireNonNull(offlineMode, "offlineMode cannot be null");
      this.offlineMode = offlineMode;
      return this;
    }

    /**
     * @param ignoreArtifactDescriptorRepositories if true to ignore additional repositories from artifact descriptors, false to merge those with the originally specified remote repositories.
     * Default value is {@code true}.
     * @return this
     */
    public MavenConfigurationBuilder ignoreArtifactDescriptorRepositories(boolean ignoreArtifactDescriptorRepositories) {
      requireNonNull(ignoreArtifactDescriptorRepositories, "ignoreArtifactDescriptorRepositories cannot be null");
      this.ignoreArtifactDescriptorRepositories = ignoreArtifactDescriptorRepositories;
      return this;
    }

    /**
     * @param userProperties user properties to use for interpolation and profile activation. The user properties have been
     * configured directly by the user on his discretion.
     * @return this.
     */
    public MavenConfigurationBuilder userProperties(Properties userProperties) {
      requireNonNull(userProperties, "userProperties cannot be null");
      this.userProperties = userProperties;
      return this;
    }

    /**
     * @param activeProfiles identifiers of those profiles that should be activated by explicit demand.
     * @return this
     */
    public MavenConfigurationBuilder activeProfiles(List<String> activeProfiles) {
      requireNonNull(activeProfiles, "activeProfiles cannot be null");
      this.activeProfiles = activeProfiles;
      return this;
    }

    /**
     * @param inactiveProfiles identifiers of those profiles that should be deactivated by explicit demand.
     * @return this
     */
    public MavenConfigurationBuilder inactiveProfiles(List<String> inactiveProfiles) {
      requireNonNull(inactiveProfiles, "inactiveProfiles cannot be null");
      this.inactiveProfiles = inactiveProfiles;
      return this;
    }

    /**
     * Builds the {@link MavenConfiguration} object.
     *
     * @return {@link MavenConfiguration} with the value sets.
     */
    public MavenConfiguration build() {
      return new ImmutableMavenConfiguration(localMavenRepository, remoteRepositories, ofNullable(userSettingsFile),
                                             ofNullable(globalSettingsFile), forcePolicyUpdateNever, forcePolicyUpdateAlways,
                                             offlineMode,
                                             ignoreArtifactDescriptorRepositories, settingsSecurityFile, activeProfiles,
                                             inactiveProfiles, userProperties, globalChecksumPolicy);
    }

  }

}
