/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.MavenPomModel;

import java.io.File;
import java.util.List;
import java.util.Optional;

/**
 * Maven client API to work with maven artifacts.
 * <p>
 * To access get an instance of the client use {@link MavenClientProvider#discoverProvider(ClassLoader)} which will resolve the
 * implementation through SPI by loading an instance of {@link MavenClientProvider} that must be in the classpath.
 * 
 * @since 1.0
 */
public interface MavenClient {

  /**
   * @return the maven configuration of this client
   */
  MavenConfiguration getMavenConfiguration();

  /**
   * Resolves for a bundle descriptor with a version range the available versions from remote repositories.
   *
   * @param bundleDescriptor descriptor of the artifact
   * @return a {@link VersionRangeResult}
   */
  VersionRangeResult resolveVersionRange(BundleDescriptor bundleDescriptor);

  /**
   * Resolves the artifact dependencies for an artifact. While resolving the dependencies for an artifact it will also take into
   * account the {@link MavenConfiguration#getActiveProfiles()} and {@link MavenConfiguration#getInactiveProfiles()}.
   * 
   * @param artifactFile                    the artifact file to resolve the dependencies for. It may be an exploded archive
   *                                        following the mule deployable artifact structure or a compressed artifact following
   *                                        the maven conventions.
   * @param includeTestDependencies         true if the test dependencies must be included in the list of direct dependencies,
   *                                        false otherwise.
   * @param includeProvidedDependencies     true if the provided dependencies must be included in the list of direct dependencies,
   *                                        false otherwise.
   * @param localRepositoryLocationSupplier a supplier of the local repository folder location. It may be empty in which case the
   *                                        one of the provided maven configuration will be used.
   * @param temporaryFolder                 to be used if the file is compressed. If present, it will be used as target to extract
   *                                        required files. Otherwise, files will be handled in memory.
   * @return the list of dependencies of the artifact.
   */
  List<BundleDependency> resolveArtifactDependencies(File artifactFile,
                                                     boolean includeTestDependencies,
                                                     boolean includeProvidedDependencies,
                                                     Optional<File> localRepositoryLocationSupplier,
                                                     Optional<File> temporaryFolder);

  /**
   * Resolves the artifact dependencies for an artifact. While resolving the dependencies for an artifact it will also take into
   * account the {@link MavenConfiguration#getActiveProfiles()} and {@link MavenConfiguration#getInactiveProfiles()}.
   *
   * @param artifactFile                    the artifact file to resolve the dependencies for. It may be an exploded archive
   *                                        following the mule deployable artifact structure or a compressed artifact following
   *                                        the maven conventions.
   * @param includeTestDependencies         true if the test dependencies must be included in the list of direct dependencies,
   *                                        false otherwise.
   * @param includeProvidedDependencies     true if the provided dependencies must be included in the list of direct dependencies,
   *                                        false otherwise.
   * @param localRepositoryLocationSupplier a supplier of the local repository folder location. It may be empty in which case the
   *                                        one of the provided maven configuration will be used.
   * @param mavenReactorResolver            {@link MavenReactorResolver} to look for artifacts first.
   * @param temporaryFolder                 to be used if the file is compressed. If present, it will be used as target to extract
   *                                        required files. Otherwise, files will be handled in memory.
   * @return the list of dependencies of the artifact.
   */
  List<BundleDependency> resolveArtifactDependencies(File artifactFile,
                                                     boolean includeTestDependencies,
                                                     boolean includeProvidedDependencies,
                                                     Optional<File> localRepositoryLocationSupplier,
                                                     Optional<MavenReactorResolver> mavenReactorResolver,
                                                     Optional<File> temporaryFolder);

  /**
   * Resolves the artifact dependencies from the list for the given descriptor. It is assumed that all the dependencies defined in
   * the list will be treated as compile scope. The remote repositories and local repository defined from MavenConfiguration will
   * be considered on this resolution. The dependencies will be collected and resolved with the remote repositories defined for
   * the MavenConfiguration. The reactor parameter allows to use this as a way to resolve an artifact dependency such a
   * mule-plugin in the same way that it is resolved when declared in an application/domain (without provided and test scope
   * dependencies for the mule-plugin).
   *
   * In case of resolving mule-plugin dependencies, clients of this API should traverse transitive dependencies on each
   * BundleDependency to collect the list of dependencies.
   *
   * @param dependencies                    list of dependencies to be resolved, all treated as compile scope.
   * @param localRepositoryLocationSupplier a supplier of the local repository folder location. It may be empty in which case the
   *                                        one of the provided maven configuration will be used.
   * @param mavenReactorResolver            {@link MavenReactorResolver} to look for artifacts first.
   * @return the list of the requested dependencies resolved with their transitive dependencies and these also with their
   *         transitive dependencies and so on.
   */
  List<BundleDependency> resolveArtifactDependencies(List<BundleDescriptor> dependencies,
                                                     Optional<File> localRepositoryLocationSupplier,
                                                     Optional<MavenReactorResolver> mavenReactorResolver);

  /**
   * Resolves dependencies of a bundle. {@link MavenConfiguration#getActiveProfiles()} and
   * {@link MavenConfiguration#getInactiveProfiles()} will not be considered when resolving the dependencies and building the pom
   * model.
   * 
   * @param includeTestDependencies true if the test dependencies must be included in the list, false otherwise.
   * @param bundleDescriptor        descriptor of the artifact
   * @return the list of dependencies of the artifact
   */
  List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies, BundleDescriptor bundleDescriptor);

  /**
   * Resolves dependencies of a bundle. {@link MavenConfiguration#getActiveProfiles()} and
   * {@link MavenConfiguration#getInactiveProfiles()} will not be considered when resolving the dependencies and building the pom
   * model.
   *
   * @param includeTestDependencies     true if the test dependencies must be included in the list, false otherwise.
   * @param includeProvidedDependencies true if the provided dependencies must be included in the list of direct dependencies,
   *                                    false otherwise.
   * @param bundleDescriptor            descriptor of the artifact
   * @return the list of dependencies of the artifact
   */
  List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies, boolean includeProvidedDependencies,
                                                             BundleDescriptor bundleDescriptor);

  /**
   * Retrieves the {@link BundleDependency} for a {@link BundleDescriptor}.
   * 
   * @param bundleDescriptor the bundle descriptor.
   * @return the resolved {@link BundleDependency}
   */
  BundleDependency resolveBundleDescriptor(BundleDescriptor bundleDescriptor);

  /**
   * Resolve the effective list of plugins.
   *
   * @param bundleDescriptors a list of plugins to resolve the effective list of plugins considering their dependencies
   * @return the effective list of plugins.
   */
  List<BundleDependency> resolvePluginBundleDescriptorsDependencies(List<BundleDescriptor> bundleDescriptors);

  /**
   * Creates the pom model by reading the pom inside the artifact. It won't solve the effective pom meaning that, for instance,
   * properties defined in parent pom will not be lookup to resolve the pom values.
   * <p>
   * The artifact can be a compressed file or an exploded artifact. The only requirement is that it contains the expected maven
   * structure.
   *
   * @param artifactFile the compressed artifact
   * @return the effective pom model
   */
  MavenPomModel getRawPomModel(File artifactFile);

  /**
   * Creates the effective pom model by reading the pom inside the artifact. Properties defined in the pom file will be resolved.
   * <p>
   * The artifact can be a compressed file or an exploded artifact. The only requirement is that it contains the expected maven
   * structure.
   *
   * @param artifactFile    the compressed artifact
   * @param temporaryFolder a folder to be used for creating temporary files.
   * @return
   */
  MavenPomModel getEffectiveModel(File artifactFile, Optional<File> temporaryFolder);

}
