/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api;

import static java.lang.String.format;

import java.util.Iterator;
import java.util.List;
import java.util.ServiceLoader;

/**
 * Provides a set of utilities to work with artifact versions.
 *
 * @since 2.0
 */
public interface VersionUtils {

  /**
   * Determines whether the specified version satisfies this constraint. In more detail, a version satisfies this constraint if it
   * matches its version range or if this constraint has no version range and the specified version equals the version recommended
   * by the constraint.
   * 
   * @param version     The version to test, must not be {@code null}.
   * @param constraints The versions to test against, must not be {@code null}.
   * @return {@code true} if the specified version satisfies any of the constraints, {@code false} otherwise.
   */
  boolean containsVersion(String version, List<String> constraints) throws IllegalArgumentException;

  /**
   * Discovers a {@link VersionUtils} through SPI.
   * 
   * @return a new version utility.
   * 
   * @since 2.0
   */
  static VersionUtils discoverVersionUtils(ClassLoader classLoader) {
    ServiceLoader<VersionUtils> factories = ServiceLoader.load(VersionUtils.class, classLoader);
    Iterator<VersionUtils> iterator = factories.iterator();
    if (!iterator.hasNext()) {
      throw new IllegalStateException(format("Could not find %s service implementation through SPI",
                                             VersionUtils.class.getName()));
    }
    return iterator.next();
  }

}
