/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api;

import org.mule.maven.client.api.model.MavenConfiguration;

import java.util.Iterator;
import java.util.ServiceLoader;

/**
 * SPI for the API to find the implementation for the {@link MavenClient}
 * 
 * @since 1.0
 */
public interface MavenClientProvider {

  /**
   * Creates a new {@link MavenClient} implementation.
   * 
   * @param mavenConfiguration maven configuration for configuring the client.
   * @return a new {@link MavenClient}
   */
  MavenClient createMavenClient(MavenConfiguration mavenConfiguration);

  /**
   * @return a provider for discovering pom files.
   */
  PomFileSupplierFactory getPomFileSuppliers();

  /**
   * @return a provider for discovering maven repository locations.
   */
  LocalRepositorySupplierFactory getLocalRepositorySuppliers();

  /**
   * @return a provider for discovering maven settings locations.
   */
  SettingsSupplierFactory getSettingsSupplierFactory();

  /**
   * Discovers a {@link MavenClientProvider} through SPI.
   * 
   * @return a new client provider.
   */
  static MavenClientProvider discoverProvider(ClassLoader classLoader) {
    ServiceLoader<MavenClientProvider> factories = ServiceLoader.load(MavenClientProvider.class, classLoader);
    Iterator<MavenClientProvider> iterator = factories.iterator();
    if (!iterator.hasNext()) {
      throw new IllegalStateException(String.format("Could not find %s service implementation through SPI",
                                                    MavenClientProvider.class.getName()));
    }
    return iterator.next();
  }

}
