/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api.model;

import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.mockito.Mockito.mock;

import java.net.MalformedURLException;
import java.net.URL;

import org.junit.Before;
import org.junit.Test;

public class ImmutableRemoteRepositoryTestCase {

  private URL testUrl;
  private Authentication authentication;
  private RepositoryPolicy releasePolicy;
  private RepositoryPolicy snapshotPolicy;

  @Before
  public void setUp() throws MalformedURLException {
    testUrl = new URL("https://example.com/repo");
    authentication = mock(Authentication.class);
    releasePolicy = mock(RepositoryPolicy.class);
    snapshotPolicy = mock(RepositoryPolicy.class);
  }

  @Test
  public void testConstructorAndGetters() {
    ImmutableRemoteRepository repository = new ImmutableRemoteRepository(
                                                                         "repo-id", testUrl, of(authentication), releasePolicy,
                                                                         snapshotPolicy);

    assertThat(repository.getId(), is("repo-id"));
    assertThat(repository.getUrl(), is(testUrl));
    assertThat(repository.getAuthentication(), is(of(authentication)));
    assertThat(repository.getReleasePolicy(), is(of(releasePolicy)));
    assertThat(repository.getSnapshotPolicy(), is(of(snapshotPolicy)));
  }

  @Test
  public void testConstructorWithEmptyAuthenticationAndPolicies() {
    ImmutableRemoteRepository repository = new ImmutableRemoteRepository(
                                                                         "repo-id", testUrl, empty(), null, null);

    assertThat(repository.getId(), is("repo-id"));
    assertThat(repository.getUrl(), is(testUrl));
    assertThat(repository.getAuthentication(), is(empty()));
    assertThat(repository.getReleasePolicy(), is(empty()));
    assertThat(repository.getSnapshotPolicy(), is(empty()));
  }

  @Test
  public void testEqualsAndHashCode() {
    ImmutableRemoteRepository repo1 = new ImmutableRemoteRepository(
                                                                    "repo-id", testUrl, of(authentication), releasePolicy,
                                                                    snapshotPolicy);
    ImmutableRemoteRepository repo2 = new ImmutableRemoteRepository(
                                                                    "repo-id", testUrl, of(authentication), releasePolicy,
                                                                    snapshotPolicy);
    ImmutableRemoteRepository repo3 = new ImmutableRemoteRepository(
                                                                    "repo-id2", testUrl, of(authentication), releasePolicy,
                                                                    snapshotPolicy);

    assertThat(repo1, is(repo2));
    assertThat(repo1.hashCode(), is(repo2.hashCode()));
    assertThat(repo1, is(not(repo3)));
  }

  @Test
  public void testNotEqualsForDifferentId() {
    ImmutableRemoteRepository repo1 = new ImmutableRemoteRepository(
                                                                    "repo-id-1", testUrl, of(authentication), releasePolicy,
                                                                    snapshotPolicy);
    ImmutableRemoteRepository repo2 = new ImmutableRemoteRepository(
                                                                    "repo-id-2", testUrl, of(authentication), releasePolicy,
                                                                    snapshotPolicy);

    assertThat(repo1, is(not(repo2)));
  }

  @Test
  public void testNotEqualsForDifferentUrl() throws MalformedURLException {
    ImmutableRemoteRepository repo1 = new ImmutableRemoteRepository(
                                                                    "repo-id", new URL("https://example.com/repo1"),
                                                                    of(authentication), releasePolicy, snapshotPolicy);
    ImmutableRemoteRepository repo2 = new ImmutableRemoteRepository(
                                                                    "repo-id", new URL("https://example.com/repo2"),
                                                                    of(authentication), releasePolicy, snapshotPolicy);

    assertThat(repo1, is(not(repo2)));
  }

  @Test
  public void testToStringContainsFields() {
    ImmutableRemoteRepository repository = new ImmutableRemoteRepository(
                                                                         "repo-id", testUrl, of(authentication), releasePolicy,
                                                                         snapshotPolicy);

    String toString = repository.toString();
    assertThat(toString, containsString("repo-id"));
    assertThat(toString, containsString("example.com/repo"));
  }
}
