/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api.model;

import static org.mule.maven.client.api.model.RepositoryPolicy.CHECKSUM_POLICY_FAIL;
import static org.mule.maven.client.api.model.RepositoryPolicy.CHECKSUM_POLICY_IGNORE;
import static org.mule.maven.client.api.model.RepositoryPolicy.CHECKSUM_POLICY_WARN;
import static org.mule.maven.client.api.model.RepositoryPolicy.UPDATE_POLICY_DAILY;
import static org.mule.maven.client.api.model.RepositoryPolicy.UPDATE_POLICY_INTERVAL;
import static org.mule.maven.client.api.model.RepositoryPolicy.UPDATE_POLICY_NEVER;
import static org.mule.maven.client.api.model.RepositoryPolicy.newRepositoryPolicyBuilder;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.containsString;

import org.junit.Test;

public class RepositoryPolicyTestCase {

  @Test
  public void testRepositoryPolicyBuilderWhenDefaultValues() {
    RepositoryPolicy policy = newRepositoryPolicyBuilder().build();

    assertThat(policy.isEnabled(), is(true));
    assertThat(policy.getUpdatePolicy(), is(UPDATE_POLICY_DAILY));
    assertThat(policy.getChecksumPolicy(), is(CHECKSUM_POLICY_WARN));
  }

  @Test
  public void testRepositoryPolicyBuilderWhenCustomValues() {
    RepositoryPolicy policy = newRepositoryPolicyBuilder()
        .enabled(false)
        .updatePolicy(UPDATE_POLICY_NEVER)
        .checksumPolicy(CHECKSUM_POLICY_FAIL)
        .build();

    assertThat(policy.isEnabled(), is(false));
    assertThat(policy.getUpdatePolicy(), is(UPDATE_POLICY_NEVER));
    assertThat(policy.getChecksumPolicy(), is(CHECKSUM_POLICY_FAIL));
  }

  @Test
  public void testEqualsWhenSameObject() {
    RepositoryPolicy policy = newRepositoryPolicyBuilder().build();
    assertThat(policy.equals(policy), is(true));
  }

  @Test
  public void testEqualsWhenDifferentObjectsWithSameValues() {
    RepositoryPolicy policy1 = newRepositoryPolicyBuilder()
        .enabled(true)
        .updatePolicy(UPDATE_POLICY_DAILY)
        .checksumPolicy(CHECKSUM_POLICY_WARN)
        .build();

    RepositoryPolicy policy2 = newRepositoryPolicyBuilder()
        .enabled(true)
        .updatePolicy(UPDATE_POLICY_DAILY)
        .checksumPolicy(CHECKSUM_POLICY_WARN)
        .build();

    assertThat(policy1, is(policy2));
  }

  @Test
  public void testEqualsWhenDifferentObjectsWithDifferentValues() {
    RepositoryPolicy policy1 = newRepositoryPolicyBuilder().enabled(true).build();
    RepositoryPolicy policy2 = newRepositoryPolicyBuilder().enabled(false).build();

    assertThat(policy1, is(not(policy2)));
  }

  @Test
  public void testHashCodeConsistency() {
    RepositoryPolicy policy = newRepositoryPolicyBuilder().build();
    int hashCode1 = policy.hashCode();
    int hashCode2 = policy.hashCode();

    assertThat(hashCode1, is(hashCode2));
  }

  @Test
  public void testHashCodeEqualityForEqualObjects() {
    RepositoryPolicy policy1 = newRepositoryPolicyBuilder().build();
    RepositoryPolicy policy2 = newRepositoryPolicyBuilder().build();

    assertThat(policy1.hashCode(), is(policy2.hashCode()));
  }

  @Test
  public void testToStringContainsExpectedValues() {
    RepositoryPolicy policy = newRepositoryPolicyBuilder()
        .enabled(true)
        .updatePolicy(UPDATE_POLICY_INTERVAL)
        .checksumPolicy(CHECKSUM_POLICY_IGNORE)
        .build();

    assertThat(policy.toString(), containsString("enabled=true"));
    assertThat(policy.toString(), containsString("checksums=ignore"));
    assertThat(policy.toString(), containsString("updates=interval"));
  }

  @Test
  public void testToStringWhenDifferentValues() {
    RepositoryPolicy policy = newRepositoryPolicyBuilder()
        .enabled(false)
        .updatePolicy(UPDATE_POLICY_NEVER)
        .checksumPolicy(CHECKSUM_POLICY_FAIL)
        .build();

    assertThat(policy.toString(), containsString("enabled=false"));
    assertThat(policy.toString(), containsString("checksums=fail"));
    assertThat(policy.toString(), containsString("updates=never"));
  }
}
