/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api.model;

import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static java.util.Arrays.asList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

import org.junit.Before;
import org.junit.Test;

public class ImmutableMavenConfigurationTestCase {

  private ImmutableMavenConfiguration config;
  private File localRepo;
  private List<RemoteRepository> remoteRepos;
  private Optional<File> userSettings;
  private Optional<File> globalSettings;
  private File settingsSecurity;
  private List<String> activeProfiles;
  private List<String> inactiveProfiles;
  private Properties userProperties;
  private String globalChecksumPolicy;

  @Before
  public void setUp() throws MalformedURLException {
    localRepo = new File("/path/to/local/repo");
    remoteRepos = asList(newRemoteRepositoryBuilder().id("id1").url(new URL("https://")).build(),
                         newRemoteRepositoryBuilder().id("id2").url(new URL("https://")).build());
    userSettings = Optional.of(new File("/path/to/user/settings.xml"));
    globalSettings = Optional.of(new File("/path/to/global/settings.xml"));
    settingsSecurity = new File("/path/to/settings-security.xml");
    activeProfiles = asList("profile1", "profile2");
    inactiveProfiles = asList("profile3");
    userProperties = new Properties();
    userProperties.setProperty("key", "value");
    globalChecksumPolicy = "STRICT";

    config = new ImmutableMavenConfiguration(
                                             localRepo, remoteRepos, userSettings, globalSettings,
                                             true, false, true, false, settingsSecurity,
                                             activeProfiles, inactiveProfiles, userProperties,
                                             globalChecksumPolicy);
  }

  @Test
  public void testGetLocalMavenRepositoryLocation() {
    assertThat(config.getLocalMavenRepositoryLocation(), is(localRepo));
  }

  @Test
  public void testGetMavenRemoteRepositories() {
    assertThat(config.getMavenRemoteRepositories(), is(remoteRepos));
  }

  @Test
  public void testGetUserSettingsLocation() {
    assertThat(config.getUserSettingsLocation(), is(userSettings));
  }

  @Test
  public void testGetGlobalSettingsLocation() {
    assertThat(config.getGlobalSettingsLocation(), is(globalSettings));
  }

  @Test
  public void testGetSettingsSecurityLocation() {
    assertThat(config.getSettingsSecurityLocation(), is(Optional.of(settingsSecurity)));
  }

  @Test
  public void testGetForcePolicyUpdateNever() {
    assertThat(config.getForcePolicyUpdateNever(), is(true));
  }

  @Test
  public void testGetForcePolicyUpdateAlways() {
    assertThat(config.getForcePolicyUpdateAlways(), is(false));
  }

  @Test
  public void testGetOfflineMode() {
    assertThat(config.getOfflineMode(), is(true));
  }

  @Test
  public void testGetIgnoreArtifactDescriptorRepositories() {
    assertThat(config.getIgnoreArtifactDescriptorRepositories(), is(false));
  }

  @Test
  public void testGetActiveProfiles() {
    assertThat(config.getActiveProfiles(), is(Optional.of(activeProfiles)));
  }

  @Test
  public void testGetInactiveProfiles() {
    assertThat(config.getInactiveProfiles(), is(Optional.of(inactiveProfiles)));
  }

  @Test
  public void testGetUserProperties() {
    assertThat(config.getUserProperties(), is(Optional.of(userProperties)));
  }

  @Test
  public void testGetGlobalChecksumPolicy() {
    assertThat(config.getGlobalChecksumPolicy(), is(globalChecksumPolicy));
  }

  @Test(expected = IllegalStateException.class)
  public void testDuplicateRemoteRepositoryIdsThrowsException() throws MalformedURLException {
    List<RemoteRepository> duplicateRepos =
        asList(newRemoteRepositoryBuilder().id("id1").url(new URL("https://")).build(),
               newRemoteRepositoryBuilder().id("id1").url(new URL("https://")).build());
    new ImmutableMavenConfiguration(
                                    localRepo, duplicateRepos, userSettings, globalSettings,
                                    true, false, true, false, settingsSecurity,
                                    activeProfiles, inactiveProfiles, userProperties,
                                    globalChecksumPolicy);
  }

  @Test
  public void testToString() {
    assertThat(config.toString(), containsString("ImmutableMavenConfiguration"));
  }

  @Test
  public void testRepositoriesToString() {
    assertThat(config.getMavenRemoteRepositories().toString(), containsString("id1"));
  }

  @Test
  public void testEqualsAndHashCode() {
    ImmutableMavenConfiguration sameConfig = new ImmutableMavenConfiguration(
                                                                             localRepo, remoteRepos, userSettings, globalSettings,
                                                                             true, false, true, false, settingsSecurity,
                                                                             activeProfiles, inactiveProfiles, userProperties,
                                                                             globalChecksumPolicy);
    assertThat(config, is(sameConfig));
    assertThat(config.hashCode(), is(sameConfig.hashCode()));
  }
}
