/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.lang.String.format;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.mule.maven.client.internal.util.FileUtils.loadFileContentFrom;
import static org.mule.maven.client.internal.util.MavenUtils.getPomUrlFromJar;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.PomFileSupplierFactory;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.function.Supplier;

public class DefaultPomFileSupplierFactory implements PomFileSupplierFactory {

  /**
   * Pom file supplier that retrieves the pom file location from an artifact folder.
   * <p>
   * It is expected that the artifact folows the maven convention for the location of the pom file.
   *
   * @param artifactFolder the folder with the uncompressed maven artifact.
   * @param artifactBundleDescriptor the descriptor of the maven artifact.
   * @return a supplier that retrieves the pom file location from an artifact folder.
   */
  public Supplier<File> uncompressPomArtifactSupplier(File artifactFolder, BundleDescriptor artifactBundleDescriptor) {
    return () -> {
      String pathToPom = Paths
          .get("META-INF", "maven", artifactBundleDescriptor.getGroupId(), artifactBundleDescriptor.getArtifactId(), "pom.xml")
          .toString();
      return new File(artifactFolder, pathToPom);
    };
  }


  /**
   * Pom file supplier that retrieves the pom file location from a compressed artifact.
   * <p>
   * It will use a temporary folder to be able to create a {@link File} object to point to.
   *
   * @param artifactFile the compressed artifact file.
   * @param artifactBundleDescriptor the bundle descriptor.
   * @param temporaryFolder the temporary folder to use to create the pom file.
   * @return a supplier that retrieves the pom file location from the compressed artifact.
   */
  public Supplier<File> compressedArtifactSupplier(File artifactFile, BundleDescriptor artifactBundleDescriptor,
                                                   File temporaryFolder) {
    return () -> {
      try {
        URL pomUrl = getPomUrlFromJar(artifactFile);
        Optional<byte[]> pomContentOptional = loadFileContentFrom(pomUrl);
        byte[] pomBytes =
            pomContentOptional.orElseThrow(() -> new RuntimeException(format("No pom file found in %s", artifactFile)));
        File pomLocation = new File(temporaryFolder, format(".mule/plugins/%s/%s/%s/%s-%s",
                                                            artifactBundleDescriptor.getGroupId(),
                                                            artifactBundleDescriptor.getArtifactId(),
                                                            artifactBundleDescriptor.getVersion(),
                                                            artifactBundleDescriptor.getArtifactId(),
                                                            artifactBundleDescriptor.getVersion()));
        copyInputStreamToFile(new ByteArrayInputStream(pomBytes), pomLocation);
        return pomLocation;
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    };
  }



}
