/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static org.mule.maven.client.internal.AetherMavenClient.MULE_PLUGIN_CLASSIFIER;
import static org.mule.maven.client.internal.AetherMavenClient.artifactToBundleDependency;
import org.mule.maven.client.api.model.BundleDependency;

import java.util.ArrayList;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.aether.graph.DependencyNode;

/**
 * Visitor that records all the {@link BundleDependency}s of a mule artifact.
 * <p>
 * By calling the method {@code {@link #visit(DependencyNode)}} over the mule artifact {@link DependencyNode} it will record all
 * the dependencies which can later be retrieved by invoking {@code {@link #getBundleDependencies()}}
 * 
 * @since 1.0
 */
public class MuleArtifactDependencyVisitor {

  private List<BundleDependency> bundleDependencies = new ArrayList<>();
  private List<DependencyNode> foundPlugins = new ArrayList<>();

  private final Map<DependencyNode, Object> visitedNodes = new IdentityHashMap<>(512);


  /**
   * Goes through the graph of dependencies and records the list of dependencies of a mule artifact that may depend on plugins. It
   * will not record the dependencies of the plugins since those are not going to be available for the artifact.
   * 
   * @param node the aether dependency node of the mule artifact.
   */
  public void visit(DependencyNode node) {
    if (!setVisited(node)) {
      return;
    }
    if (isPlugin(node)) {
      if (foundPlugins.contains(node)) {
        return;
      }
      foundPlugins.add(node);
    }
    addDependency(node);
    visitChildren(node);
  }

  private boolean isPlugin(DependencyNode node) {
    return node != null && node.getArtifact() != null && MULE_PLUGIN_CLASSIFIER.equals(node.getArtifact().getClassifier());
  }

  private void visitChildren(DependencyNode node) {
    if (isPlugin(node)) {
      for (DependencyNode child : node.getChildren()) {
        if (isPlugin(child)) {
          visit(child);
        }
      }
    } else {
      for (DependencyNode child : node.getChildren()) {
        visit(child);
      }
    }
  }

  private void addDependency(DependencyNode node) {
    if (node.getArtifact() != null && node.getArtifact().getFile() != null) {
      bundleDependencies.add(artifactToBundleDependency(node.getArtifact(), node.getDependency().getScope()));
    }
  }

  /**
   * Marks the specified node as being visited and determines whether the node has been visited before.
   *
   * @param node The node being visited, must not be {@code null}.
   * @return {@code true} if the node has not been visited before, {@code false} if the node was already visited.
   */
  private boolean setVisited(DependencyNode node) {
    return visitedNodes.put(node, Boolean.TRUE) == null;
  }

  /**
   * @return collection of dependencies of the mule artifact
   */
  public List<BundleDependency> getBundleDependencies() {
    return bundleDependencies;
  }
}
