/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.eclipse.aether.repository.RepositoryPolicy.CHECKSUM_POLICY_IGNORE;
import static org.eclipse.aether.repository.RepositoryPolicy.UPDATE_POLICY_NEVER;
import static org.eclipse.aether.resolution.ArtifactDescriptorPolicy.STRICT;
import static org.eclipse.aether.resolution.ResolutionErrorPolicy.CACHE_NOT_FOUND;
import static org.slf4j.LoggerFactory.getLogger;

import java.io.File;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.function.Consumer;

import org.apache.maven.repository.internal.DefaultArtifactDescriptorReader;
import org.apache.maven.repository.internal.DefaultVersionRangeResolver;
import org.apache.maven.repository.internal.DefaultVersionResolver;
import org.apache.maven.repository.internal.MavenRepositorySystemUtils;
import org.eclipse.aether.DefaultRepositoryCache;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.collection.DependencyGraphTransformer;
import org.eclipse.aether.connector.basic.BasicRepositoryConnectorFactory;
import org.eclipse.aether.impl.ArtifactDescriptorReader;
import org.eclipse.aether.impl.DefaultServiceLocator;
import org.eclipse.aether.impl.VersionRangeResolver;
import org.eclipse.aether.impl.VersionResolver;
import org.eclipse.aether.internal.impl.DefaultRepositorySystem;
import org.eclipse.aether.repository.AuthenticationSelector;
import org.eclipse.aether.repository.LocalRepository;
import org.eclipse.aether.repository.WorkspaceReader;
import org.eclipse.aether.spi.connector.RepositoryConnectorFactory;
import org.eclipse.aether.spi.connector.transport.TransporterFactory;
import org.eclipse.aether.transfer.TransferCancelledException;
import org.eclipse.aether.transfer.TransferEvent;
import org.eclipse.aether.transfer.TransferListener;
import org.eclipse.aether.transport.file.FileTransporterFactory;
import org.eclipse.aether.transport.http.HttpTransporterFactory;
import org.eclipse.aether.util.graph.transformer.ChainedDependencyGraphTransformer;
import org.eclipse.aether.util.repository.SimpleResolutionErrorPolicy;
import org.slf4j.Logger;

/**
 * Holds the state for aether repository state for the resolution of dependencies of a particular artifact.
 */
public class AetherRepositoryState {

  private static final Logger LOGGER = getLogger(AetherRepositoryState.class);

  private DefaultRepositorySystemSession session;
  private RepositorySystem system;

  public AetherRepositoryState(File localRepositoryFolder, Optional<WorkspaceReader> workspaceReader,
                               Optional<AuthenticationSelector> authenticationSelector, boolean updatePolicyNever,
                               boolean offline,
                               Consumer<DefaultRepositorySystemSession> sessionConfigurator) {
    this(localRepositoryFolder, workspaceReader, offline, true,
         authenticationSelector, updatePolicyNever, sessionConfigurator);
  }

  public AetherRepositoryState(File localRepositoryFolder, Optional<WorkspaceReader> workspaceReader, boolean offline,
                               boolean ignoreArtifactDescriptorRepositories,
                               Optional<AuthenticationSelector> authenticationSelector, boolean updatePolicyNever) {
    this(localRepositoryFolder, workspaceReader, offline, ignoreArtifactDescriptorRepositories, authenticationSelector,
         updatePolicyNever, _x -> {
         });
  }

  public AetherRepositoryState(File localRepositoryFolder, Optional<WorkspaceReader> workspaceReader, boolean offline,
                               boolean ignoreArtifactDescriptorRepositories,
                               Optional<AuthenticationSelector> authenticationSelector, boolean updatePolicyNever,
                               Consumer<DefaultRepositorySystemSession> sessionConfigurator) {
    createRepositorySession(localRepositoryFolder, workspaceReader, offline, ignoreArtifactDescriptorRepositories,
                            authenticationSelector, updatePolicyNever, sessionConfigurator);
  }

  private void createRepositorySession(File localRepositoryFolder, Optional<WorkspaceReader> workspaceReader, boolean offline,
                                       boolean ignoreArtifactDescriptorRepositories,
                                       Optional<AuthenticationSelector> authenticationSelector,
                                       boolean updatePolicyNever,
                                       Consumer<DefaultRepositorySystemSession> sessionConfigurator) {
    session = newDefaultRepositorySystemSession(updatePolicyNever, sessionConfigurator);
    RepositorySystem repositorySystem = createRepositorySystem();

    session.setLocalRepositoryManager(repositorySystem
        .newLocalRepositoryManager(session, new LocalRepository(localRepositoryFolder)));
    session.setOffline(offline);
    //TODO gfernandes implement support for proxies and mirrors
    //session.setProxySelector()
    //session.setMirrorSelector()
    authenticationSelector.ifPresent(session::setAuthenticationSelector);

    session.setDependencyGraphTransformer(new ChainedDependencyGraphTransformer(new DependencyGraphTransformer[] {
        new MulePluginDependencyGraphTransformer(), session.getDependencyGraphTransformer()}));

    session.setArtifactDescriptorPolicy((session, request) -> STRICT);
    session.setIgnoreArtifactDescriptorRepositories(ignoreArtifactDescriptorRepositories);
    workspaceReader.ifPresent(session::setWorkspaceReader);
    system = repositorySystem;
  }

  private RepositorySystem createRepositorySystem() {
    DefaultServiceLocator locator = new DefaultServiceLocator();
    locator.addService(RepositoryConnectorFactory.class, BasicRepositoryConnectorFactory.class);
    locator.addService(TransporterFactory.class, FileTransporterFactory.class);
    locator.addService(RepositorySystem.class, DefaultRepositorySystem.class);
    locator.addService(TransporterFactory.class, HttpTransporterFactory.class);
    locator.addService(VersionResolver.class, DefaultVersionResolver.class);
    locator.addService(VersionRangeResolver.class, DefaultVersionRangeResolver.class);
    locator.addService(ArtifactDescriptorReader.class, DefaultArtifactDescriptorReader.class);
    locator.setErrorHandler(new DefaultServiceLocator.ErrorHandler() {

      @Override
      public void serviceCreationFailed(Class<?> type, Class<?> impl, Throwable exception) {
        LOGGER.warn(exception.getMessage());
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug(exception.getMessage(), exception);
        }
      }
    });

    // Customization of Aether Services
    loadAetherServiceRegister().ifPresent(aetherServiceRegister -> aetherServiceRegister.registerServices(locator));

    return locator.getService(RepositorySystem.class);
  }

  /**
   * Use the ServiceLoader mechanism to load (if present) a {@link AetherServiceRegister}.
   *
   * @return {@link AetherServiceRegister}
   */
  private Optional<AetherServiceRegister> loadAetherServiceRegister() {
    for (AetherServiceRegister serviceRegister : ServiceLoader.load(AetherServiceRegister.class)) {
      return of(serviceRegister);
    }
    return empty();
  }

  public DefaultRepositorySystemSession getSession() {
    return session;
  }

  public RepositorySystem getSystem() {
    return system;
  }

  private static DefaultRepositorySystemSession newDefaultRepositorySystemSession(boolean updatePolicyNever,
                                                                                  Consumer<DefaultRepositorySystemSession> sessionConfigurator) {
    final DefaultRepositorySystemSession session = MavenRepositorySystemUtils.newSession();
    session.setResolutionErrorPolicy(new SimpleResolutionErrorPolicy(CACHE_NOT_FOUND));
    session.setCache(new DefaultRepositoryCache());
    if (updatePolicyNever) {
      session.setUpdatePolicy(UPDATE_POLICY_NEVER);
    }
    session.setChecksumPolicy(CHECKSUM_POLICY_IGNORE);
    session.setTransferListener(new LoggingTransferListener());
    sessionConfigurator.accept(session);
    return session;
  }

  private static final class LoggingTransferListener implements TransferListener {

    @Override
    public void transferSucceeded(TransferEvent event) {
      LOGGER.info("Transfer {} for '{}' from {}", event.getType(), event.getResource().getResourceName(),
                  event.getResource().getRepositoryUrl());
    }

    @Override
    public void transferStarted(TransferEvent event) throws TransferCancelledException {
      LOGGER.trace("Transfer {} for '{}'", event.getType(), event.getResource().getResourceName());
    }

    @Override
    public void transferProgressed(TransferEvent event) throws TransferCancelledException {
      LOGGER.trace("Transfer {} for '{}'", event.getType(), event.getResource().getResourceName());
    }

    @Override
    public void transferInitiated(TransferEvent event) throws TransferCancelledException {
      LOGGER.trace("Transfer {} for '{}'", event.getType(), event.getResource().getResourceName());
    }

    @Override
    public void transferFailed(TransferEvent event) {
      LOGGER.warn("Transfer {} for '{}' ({})", event.getType(), event.getResource(), event.getException());
    }

    @Override
    public void transferCorrupted(TransferEvent event) throws TransferCancelledException {
      LOGGER.warn("Transfer {} for '{}' ({})", event.getType(), event.getResource(), event.getException());
    }

  }

}
