/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.client.internal.AetherResolutionContext;

import com.google.common.io.Files;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import org.apache.maven.repository.internal.MavenRepositorySystemUtils;
import org.eclipse.aether.repository.Authentication;
import org.eclipse.aether.repository.AuthenticationContext;
import org.eclipse.aether.repository.Proxy;
import org.junit.Test;

public class AetherResolutionContextTestCase {

  private static final String MAVEN_CENTRAL = "https://repo.maven.apache.org/maven2/";
  private static final String DOWNLOADS_PLANETMIRROR_URL = "http://downloads.planetmirror.com/pub/maven2";
  private static final String MULESOFT_PRIVATE_URL = "https://repository.mulesoft.org/nexus/content/repositories/private/";

  @Test
  public void mirrorFromSettings() throws MalformedURLException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-mirror.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(Files.createTempDir())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(MAVEN_CENTRAL))
            .id("central")
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getMirrorSelector().isPresent(), is(true));
    final org.eclipse.aether.repository.RemoteRepository mirror = context.getMirrorSelector().get()
        .getMirror(new org.eclipse.aether.repository.RemoteRepository.Builder("central", "default", MAVEN_CENTRAL).build());
    assertThat(mirror, notNullValue());
    assertThat(mirror.getUrl(), is(DOWNLOADS_PLANETMIRROR_URL));

    // Explicitly defined remoteRepositories to MavenConfiguration should also have been configured with mirror and authenticator
    assertThat(context.getRemoteRepositories().size(), is(1));
    org.eclipse.aether.repository.RemoteRepository repositoryMirrored = context.getRemoteRepositories().get(0);
    assertThat(repositoryMirrored.getUrl(), is(DOWNLOADS_PLANETMIRROR_URL));
    assertThat(repositoryMirrored.getAuthentication(), notNullValue());
    AuthenticationContext authContext =
        AuthenticationContext.forRepository(MavenRepositorySystemUtils.newSession(), repositoryMirrored);
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("username"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("password"));

    final List<org.eclipse.aether.repository.RemoteRepository> mirroredRepositories = mirror.getMirroredRepositories();
    assertThat(mirroredRepositories.size(), is(1));
    assertThat(mirroredRepositories.get(0).getUrl(), is(MAVEN_CENTRAL));
  }

  @Test
  public void proxyFromSettings() throws MalformedURLException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-proxy.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(Files.createTempDir())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(MAVEN_CENTRAL))
            .id("central")
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    final org.eclipse.aether.repository.RemoteRepository repository =
        new org.eclipse.aether.repository.RemoteRepository.Builder("central", "default", MAVEN_CENTRAL).build();
    assertThat(AuthenticationContext.forRepository(MavenRepositorySystemUtils.newSession(), repository), nullValue());

    assertThat(context.getProxySelector().isPresent(), is(true));
    final Proxy proxy = context.getProxySelector().get().getProxy(repository);
    assertThat(proxy, notNullValue());
    assertThat(proxy.getHost(), is("proxy.testing.com"));
    assertThat(proxy.getAuthentication(), notNullValue());

    assertThat(context.getRemoteRepositories().size(), is(1));

    AuthenticationContext authContext =
        AuthenticationContext.forProxy(MavenRepositorySystemUtils.newSession(), context.getRemoteRepositories().get(0));
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("proxyuser"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("somepassword"));

    assertThat(context.getProxySelector().get()
        .getProxy(new org.eclipse.aether.repository.RemoteRepository.Builder("ibiblio", "default", "ibiblio.org/repository")
            .build()), nullValue());
  }

  @Test
  public void proxyAndMirrorFromSettings() throws MalformedURLException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-proxy-and-mirror.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(Files.createTempDir())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(MAVEN_CENTRAL))
            .id("central")
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getProxySelector().isPresent(), is(true));
    final Proxy proxy = context.getProxySelector().get()
        .getProxy(new org.eclipse.aether.repository.RemoteRepository.Builder("central", "default", MAVEN_CENTRAL).build());
    assertThat(proxy, notNullValue());
    assertThat(proxy.getHost(), is("proxy.prod.com"));
    assertThat(proxy.getAuthentication(), notNullValue());

    assertThat(context.getMirrorSelector().isPresent(), is(true));
    final org.eclipse.aether.repository.RemoteRepository mirror = context.getMirrorSelector().get()
        .getMirror(new org.eclipse.aether.repository.RemoteRepository.Builder("central", "default", MAVEN_CENTRAL).build());
    assertThat(mirror, notNullValue());
    assertThat(mirror.getUrl(), is(DOWNLOADS_PLANETMIRROR_URL));
  }

  @Test
  public void authenticatorFromSettings() throws MalformedURLException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-servers.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(Files.createTempDir())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .id("planetmirror.com")
            .authentication(org.mule.maven.client.api.model.Authentication.newAuthenticationBuilder()
                .username("myuser")
                .password("mypass")
                .build())
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getAuthenticatorSelector().isPresent(), is(true));
    final Authentication authentication = context.getAuthenticatorSelector().get()
        .getAuthentication(
                           new org.eclipse.aether.repository.RemoteRepository.Builder("planetmirror.com", "default",
                                                                                      DOWNLOADS_PLANETMIRROR_URL)
                                                                                          .build());
    assertThat(authentication, notNullValue());

    assertThat(context.getRemoteRepositories().size(), is(1));

    AuthenticationContext authContext =
        AuthenticationContext.forRepository(MavenRepositorySystemUtils.newSession(), context.getRemoteRepositories().get(0));
    // credentials are overwritten from settings servers section
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("username"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("password"));
  }

  @Test
  public void settingsWithProxyAndAuthenticatorIsAppliedToDeclaredRepositoriesFromSettings() throws MalformedURLException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-servers-and-remote-repositories.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(Files.createTempDir())
        .userSettingsLocation(toFile(settingsUrl))
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getAuthenticatorSelector().isPresent(), is(true));

    assertThat(context.getRemoteRepositories().size(), is(1));
    org.eclipse.aether.repository.RemoteRepository mergedRepository = context.getRemoteRepositories().get(0);

    // Authentication
    Authentication authentication = mergedRepository.getAuthentication();
    assertThat(authentication, notNullValue());
    AuthenticationContext authContext =
        AuthenticationContext.forRepository(MavenRepositorySystemUtils.newSession(), mergedRepository);
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("username"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("password"));

    // Proxy
    Proxy proxy = mergedRepository.getProxy();
    assertThat(proxy, notNullValue());
    assertThat(proxy.getHost(), is("proxy.prod.com"));
    assertThat(proxy.getAuthentication(), notNullValue());
  }

  @Test
  public void authenticatorFromMavenConfiguration() throws MalformedURLException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-servers.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(Files.createTempDir())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .id("other-planetmirror.com")
            .authentication(org.mule.maven.client.api.model.Authentication.newAuthenticationBuilder()
                .username("myuser")
                .password("mypass")
                .build())
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getProxySelector().isPresent(), is(true));
    final Authentication authentication = context.getAuthenticatorSelector().get()
        .getAuthentication(
                           new org.eclipse.aether.repository.RemoteRepository.Builder("other-planetmirror.com", "default",
                                                                                      DOWNLOADS_PLANETMIRROR_URL)
                                                                                          .build());
    // The authenticatorSelector will not have an entry for the explicit remote repository
    assertThat(authentication, nullValue());

    assertThat(context.getRemoteRepositories().size(), is(1));

    AuthenticationContext authContext =
        AuthenticationContext.forRepository(MavenRepositorySystemUtils.newSession(), context.getRemoteRepositories().get(0));
    // credentials are not overwritten from settings servers section
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("myuser"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("mypass"));
  }

}
