/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.io.File.separator;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.mule.maven.client.internal.MavenEnvironmentVariables.MAVEN_CMD_LINE_ARGS;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.maven.client.api.SettingsSupplierFactory;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;

import java.io.File;
import java.util.Optional;

import org.slf4j.Logger;

/**
 * Provides a set of suppliers to be used to resolve maven settings files.
 *
 * @since 1.0
 */
public class DefaultSettingsSupplierFactory implements SettingsSupplierFactory {

  private static final Logger LOGGER = getLogger(DefaultSettingsSupplierFactory.class);

  public static final String SETTINGS_XML = "settings.xml";
  public static final String MAVEN_GLOBAL_SETTINGS_PATH = "conf" + separator + SETTINGS_XML;

  private MavenEnvironmentVariables mavenEnvironmentVariables;

  public DefaultSettingsSupplierFactory(MavenEnvironmentVariables mavenEnvironmentVariables) {
    requireNonNull(mavenEnvironmentVariables, "mavenEnvironmentVariables cannot be null");

    this.mavenEnvironmentVariables = mavenEnvironmentVariables;
  }

  @Override
  public Optional<File> environmentGlobalSettingsSupplier() {
    MavenArguments mavenArguments = parseMavenArguments();

    if (mavenArguments.getGlobalSettings().isPresent()) {
      final File file = mavenArguments.getGlobalSettings().get();
      if (file.exists()) {
        logGlobalSettings(file, MAVEN_CMD_LINE_ARGS);
        return of(mavenArguments.getGlobalSettings().get());
      }
    }

    File globalSettingsFile = mavenEnvironmentVariables.getSettingsFileAsSystemOrEnvProperty(GLOBAL_SETTINGS_SYSTEM_PROPERTY);
    if (globalSettingsFile != null) {
      logGlobalSettings(globalSettingsFile, GLOBAL_SETTINGS_SYSTEM_PROPERTY);
      return of(globalSettingsFile);
    }

    String mavenHome = mavenEnvironmentVariables.getM2HomeEnv();
    mavenHome = mavenHome != null ? mavenHome : mavenEnvironmentVariables.getMavenHomeEnv();
    mavenHome = mavenHome != null ? mavenHome : mavenEnvironmentVariables.getMavenHomeProperty();
    if (mavenHome != null) {
      File globalSettings = new File(mavenHome, MAVEN_GLOBAL_SETTINGS_PATH);
      if (globalSettings.exists()) {
        logGlobalSettings(globalSettings, mavenHome);
        return of(globalSettings);
      }
    }
    return empty();
  }

  @Override
  public Optional<File> environmentUserSettingsSupplier() {
    MavenArguments mavenArguments = parseMavenArguments();

    if (mavenArguments.getSettings().isPresent()) {
      final File file = mavenArguments.getSettings().get();
      if (file.exists()) {
        logUserSettings(file, MAVEN_CMD_LINE_ARGS);
        return of(file);
      }
    }

    File userSettingsFile = mavenEnvironmentVariables.getSettingsFileAsSystemOrEnvProperty(USER_SETTINGS_SYSTEM_PROPERTY);
    if (userSettingsFile != null) {
      logUserSettings(userSettingsFile, USER_SETTINGS_SYSTEM_PROPERTY);
      return of(userSettingsFile);
    }

    File userM2Folder = new File(mavenEnvironmentVariables.getUserHome(), ".m2");
    if (userM2Folder.exists()) {
      final File file = new File(userM2Folder, SETTINGS_XML);
      if (file.exists()) {
        logUserSettings(file, userM2Folder.getAbsolutePath());
        return of(file);
      }
    }
    return empty();
  }

  private void logUserSettings(File userSettings, String location) {
    LOGGER.debug("Resolved Maven user settings '{}' from '{}'", userSettings.getAbsolutePath(), location);
  }

  private void logGlobalSettings(File globalSettings, String location) {
    LOGGER.debug("Resolved Maven global settings '{}' from '{}'", globalSettings.getAbsolutePath(), location);
  }

  private String[] getMavenCmdLineArgs() {
    String mavenCmdLineArgs = mavenEnvironmentVariables.getMavenCmdLineArgsEnv();
    if (mavenCmdLineArgs != null) {
      return mavenCmdLineArgs.trim().split(" ");
    }
    return new String[0];
  }

  private MavenArguments parseMavenArguments() {
    MavenArguments mavenArguments = new MavenArguments();
    JCommander.newBuilder()
        .addObject(mavenArguments)
        .acceptUnknownOptions(true)
        .build()
        .parse(getMavenCmdLineArgs());
    return mavenArguments;
  }

  private class MavenArguments {

    @Parameter(names = {"-s", "--settings"}, description = "User settings")
    private File settings = null;

    @Parameter(names = {"-gs", "--global-settings"}, description = "Global settings")
    private File globalSettings = null;

    public Optional<File> getSettings() {
      return ofNullable(settings);
    }

    public Optional<File> getGlobalSettings() {
      return ofNullable(globalSettings);
    }
  }

}
