/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.lang.String.format;
import static java.lang.System.getProperties;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FilenameUtils.getExtension;
import static org.apache.maven.model.building.ModelBuildingRequest.VALIDATION_LEVEL_MINIMAL;
import static org.eclipse.aether.util.artifact.ArtifactIdUtils.toId;
import static org.eclipse.aether.util.artifact.JavaScopes.COMPILE;
import static org.eclipse.aether.util.artifact.JavaScopes.PROVIDED;
import static org.eclipse.aether.util.artifact.JavaScopes.TEST;
import static org.eclipse.aether.util.graph.transformer.ConflictResolver.NODE_DATA_WINNER;
import static org.mule.maven.client.internal.BundleDependencyHelper.artifactToBundleDependency;
import static org.mule.maven.client.internal.util.MavenUtils.getPomModel;
import static org.mule.maven.client.internal.util.Preconditions.checkState;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenReactorResolver;
import org.mule.maven.client.api.PomFileSupplierFactory;
import org.mule.maven.client.api.VersionRangeResult;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;

import com.google.common.collect.ImmutableList;

import java.io.File;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

import org.apache.maven.artifact.versioning.InvalidVersionSpecificationException;
import org.apache.maven.artifact.versioning.VersionRange;
import org.apache.maven.model.Model;
import org.apache.maven.model.Profile;
import org.apache.maven.model.building.DefaultModelBuilderFactory;
import org.apache.maven.model.building.DefaultModelBuildingRequest;
import org.apache.maven.model.building.ModelBuildingException;
import org.apache.maven.repository.internal.MavenWorkspaceReader;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositoryException;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.collection.CollectRequest;
import org.eclipse.aether.collection.CollectResult;
import org.eclipse.aether.collection.DependencyCollectionException;
import org.eclipse.aether.graph.Dependency;
import org.eclipse.aether.graph.DependencyNode;
import org.eclipse.aether.graph.Exclusion;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.repository.WorkspaceReader;
import org.eclipse.aether.repository.WorkspaceRepository;
import org.eclipse.aether.resolution.ArtifactDescriptorException;
import org.eclipse.aether.resolution.ArtifactDescriptorRequest;
import org.eclipse.aether.resolution.ArtifactDescriptorResult;
import org.eclipse.aether.resolution.ArtifactRequest;
import org.eclipse.aether.resolution.ArtifactResolutionException;
import org.eclipse.aether.resolution.ArtifactResult;
import org.eclipse.aether.resolution.DependencyRequest;
import org.eclipse.aether.resolution.DependencyResolutionException;
import org.eclipse.aether.resolution.DependencyResult;
import org.eclipse.aether.resolution.VersionRangeRequest;
import org.eclipse.aether.resolution.VersionRangeResolutionException;
import org.eclipse.aether.util.filter.PatternInclusionsDependencyFilter;
import org.eclipse.aether.util.graph.visitor.PathRecordingDependencyVisitor;
import org.eclipse.aether.util.graph.visitor.PreorderNodeListGenerator;
import org.eclipse.aether.util.repository.ChainedWorkspaceReader;
import org.slf4j.Logger;

public class AetherMavenClient implements MavenClient {

  public static final String MULE_PLUGIN_CLASSIFIER = "mule-plugin";
  public static final String MULE_DOMAIN_CLASSIFIER = "mule-domain";

  private static final Logger LOGGER = getLogger(AetherMavenClient.class);
  private static final String POM = "pom";
  public static final String POM_XML = "pom.xml";

  private PomFileSupplierFactory pomFileSupplierFactory = new DefaultPomFileSupplierFactory();

  private MavenConfiguration mavenConfiguration;
  private AetherResolutionContext aetherResolutionContext;
  private Optional<Consumer<DefaultRepositorySystemSession>> sessionConfigurator = empty();

  public AetherMavenClient(MavenConfiguration mavenConfiguration) {
    this.mavenConfiguration = mavenConfiguration;
    this.aetherResolutionContext = new AetherResolutionContext(mavenConfiguration);
  }

  private AetherRepositoryState getRepositoryState(File localRepositoryLocation,
                                                   Optional<WorkspaceReader> workspaceReader) {
    return AetherRepositoryStateFactory.newAetherRepositoryState(localRepositoryLocation, aetherResolutionContext,
                                                                 mavenConfiguration, sessionConfigurator, workspaceReader);
  }

  @Override
  public MavenConfiguration getMavenConfiguration() {
    return this.mavenConfiguration;
  }

  public void setSessionConfigurator(Consumer<DefaultRepositorySystemSession> sessionConfigurator) {
    this.sessionConfigurator = of(sessionConfigurator);
  }

  @Override
  public List<BundleDependency> resolveArtifactDependencies(File artifactFile,
                                                            boolean includeTestDependencies,
                                                            boolean includeProvidedDependencies,
                                                            Optional<File> localRepositoryLocationSupplier,
                                                            Optional<File> temporaryFolder) {
    return resolveArtifactDependencies(artifactFile, includeTestDependencies, includeProvidedDependencies,
                                       localRepositoryLocationSupplier, empty(), temporaryFolder);
  }

  @Override
  public List<BundleDependency> resolveArtifactDependencies(File artifactFile,
                                                            boolean includeTestDependencies,
                                                            boolean includeProvidedDependencies,
                                                            Optional<File> localRepositoryLocationSupplier,
                                                            Optional<MavenReactorResolver> mavenReactorResolver,
                                                            Optional<File> temporaryFolder) {
    try {
      File localRepositoryLocation =
          localRepositoryLocationSupplier.orElse(this.aetherResolutionContext.getLocalRepositoryLocation());

      AetherRepositoryState repositoryState = getRepositoryState(localRepositoryLocation, empty());

      Model model = getEffectiveModel(artifactFile, temporaryFolder, of(repositoryState));

      if (model.getProfiles().stream().map(Profile::getActivation).filter(Objects::nonNull)
          .anyMatch(activation -> activation.getFile() != null)) {
        throw new UnsupportedOperationException(format("Error while resolving dependencies for %s due to profiles activation by file are not supported",
                                                       model.getId()));
      }

      Artifact artifact = new DefaultArtifact(model.getGroupId(), model.getArtifactId(),
                                              null,
                                              POM,
                                              model.getVersion());

      WorkspaceReader workspaceReader = new PomWorkspaceReader(model);
      if (mavenReactorResolver.isPresent()) {
        workspaceReader =
            new ChainedWorkspaceReader(workspaceReader, new MavenWorkspaceReaderAdapter(mavenReactorResolver.get()));
      }

      repositoryState.setWorkspaceReader(workspaceReader);

      List<Dependency> dependencies =
          model.getDependencies().stream().map(modelDependency -> convertModelDependency(modelDependency)).collect(toList());
      List<Dependency> managedDependencies =
          model.getDependencyManagement() == null ? emptyList() : model.getDependencyManagement().getDependencies().stream()
              .map(modelDependency -> convertModelDependency(modelDependency))
              .collect(toList());
      return resolveDependencies(artifact,
                                 dependencies,
                                 managedDependencies,
                                 includeTestDependencies,
                                 includeProvidedDependencies,
                                 repositoryState);
    } catch (DependencyResolutionException e) {
      DependencyNode node = e.getResult().getRoot();
      logUnresolvedArtifacts(node, e);
      throw new RuntimeException(format("There was an issue solving the dependencies for the artifact [%s]",
                                        artifactFile.getAbsolutePath()),
                                 e);
    } catch (DependencyCollectionException e) {
      throw new RuntimeException(format("There was an issue resolving the dependency tree for the artifact [%s]",
                                        artifactFile.getAbsolutePath()),
                                 e);
    } catch (ArtifactDescriptorException e) {
      throw new RuntimeException(format("There was an issue resolving the artifact descriptor for the artifact [%s]",
                                        artifactFile.getAbsolutePath()),
                                 e);
    } catch (RepositoryException e) {
      throw new RuntimeException(e);
    }
  }

  public Model getEffectiveModel(File artifactFile, Optional<File> temporaryFolder,
                                 Optional<AetherRepositoryState> repositoryStateOptional) {
    Model model;
    try {
      AetherRepositoryState repositoryState = repositoryStateOptional
          .orElseGet(() -> getRepositoryState(this.aetherResolutionContext.getLocalRepositoryLocation(), empty()));
      DefaultModelBuildingRequest defaultModelBuildingRequest = new DefaultModelBuildingRequest();
      defaultModelBuildingRequest.setPomFile(getPomFile(artifactFile, temporaryFolder));
      mavenConfiguration.getUserProperties().ifPresent(properties -> defaultModelBuildingRequest.setUserProperties(properties));
      mavenConfiguration.getActiveProfiles().ifPresent(profiles -> defaultModelBuildingRequest.setActiveProfileIds(profiles));
      mavenConfiguration.getInactiveProfiles()
          .ifPresent(profiles -> defaultModelBuildingRequest.setInactiveProfileIds(profiles));
      defaultModelBuildingRequest.setTwoPhaseBuilding(false);
      defaultModelBuildingRequest.setProcessPlugins(false);
      defaultModelBuildingRequest.setModelCache(DefaultModelCache.newInstance(repositoryState.getSession()));
      defaultModelBuildingRequest
          .setModelResolver(repositoryState.createModelResolver(aetherResolutionContext.getRemoteRepositories()));
      defaultModelBuildingRequest.setSystemProperties(getProperties());
      defaultModelBuildingRequest.setValidationLevel(VALIDATION_LEVEL_MINIMAL);

      model = new DefaultModelBuilderFactory().newInstance().build(defaultModelBuildingRequest).getEffectiveModel();
    } catch (ModelBuildingException e) {
      throw new RuntimeException(e);
    }
    return model;
  }

  @Override
  public List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies,
                                                                    BundleDescriptor bundleDescriptor) {
    return resolveBundleDescriptorDependencies(includeTestDependencies, false, bundleDescriptor);
  }

  @Override
  public List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies,
                                                                    boolean includeProvidedDependencies,
                                                                    BundleDescriptor bundleDescriptor) {
    try {
      AetherRepositoryState repositoryState =
          getRepositoryState(this.aetherResolutionContext.getLocalRepositoryLocation(), empty());
      Artifact artifact = createArtifactFromBundleDescriptor(bundleDescriptor);
      ArtifactResult artifactResult = repositoryState.getSystem().resolveArtifact(repositoryState.getSession(),
                                                                                  new ArtifactRequest(artifact,
                                                                                                      aetherResolutionContext
                                                                                                          .getRemoteRepositories(),
                                                                                                      null));
      ArtifactDescriptorResult artifactDescriptorResult =
          repositoryState.getSystem().readArtifactDescriptor(repositoryState.getSession(),
                                                             new ArtifactDescriptorRequest(artifact,
                                                                                           aetherResolutionContext
                                                                                               .getRemoteRepositories(),
                                                                                           null));
      return resolveDependencies(artifactResult.getArtifact(),
                                 artifactDescriptorResult.getDependencies(),
                                 emptyList(),
                                 includeTestDependencies,
                                 includeProvidedDependencies,
                                 repositoryState);
    } catch (RepositoryException e) {
      throw new BundleDependenciesResolutionException(e);
    }
  }

  private Artifact createArtifactFromBundleDescriptor(BundleDescriptor bundleDescriptor) {
    return new DefaultArtifact(bundleDescriptor.getGroupId(), bundleDescriptor.getArtifactId(),
                               bundleDescriptor.getClassifier().orElse(null),
                               bundleDescriptor.getType(),
                               bundleDescriptor.getVersion());
  }

  @Override
  public BundleDependency resolveBundleDescriptor(BundleDescriptor bundleDescriptor) {
    try {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Resolving artifact with resolution context: " + aetherResolutionContext.toString());
      }
      AetherRepositoryState repositoryState = getRepositoryState(mavenConfiguration.getLocalMavenRepositoryLocation(), empty());
      Artifact artifact = createArtifactFromBundleDescriptor(bundleDescriptor);
      ArtifactRequest artifactRequest =
          new ArtifactRequest(artifact, this.aetherResolutionContext.getRemoteRepositories(), null);
      ArtifactResult artifactResult =
          repositoryState.getSystem().resolveArtifact(repositoryState.getSession(), artifactRequest);
      Artifact resolvedArtifact = artifactResult.getArtifact();
      return artifactToBundleDependency(resolvedArtifact, "compile").build();
    } catch (ArtifactResolutionException e) {
      throw new BundleDependenciesResolutionException(e);
    }
  }

  @Override
  public Model getRawPomModel(File artifactFile) {
    return getPomModel(artifactFile);
  }

  @Override
  public Model getEffectiveModel(File artifactFile, Optional<File> temporaryFolder) {
    return this.getEffectiveModel(artifactFile, temporaryFolder, empty());
  }

  @Override
  public List<BundleDependency> resolvePluginBundleDescriptorsDependencies(List<BundleDescriptor> bundleDescriptors) {
    AetherRepositoryState repositoryState =
        getRepositoryState(mavenConfiguration.getLocalMavenRepositoryLocation(), empty());

    try {
      CollectRequest collectRequest = new CollectRequest();
      collectRequest.setDependencies(bundleDescriptors.stream().map(bundleDescriptor -> {
        // This method is meant to resolve mule-plugin only
        return new Dependency(new DefaultArtifact(bundleDescriptor.getGroupId(), bundleDescriptor.getArtifactId(),
                                                  MULE_PLUGIN_CLASSIFIER, "jar", bundleDescriptor.getVersion()),
                              COMPILE);
      }).collect(toList()));
      return doResolveDependencies(bundleDescriptors.toString(), repositoryState, collectRequest);
    } catch (DependencyResolutionException e) {
      DependencyNode node = e.getResult().getRoot();
      logUnresolvedArtifacts(node, e);
      throw new RuntimeException(format("There was an issue solving the dependencies for the bundleDescriptors [%s]",
                                        bundleDescriptors),
                                 e);
    } catch (DependencyCollectionException e) {
      throw new RuntimeException(format("There was an issue resolving the dependency tree for the bundleDescriptors [%s]",
                                        bundleDescriptors),
                                 e);
    } catch (RepositoryException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public VersionRangeResult resolveVersionRange(BundleDescriptor bundleDescriptor) {
    AetherRepositoryState repositoryState =
        getRepositoryState(this.aetherResolutionContext.getLocalRepositoryLocation(), empty());
    try {
      VersionRange.createFromVersionSpec(bundleDescriptor.getVersion());
      final org.eclipse.aether.resolution.VersionRangeResult versionRangeResult = repositoryState
          .getSystem().resolveVersionRange(repositoryState.getSession(),
                                           new VersionRangeRequest(createArtifactFromBundleDescriptor(bundleDescriptor),
                                                                   aetherResolutionContext
                                                                       .getRemoteRepositories(),
                                                                   null));
      return new DefaultVersionRangeResult(versionRangeResult);
    } catch (InvalidVersionSpecificationException | VersionRangeResolutionException e) {
      throw new RuntimeException(format("Couldn't resolve version range for bundleDescriptor: '%s'",
                                        bundleDescriptor.toString(), e));
    }
  }

  private List<BundleDependency> resolveDependencies(Artifact artifact,
                                                     List<Dependency> dependencies,
                                                     List<Dependency> managedDependencies,
                                                     boolean enableTestDependencies,
                                                     boolean enableProvidedDependencies,
                                                     AetherRepositoryState repositoryState)
      throws RepositoryException {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("About to fetch required dependencies for artifact: {}. This may take a while...", toId(artifact));
    }

    final CollectRequest collectRequest = new CollectRequest();
    List<Dependency> requestedDependencies = emptyList();
    if (enableTestDependencies) {
      requestedDependencies = dependencies.stream()
          .filter(modelDependency -> TEST.equalsIgnoreCase(modelDependency.getScope()))
          .map(modelDependency -> modelDependency.setScope(COMPILE))
          .collect(toList());
    }
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Resolving dependencies for: {} from list of dependencies: {}", toId(artifact),
                   requestedDependencies);
    }

    // Include root dependency in order to get remote repositories from pom and allow aether to use selectors (mirrors, proxies,
    // authentication).
    collectRequest.setRoot(new Dependency(artifact, COMPILE));
    collectRequest.setRootArtifact(artifact);
    // Direct dependencies modified as merged and take precedence over the ones in pom
    collectRequest.setDependencies(requestedDependencies);
    if (!managedDependencies.isEmpty()) {
      collectRequest.setManagedDependencies(managedDependencies);
    }
    List<BundleDependency> bundleDependencies = doResolveDependencies(toId(artifact), repositoryState, collectRequest);
    // root dependency is not wanted as result (always)
    bundleDependencies.remove(0);
    if (enableProvidedDependencies) {
      // Append direct dependencies declared as provided, due to Aether excludes them during the resolution
      bundleDependencies.addAll(dependencies.stream()
          .filter(dependency -> PROVIDED.equalsIgnoreCase(dependency.getScope()))
          .map(
               dependency -> artifactToBundleDependency(dependency.getArtifact(), dependency.getScope()).build())
          .collect(toList()));
    }
    return bundleDependencies;
  }

  private List<BundleDependency> doResolveDependencies(String artifactId, AetherRepositoryState repositoryState,
                                                       CollectRequest collectRequest)
      throws DependencyCollectionException, DependencyResolutionException {
    // Set remote repositories first
    collectRequest.setRepositories(ImmutableList.<RemoteRepository>builder()
        .addAll(aetherResolutionContext.getRemoteRepositories())
        .build());

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Collecting transitive dependencies of artifact: {} using request: {}", artifactId, collectRequest);
    }
    final CollectResult collectResult =
        repositoryState.getSystem().collectDependencies(repositoryState.getSession(), collectRequest);
    final DependencyRequest dependencyRequest = new DependencyRequest();
    dependencyRequest.setRoot(collectResult.getRoot());
    dependencyRequest.setCollectRequest(collectRequest);

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Collecting and resolving transitive dependencies of artifact: {}", artifactId);
    }
    dependencyRequest.setFilter((node, parents) -> !node.getData().containsKey(NODE_DATA_WINNER));
    final DependencyResult dependencyResult =
        repositoryState.getSystem().resolveDependencies(repositoryState.getSession(), dependencyRequest);

    PreorderNodeListGenerator nlg = new PreorderNodeListGenerator();
    dependencyResult.getRoot().accept(nlg);

    BundleDependencyHelper bundleDependencyHelper = new BundleDependencyHelper();

    return nlg.getNodes().stream()
        .filter(node -> !node.getData().containsKey(NODE_DATA_WINNER))
        .map(node -> bundleDependencyHelper.getBundleDependency(node))
        .collect(toList());
  }

  private static Dependency convertModelDependency(org.apache.maven.model.Dependency modelDependency) {
    Artifact artifactDependency =
        new DefaultArtifact(modelDependency.getGroupId(), modelDependency.getArtifactId(), modelDependency.getClassifier(),
                            modelDependency.getType(), modelDependency.getVersion());
    Dependency dependency =
        new Dependency(artifactDependency, modelDependency.getScope() != null ? modelDependency.getScope() : COMPILE);
    dependency.setOptional(modelDependency.isOptional());
    dependency.setExclusions(dependency.getExclusions()
        .stream().map(modelExclusion -> new Exclusion(modelExclusion.getGroupId(), modelExclusion.getArtifactId(),
                                                      modelExclusion.getClassifier(), modelExclusion.getExtension()))
        .collect(toList()));
    return dependency;
  }

  private File getPomFile(File artifactFile, Optional<File> temporaryFolder) {
    if (artifactFile.isFile() && (getExtension(artifactFile.getName()).equals(POM) || artifactFile.getName().equals(POM_XML))) {
      return artifactFile;
    } else {
      Model rawPomModel = getRawPomModel(artifactFile);
      BundleDescriptor bundleDescriptor = new BundleDescriptor.Builder()
          .setGroupId(rawPomModel.getGroupId() != null ? rawPomModel.getGroupId() : rawPomModel.getParent().getGroupId())
          .setArtifactId(rawPomModel.getArtifactId())
          .setVersion(rawPomModel.getVersion() != null ? rawPomModel.getVersion() : rawPomModel.getParent().getVersion())
          .build();

      if (artifactFile.isDirectory()) {
        return pomFileSupplierFactory.uncompressPomArtifactSupplier(artifactFile, bundleDescriptor).get();
      } else {
        checkState(temporaryFolder.isPresent(), "temporary folder not provided but the artifact is compressed");
        return pomFileSupplierFactory.compressedArtifactSupplier(artifactFile, bundleDescriptor, temporaryFolder.get()).get();
      }
    }
  }

  private void logUnresolvedArtifacts(DependencyNode node, DependencyResolutionException e) {
    List<ArtifactResult> artifactResults = e.getResult().getArtifactResults().stream()
        .filter(artifactResult -> !artifactResult.getExceptions().isEmpty()).collect(toList());

    final List<String> patternInclusion =
        artifactResults.stream().map(artifactResult -> toId(artifactResult.getRequest().getArtifact()))
            .collect(toList());

    PathRecordingDependencyVisitor visitor =
        new PathRecordingDependencyVisitor(new PatternInclusionsDependencyFilter(patternInclusion),
                                           node.getArtifact() != null);
    node.accept(visitor);

    visitor.getPaths().stream().forEach(path -> {
      List<DependencyNode> unresolvedArtifactPath =
          path.stream().filter(dependencyNode -> dependencyNode.getArtifact() != null).collect(toList());
      if (!unresolvedArtifactPath.isEmpty()) {
        if (LOGGER.isWarnEnabled()) {
          LOGGER.warn("Dependency path to not resolved artifacts -> " + unresolvedArtifactPath.toString());
        }
      }
    });
  }

  /**
   * Custom implementation of a {@link MavenWorkspaceReader} meant to be tightly used with the mule artifacts, where the POM file
   * is inside the exploded artifact or the packaged artifact. For any other {@link Artifact} it will return values that will
   * force the dependency mechanism to look for in a different {@link WorkspaceReader}.
   *
   * @since 1.0
   */
  private class PomWorkspaceReader implements MavenWorkspaceReader {

    final WorkspaceRepository workspaceRepository;
    private final Model model;

    /**
     * @param model Maven effective model.
     */
    PomWorkspaceReader(Model model) {
      this.workspaceRepository = new WorkspaceRepository(format("workspace-repository-%s", model.getName()));
      this.model = model;
    }

    @Override
    public WorkspaceRepository getRepository() {
      return workspaceRepository;
    }

    @Override
    public Model findModel(Artifact artifact) {
      if (checkArtifact(artifact)) {
        return model;
      }
      return null;
    }

    @Override
    public File findArtifact(Artifact artifact) {
      if (checkArtifact(artifact)) {
        return model.getPomFile();
      }
      return null;
    }

    @Override
    public List<String> findVersions(Artifact artifact) {
      if (checkArtifact(artifact)) {
        return singletonList(artifact.getVersion());
      }
      return emptyList();
    }

    private boolean checkArtifact(Artifact artifact) {
      return this.model.getGroupId().equals(artifact.getGroupId())
          && this.model.getArtifactId().equals(artifact.getArtifactId())
          && this.model.getVersion().equals(artifact.getVersion())
          && artifact.getExtension().equals(POM);
    }
  }

  /**
   * Custom implementation of a {@link WorkspaceReader} meant to be adapt {@link MavenReactorResolver} to {@link WorkspaceReader}.
   *
   * @since 1.1
   */
  private class MavenWorkspaceReaderAdapter implements WorkspaceReader {

    private final WorkspaceRepository workspaceRepository;
    private final MavenReactorResolver mavenReactorResolver;


    MavenWorkspaceReaderAdapter(MavenReactorResolver mavenReactorResolver) {
      this.workspaceRepository = new WorkspaceRepository(format("worskpace-repository-reactor"));
      this.mavenReactorResolver = mavenReactorResolver;
    }

    @Override
    public WorkspaceRepository getRepository() {
      return workspaceRepository;
    }

    @Override
    public File findArtifact(Artifact artifact) {
      return mavenReactorResolver.findArtifact(toBundleDescriptor(artifact));
    }

    public BundleDescriptor toBundleDescriptor(Artifact artifact) {
      return new BundleDescriptor.Builder()
          .setGroupId(artifact.getGroupId())
          .setArtifactId(artifact.getArtifactId())
          .setVersion(artifact.getVersion())
          .setBaseVersion(artifact.getBaseVersion())
          .setType(artifact.getExtension())
          .build();
    }

    @Override
    public List<String> findVersions(Artifact artifact) {
      return mavenReactorResolver.findVersions(toBundleDescriptor(artifact));
    }

  }

}
