/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.lang.System.getProperty;
import org.mule.maven.client.api.LocalRepositorySupplierFactory;

import java.io.File;
import java.util.function.Supplier;

/**
 * Provides a set of suppliers to be used to resolve maven repository folder.
 * 
 * @since 1.0
 */
public class DefaultLocalRepositorySupplierFactory implements LocalRepositorySupplierFactory {

  private static final String USER_HOME = "user.home";
  private static final String M2_REPO = "/.m2/repository";
  private static String userHome = getProperty(USER_HOME);

  public Supplier<File> fixedFolderSupplier(File folder) {
    return () -> folder;
  }

  public Supplier<File> environmentMavenRepositorySupplier() {
    return () -> getMavenLocalRepository();
  }

  public Supplier<File> composeSuppliers(Supplier<File>... suppliers) {
    return () -> {
      for (Supplier<File> supplier : suppliers) {
        File file = supplier.get();
        if (file != null) {
          return file;
        }
      }
      return null;
    };
  }

  /**
   * @return a file pointing to the user local maven repository.
   */
  private static File getMavenLocalRepository() {
    String localRepositoryProperty = getProperty("localRepository");
    if (localRepositoryProperty == null) {
      localRepositoryProperty = userHome + M2_REPO;
    }

    File mavenLocalRepositoryLocation = new File(localRepositoryProperty);
    if (!mavenLocalRepositoryLocation.exists()) {
      throw new IllegalArgumentException("Maven repository location couldn't be found, please check your configuration");
    }
    return mavenLocalRepositoryLocation;
  }

}
