/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static java.util.Collections.emptyMap;
import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.maven.repository.internal.MavenRepositorySystemUtils.newSession;
import static org.eclipse.aether.repository.RepositoryPolicy.CHECKSUM_POLICY_FAIL;
import static org.eclipse.aether.repository.RepositoryPolicy.CHECKSUM_POLICY_WARN;
import static org.eclipse.aether.repository.RepositoryPolicy.UPDATE_POLICY_DAILY;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.client.internal.AetherResolutionContext;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;

import org.eclipse.aether.repository.Authentication;
import org.eclipse.aether.repository.AuthenticationContext;
import org.eclipse.aether.repository.Proxy;
import org.eclipse.aether.repository.RepositoryPolicy;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class AetherResolutionContextTestCase {

  private static final String MAVEN_CENTRAL = "https://repo.maven.apache.org/maven2/";
  private static final String DOWNLOADS_PLANETMIRROR_URL = "http://downloads.planetmirror.com/pub/maven2";

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Test
  public void mavenConfigurationLocalRepositoryTakesPrecedenceOverSettings() throws IOException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-local-repository.xml");

    File localRepository = temporaryFolder.newFolder();
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .userSettingsLocation(toFile(settingsUrl))
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);
    assertThat(context.getLocalRepositoryLocation().getAbsolutePath(), equalTo(localRepository.getAbsolutePath()));
  }

  @Test
  public void mirrorFromSettings() throws IOException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-mirror.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.newFolder())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(MAVEN_CENTRAL))
            .id("central")
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getMirrorSelector().isPresent(), is(true));
    final org.eclipse.aether.repository.RemoteRepository mirror = context.getMirrorSelector().get()
        .getMirror(new org.eclipse.aether.repository.RemoteRepository.Builder("central", "default", MAVEN_CENTRAL).build());
    assertThat(mirror, notNullValue());
    assertThat(mirror.getUrl(), is(DOWNLOADS_PLANETMIRROR_URL));

    // Explicitly defined remoteRepositories to MavenConfiguration should also have been configured with mirror and authenticator
    assertThat(context.getRemoteRepositories().size(), is(1));
    org.eclipse.aether.repository.RemoteRepository repositoryMirrored = context.getRemoteRepositories().get(0);
    assertThat(repositoryMirrored.getUrl(), is(DOWNLOADS_PLANETMIRROR_URL));
    assertThat(repositoryMirrored.getAuthentication(), notNullValue());
    AuthenticationContext authContext =
        AuthenticationContext.forRepository(newSession(), repositoryMirrored);
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("username"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("password"));

    final List<org.eclipse.aether.repository.RemoteRepository> mirroredRepositories = mirror.getMirroredRepositories();
    assertThat(mirroredRepositories.size(), is(1));
    assertThat(mirroredRepositories.get(0).getUrl(), is(MAVEN_CENTRAL));
  }

  @Test
  public void proxyFromSettings() throws IOException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-proxy.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.newFolder())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(MAVEN_CENTRAL))
            .id("central")
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    final org.eclipse.aether.repository.RemoteRepository repository =
        new org.eclipse.aether.repository.RemoteRepository.Builder("central", "default", MAVEN_CENTRAL).build();
    assertThat(AuthenticationContext.forRepository(newSession(), repository), nullValue());

    assertThat(context.getProxySelector().isPresent(), is(true));
    final Proxy proxy = context.getProxySelector().get().getProxy(repository);
    assertThat(proxy, notNullValue());
    assertThat(proxy.getHost(), is("proxy.testing.com"));
    assertThat(proxy.getAuthentication(), notNullValue());

    assertThat(context.getRemoteRepositories().size(), is(1));

    AuthenticationContext authContext =
        AuthenticationContext.forProxy(newSession(), context.getRemoteRepositories().get(0));
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("proxyuser"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("somepassword"));

    assertThat(context.getProxySelector().get()
        .getProxy(new org.eclipse.aether.repository.RemoteRepository.Builder("ibiblio", "default", "ibiblio.org/repository")
            .build()), nullValue());
  }

  @Test
  public void proxyAndMirrorFromSettings() throws IOException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-proxy-and-mirror.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.newFolder())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(MAVEN_CENTRAL))
            .id("central")
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getProxySelector().isPresent(), is(true));
    final Proxy proxy = context.getProxySelector().get()
        .getProxy(new org.eclipse.aether.repository.RemoteRepository.Builder("central", "default", MAVEN_CENTRAL).build());
    assertThat(proxy, notNullValue());
    assertThat(proxy.getHost(), is("proxy.prod.com"));
    assertThat(proxy.getAuthentication(), notNullValue());

    assertThat(context.getMirrorSelector().isPresent(), is(true));
    final org.eclipse.aether.repository.RemoteRepository mirror = context.getMirrorSelector().get()
        .getMirror(new org.eclipse.aether.repository.RemoteRepository.Builder("central", "default", MAVEN_CENTRAL).build());
    assertThat(mirror, notNullValue());
    assertThat(mirror.getUrl(), is(DOWNLOADS_PLANETMIRROR_URL));
  }

  @Test
  public void authenticatorFromSettings() throws IOException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-servers.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.newFolder())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .id("planetmirror.com")
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getAuthenticatorSelector().isPresent(), is(true));
    final Authentication authentication = context.getAuthenticatorSelector().get()
        .getAuthentication(
                           new org.eclipse.aether.repository.RemoteRepository.Builder("planetmirror.com", "default",
                                                                                      DOWNLOADS_PLANETMIRROR_URL)
                                                                                          .build());
    assertThat(authentication, notNullValue());

    assertThat(context.getRemoteRepositories().size(), is(1));

    AuthenticationContext authContext =
        AuthenticationContext.forRepository(newSession(), context.getRemoteRepositories().get(0));
    // credentials are overwritten from settings servers section
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("username"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("password"));
  }

  @Test
  public void authenticatorFromEncryptedSettings() throws IOException {
    URL settingsUrl = getClass().getClassLoader().getResource("encrypted-settings-with-servers.xml");
    URL settingsSecurityUrl = getClass().getClassLoader().getResource("settings-security.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.newFolder())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .id("planetmirror.com")
            .build())
        .settingsSecurityLocation(toFile(settingsSecurityUrl))
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getAuthenticatorSelector().isPresent(), is(true));
    final Authentication authentication = context.getAuthenticatorSelector().get()
        .getAuthentication(
                           new org.eclipse.aether.repository.RemoteRepository.Builder("planetmirror.com", "default",
                                                                                      DOWNLOADS_PLANETMIRROR_URL)
                                                                                          .build());
    assertThat(authentication, notNullValue());

    assertThat(context.getRemoteRepositories().size(), is(1));

    AuthenticationContext authContext =
        AuthenticationContext.forRepository(newSession(), context.getRemoteRepositories().get(0));
    // credentials are overwritten from settings servers section
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("username"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("thisissecret"));
  }

  @Test
  public void settingsWithProxyAndAuthenticatorIsAppliedToDeclaredRepositoriesFromSettings() throws IOException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-servers-and-remote-repositories.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.newFolder())
        .userSettingsLocation(toFile(settingsUrl))
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    assertThat(context.getAuthenticatorSelector().isPresent(), is(true));

    assertThat(context.getRemoteRepositories().size(), is(1));
    org.eclipse.aether.repository.RemoteRepository mergedRepository = context.getRemoteRepositories().get(0);

    // Authentication
    Authentication authentication = mergedRepository.getAuthentication();
    assertThat(authentication, notNullValue());
    AuthenticationContext authContext =
        AuthenticationContext.forRepository(newSession(), mergedRepository);
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("username"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("password"));

    // Proxy
    Proxy proxy = mergedRepository.getProxy();
    assertThat(proxy, notNullValue());
    assertThat(proxy.getHost(), is("proxy.prod.com"));
    assertThat(proxy.getAuthentication(), notNullValue());

    // UpdatePolicy and ChecksumPolicy
    final RepositoryPolicy snapshotsPolicy = mergedRepository.getPolicy(true);
    assertThat(snapshotsPolicy.isEnabled(), is(true));
    // Default value for update policy should be the same as Maven which is daily
    assertThat(snapshotsPolicy.getUpdatePolicy(), is(UPDATE_POLICY_DAILY));
    assertThat(snapshotsPolicy.getChecksumPolicy(), is(CHECKSUM_POLICY_FAIL));

    final RepositoryPolicy releasesPolicy = mergedRepository.getPolicy(false);
    assertThat(releasesPolicy.isEnabled(), is(false));
    assertThat(releasesPolicy.getUpdatePolicy(), is(UPDATE_POLICY_DAILY));
    assertThat(releasesPolicy.getChecksumPolicy(), is(CHECKSUM_POLICY_WARN));
  }

  @Test
  public void authenticatorFromMavenConfigurationNoSettings() throws IOException {
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.newFolder())
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .id("other-planetmirror.com")
            .authentication(org.mule.maven.client.api.model.Authentication.newAuthenticationBuilder()
                .username("myuser")
                .password("mypass")
                .build())
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);
    // RemoteRepository from Aether created for MavenConfiguration RemoteRepository entity should have the authentication set.
    org.eclipse.aether.repository.RemoteRepository remoteRepositoryFromConfiguration = context.getRemoteRepositories().get(0);
    AuthenticationContext authContext = AuthenticationContext.forRepository(newSession(), remoteRepositoryFromConfiguration);

    remoteRepositoryFromConfiguration.getAuthentication().fill(authContext, AuthenticationContext.USERNAME, emptyMap());
    remoteRepositoryFromConfiguration.getAuthentication().fill(authContext, AuthenticationContext.USERNAME, emptyMap());

    assertThat(authContext.get(AuthenticationContext.USERNAME), is("myuser"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("mypass"));
  }

  @Test
  public void authenticatorSetAsPartOfAuthenticatorSelector() throws IOException {
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-servers.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.newFolder())
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .id("other-planetmirror.com")
            .authentication(org.mule.maven.client.api.model.Authentication.newAuthenticationBuilder()
                .username("myuser")
                .password("mypass")
                .build())
            .build())
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);

    org.eclipse.aether.repository.RemoteRepository otherRepository =
        new org.eclipse.aether.repository.RemoteRepository.Builder("other-planetmirror.com", "default",
                                                                   "http://another-url.com")
                                                                       .build();
    Authentication authentication = context.getAuthenticatorSelector().get().getAuthentication(otherRepository);
    // The authenticatorSelector will have an entry for the explicit remote repository declared on MavenConfiguration
    assertThat(authentication, notNullValue());

    AuthenticationContext authContext = AuthenticationContext.forRepository(newSession(),
                                                                            new org.eclipse.aether.repository.RemoteRepository.Builder(otherRepository)
                                                                                .setAuthentication(authentication).build());

    authentication.fill(authContext, AuthenticationContext.USERNAME, emptyMap());
    authentication.fill(authContext, AuthenticationContext.USERNAME, emptyMap());

    // credentials should come from remote repository declared through API
    assertThat(authContext.get(AuthenticationContext.USERNAME), is("myuser"));
    assertThat(authContext.get(AuthenticationContext.PASSWORD), is("mypass"));
  }

  @Test
  public void usingLocalRepository() throws IOException {
    File localRepository = temporaryFolder.newFolder();
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-servers.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .userSettingsLocation(toFile(settingsUrl))
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);
    assertThat(context.getLocalRepositoryLocation(), is(localRepository));
  }

}
