/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Collections.newSetFromMap;
import static java.util.stream.Collectors.toList;
import static org.mule.maven.client.internal.ApiDependencyGraphTransformer.isApi;
import static org.mule.maven.client.internal.MulePluginDependencyGraphTransformer.isPlugin;

import java.util.IdentityHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import org.eclipse.aether.RepositoryException;
import org.eclipse.aether.collection.DependencyGraphTransformationContext;
import org.eclipse.aether.collection.DependencyGraphTransformer;
import org.eclipse.aether.graph.DefaultDependencyNode;
import org.eclipse.aether.graph.DependencyNode;

public class OneInstancePerNodeGraphTransformer implements DependencyGraphTransformer {

  @Override
  public DependencyNode transformGraph(DependencyNode node, DependencyGraphTransformationContext context)
      throws RepositoryException {
    rebuildIsolatedDependencies(node, newSetFromMap(new IdentityHashMap<>()));
    return node;
  }

  private void rebuildIsolatedDependencies(DependencyNode node, Set<DependencyNode> visited) {
    List<DependencyNode> newChildren = new LinkedList<>();
    if (visited.contains(node)) {
      return;
    }
    visited.add(node);
    for (DependencyNode child : node.getChildren()) {
      if (isPlugin(child)) {
        //Compare by instance
        newChildren.add(copy(child, new IdentityHashMap<>(), d -> true));
      } else if (isApi(child)) {
        newChildren.add(copy(child, new IdentityHashMap<>(), ApiDependencyGraphTransformer::isApi));
      } else {
        this.rebuildIsolatedDependencies(child, visited);
        newChildren.add(child);
      }
    }
    node.setChildren(newChildren);
  }

  private DependencyNode copy(DependencyNode node, Map<DependencyNode, DependencyNode> visitedMapping,
                              Predicate<DependencyNode> shouldCopy) {
    if (visitedMapping.containsKey(node)) {
      return visitedMapping.get(node);
    }
    DependencyNode newNode = new DefaultDependencyNode(node);
    visitedMapping.put(node, newNode);
    List<DependencyNode> children =
        node.getChildren().stream().map(n -> shouldCopy.test(n) ? copy(n, visitedMapping, shouldCopy) : n).collect(toList());
    newNode.setChildren(children);
    return newNode;
  }

}
