/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Collections.newSetFromMap;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static org.mule.maven.client.internal.ApiDependencyGraphTransformer.isApi;
import static org.mule.maven.client.internal.MulePluginDependencyGraphTransformer.isPlugin;

import java.util.IdentityHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

import org.eclipse.aether.RepositoryException;
import org.eclipse.aether.collection.DependencyGraphTransformationContext;
import org.eclipse.aether.collection.DependencyGraphTransformer;
import org.eclipse.aether.graph.DefaultDependencyNode;
import org.eclipse.aether.graph.DependencyNode;

/**
 * {@link DependencyGraphTransformer} implementation that makes sure that there are no {@link DependencyNode} instances
 * shared amongst plugins, apis and non-plugin subtrees.
 * <p>
 * This transformer was created after finding out that the received graph was reusing instances of {@link DependencyNode}.
 * That caused that, in some cases, the logic implemented by {@link MulePluginDependencyGraphTransformer} modified more than
 * the required nodes. As a consequence, the final dependency graph was wrong.
 * <p/>
 * For example, with an application like this:
 * <p>
 *  APP
 *  |__plugin
 *  |    \_lib
 *  |       \_transitive
 *  |__other-lib
 *       \_lib
 *          \_transitive
 * <p/>
 * The {@link DependencyNode} representing the `transitive` dependency, would be the same for all branches.
 * In that case, when {@link MulePluginDependencyGraphTransformer} changes it's artifactId so that it doesn't conflict with non-plugin dependencies,
 * since the {@link DependencyNode} instance is the same, the artifactId will be changed for multiple branches. That causes the dependency to be filtered
 * because it will be considered as if it was only declared as a plugin dependency.
 * <p/>
 * To fix this issue, this implementation will recreate all {@link DependencyNode}s that hang from a plugin's to make sure
 * that if it's modified, it only affects that particular instance.
 * For api dependencies, the logic is similar, but we only recreate other api dependencies. That is because, non-api dependencies
 * should not be considered as dependencies in an "api parent context" and should conflict with other dependencies in the general graph.
 *
 * @since 1.4.3, 1.5.0
 */
public class OneInstancePerNodeGraphTransformer implements DependencyGraphTransformer {

  @Override
  public DependencyNode transformGraph(DependencyNode node, DependencyGraphTransformationContext context)
      throws RepositoryException {
    rebuildIsolatedDependencies(node, newSetFromMap(new IdentityHashMap<>()));
    return node;
  }

  private void rebuildIsolatedDependencies(DependencyNode node, Set<DependencyNode> visited) {
    List<DependencyNode> newChildren = new LinkedList<>();
    if (visited.contains(node)) {
      return;
    }
    visited.add(node);
    for (DependencyNode child : node.getChildren()) {
      if (isPlugin(child) || isApi(child)) {
        newChildren.add(copy(child, new CopyContext(child)));
      } else {
        this.rebuildIsolatedDependencies(child, visited);
        newChildren.add(child);
      }
    }
    node.setChildren(newChildren);
  }

  private DependencyNode copy(DependencyNode node,
                              CopyContext context) {

    return context
        .get(node)
        .orElseGet(
                   () -> {
                     DependencyNode copyNode = new DefaultDependencyNode(node);
                     context.register(node, copyNode);
                     copyNode.setChildren(
                                          node.getChildren().stream().map(c -> {
                                            if (context.shouldCopy(c)) {
                                              return copy(c, context.getContextFor(c));
                                            }
                                            return c;
                                          }).collect(toList()));
                     return copyNode;
                   });
  }

  /**
   * POJO to store information about already visited nodes.
   */
  private static class CopyContext {

    private final Map<DependencyNode, DependencyNode> globalCache;
    private final Map<DependencyNode, DependencyNode> localCache;
    private final Predicate<DependencyNode> copyStrategy;

    private CopyContext(DependencyNode parentNode) {
      this(parentNode, new IdentityHashMap<>(), new IdentityHashMap<>());
    }

    private CopyContext(DependencyNode parentNode,
                        Map<DependencyNode, DependencyNode> globalCache,
                        Map<DependencyNode, DependencyNode> localCache) {
      this.copyStrategy = getCopyStrategy(parentNode);
      this.globalCache = globalCache;
      this.localCache = localCache;
    }

    private Predicate<DependencyNode> getCopyStrategy(DependencyNode parentNode) {
      if (isApi(parentNode)) {
        return ApiDependencyGraphTransformer::isApi;
      }
      return d -> true;
    }

    private Optional<DependencyNode> get(DependencyNode node) {
      if (globalCache.containsKey(node)) {
        return ofNullable(globalCache.get(node));
      }
      if (localCache.containsKey(node)) {
        return ofNullable(localCache.get(node));
      }
      return empty();
    }

    private void register(DependencyNode originalNode, DependencyNode copyNode) {
      if (isPlugin(originalNode) || isApi(originalNode)) {
        this.globalCache.put(originalNode, copyNode);
      } else {
        this.localCache.put(originalNode, copyNode);
      }
    }

    private CopyContext getContextFor(DependencyNode node) {
      if (isPlugin(node)) {
        return new CopyContext(node, this.globalCache, new IdentityHashMap<>());
      }
      return this;
    }

    private boolean shouldCopy(DependencyNode dependencyNode) {
      return copyStrategy.test(dependencyNode);
    }

  }

}
