/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.byteCountToDisplaySize;
import static org.eclipse.aether.repository.RepositoryPolicy.UPDATE_POLICY_ALWAYS;
import static org.eclipse.aether.repository.RepositoryPolicy.UPDATE_POLICY_NEVER;
import static org.eclipse.aether.resolution.ArtifactDescriptorPolicy.STRICT;
import static org.eclipse.aether.resolution.ResolutionErrorPolicy.CACHE_NOT_FOUND;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.internal.dependency.PluginAwareExclusionDependencySelector;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import java.util.ServiceLoader;
import java.util.function.Consumer;

import org.apache.maven.model.resolution.ModelResolver;
import org.apache.maven.repository.internal.DefaultArtifactDescriptorReader;
import org.apache.maven.repository.internal.DefaultVersionRangeResolver;
import org.apache.maven.repository.internal.DefaultVersionResolver;
import org.apache.maven.repository.internal.MavenRepositorySystemUtils;
import org.eclipse.aether.DefaultRepositoryCache;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.collection.DependencyGraphTransformer;
import org.eclipse.aether.collection.DependencySelector;
import org.eclipse.aether.connector.basic.BasicRepositoryConnectorFactory;
import org.eclipse.aether.impl.ArtifactDescriptorReader;
import org.eclipse.aether.impl.ArtifactResolver;
import org.eclipse.aether.impl.DefaultServiceLocator;
import org.eclipse.aether.impl.RemoteRepositoryManager;
import org.eclipse.aether.impl.VersionRangeResolver;
import org.eclipse.aether.impl.VersionResolver;
import org.eclipse.aether.internal.impl.DefaultRepositorySystem;
import org.eclipse.aether.repository.AuthenticationSelector;
import org.eclipse.aether.repository.LocalRepository;
import org.eclipse.aether.repository.MirrorSelector;
import org.eclipse.aether.repository.ProxySelector;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.repository.WorkspaceReader;
import org.eclipse.aether.spi.connector.RepositoryConnectorFactory;
import org.eclipse.aether.spi.connector.transport.TransporterFactory;
import org.eclipse.aether.transfer.TransferCancelledException;
import org.eclipse.aether.transfer.TransferEvent;
import org.eclipse.aether.transfer.TransferListener;
import org.eclipse.aether.transport.file.FileTransporterFactory;
import org.eclipse.aether.transport.http.HttpTransporterFactory;
import org.eclipse.aether.util.graph.selector.AndDependencySelector;
import org.eclipse.aether.util.graph.selector.ExclusionDependencySelector;
import org.eclipse.aether.util.graph.selector.OptionalDependencySelector;
import org.eclipse.aether.util.graph.selector.ScopeDependencySelector;
import org.eclipse.aether.util.graph.transformer.ChainedDependencyGraphTransformer;
import org.eclipse.aether.util.graph.transformer.ConflictResolver;
import org.eclipse.aether.util.graph.transformer.JavaDependencyContextRefiner;
import org.eclipse.aether.util.graph.transformer.JavaScopeDeriver;
import org.eclipse.aether.util.graph.transformer.JavaScopeSelector;
import org.eclipse.aether.util.graph.transformer.NearestVersionSelector;
import org.eclipse.aether.util.graph.transformer.SimpleOptionalitySelector;
import org.eclipse.aether.util.repository.SimpleResolutionErrorPolicy;
import org.slf4j.Logger;

/**
 * Holds the state for aether repository state for the resolution of dependencies of a particular artifact.
 */
public class AetherRepositoryState {

  private static final Logger LOGGER = getLogger(AetherRepositoryState.class);

  private DefaultServiceLocator locator;

  private DefaultRepositorySystemSession session;
  private RepositorySystem system;

  public AetherRepositoryState(File localRepositoryFolder, Optional<WorkspaceReader> workspaceReader,
                               Optional<AuthenticationSelector> authenticationSelector, Optional<ProxySelector> proxySelector,
                               Optional<MirrorSelector> mirrorSelector, boolean noSnapshotUpdates, boolean offline,
                               boolean ignoreArtifactDescriptorRepositories,
                               Optional<Properties> userProperties,
                               Consumer<DefaultRepositorySystemSession> sessionConfigurator) {
    this(localRepositoryFolder, workspaceReader, offline, ignoreArtifactDescriptorRepositories,
         authenticationSelector, proxySelector, mirrorSelector, noSnapshotUpdates, false, userProperties, sessionConfigurator,
         null);
  }

  public AetherRepositoryState(File localRepositoryFolder, Optional<WorkspaceReader> workspaceReader,
                               Optional<AuthenticationSelector> authenticationSelector, Optional<ProxySelector> proxySelector,
                               Optional<MirrorSelector> mirrorSelector, boolean noSnapshotUpdates, boolean updateSnapshots,
                               boolean offline,
                               boolean ignoreArtifactDescriptorRepositories,
                               Optional<Properties> userProperties,
                               Consumer<DefaultRepositorySystemSession> sessionConfigurator, String globalChecksumPolicy) {
    this(localRepositoryFolder, workspaceReader, offline, ignoreArtifactDescriptorRepositories,
         authenticationSelector, proxySelector, mirrorSelector, noSnapshotUpdates, updateSnapshots, userProperties,
         sessionConfigurator, globalChecksumPolicy);
  }

  private AetherRepositoryState(File localRepositoryFolder, Optional<WorkspaceReader> workspaceReader, boolean offline,
                                boolean ignoreArtifactDescriptorRepositories,
                                Optional<AuthenticationSelector> authenticationSelector,
                                Optional<ProxySelector> proxySelector, Optional<MirrorSelector> mirrorSelector,
                                boolean noSnapshotUpdates,
                                boolean updateSnapshots,
                                Optional<Properties> userProperties,
                                Consumer<DefaultRepositorySystemSession> sessionConfigurator,
                                String globalChecksumPolicy) {
    createRepositorySession(localRepositoryFolder, workspaceReader, offline, ignoreArtifactDescriptorRepositories,
                            authenticationSelector, proxySelector, mirrorSelector, noSnapshotUpdates, updateSnapshots,
                            userProperties,
                            sessionConfigurator, globalChecksumPolicy);
  }

  private void createRepositorySession(File localRepositoryFolder, Optional<WorkspaceReader> workspaceReader, boolean offline,
                                       boolean ignoreArtifactDescriptorRepositories,
                                       Optional<AuthenticationSelector> authenticationSelector,
                                       Optional<ProxySelector> proxySelector,
                                       Optional<MirrorSelector> mirrorSelector,
                                       boolean noSnapshotUpdates,
                                       boolean updateSnapshots,
                                       Optional<Properties> userProperties,
                                       Consumer<DefaultRepositorySystemSession> sessionConfigurator,
                                       String globalChecksumPolicy) {
    session = newDefaultRepositorySystemSession(noSnapshotUpdates, updateSnapshots, sessionConfigurator, globalChecksumPolicy);
    RepositorySystem repositorySystem = createRepositorySystem();

    session.setLocalRepositoryManager(repositorySystem
        .newLocalRepositoryManager(session, new LocalRepository(localRepositoryFolder)));
    session.setOffline(offline);
    session.setConfigProperty(ConflictResolver.CONFIG_PROP_VERBOSE, true);
    proxySelector.ifPresent(session::setProxySelector);
    mirrorSelector.ifPresent(session::setMirrorSelector);
    authenticationSelector.ifPresent(session::setAuthenticationSelector);

    DependencyGraphTransformer transformer =
        new ConflictResolver(new MuleVersionSelector(new NearestVersionSelector()), new JavaScopeSelector(),
                             new SimpleOptionalitySelector(), new JavaScopeDeriver());

    List<DependencyGraphTransformer> graphTransformers = new ArrayList<>();
    graphTransformers.add(new OneInstancePerNodeGraphTransformer());
    graphTransformers.add(new MuleDomainDependencyGraphTransformer());
    graphTransformers.add(new ApiDependencyGraphTransformer());
    graphTransformers.add(new MulePluginDependencyGraphTransformer());
    graphTransformers.add(transformer);
    graphTransformers.add(new JavaDependencyContextRefiner());
    graphTransformers.add(new ArtifactRestorerTransformer());

    session.setDependencyGraphTransformer(new ChainedDependencyGraphTransformer(graphTransformers
        .toArray(new DependencyGraphTransformer[0])));

    //This is how the default dependency selector is created, except from our own implementation.
    DependencySelector dependencySelector = new AndDependencySelector(
                                                                      new ScopeDependencySelector("test", "provided"),
                                                                      new OptionalDependencySelector(),
                                                                      new PluginAwareExclusionDependencySelector(new ExclusionDependencySelector()));

    session.setDependencySelector(dependencySelector);

    session.setArtifactDescriptorPolicy((session, request) -> STRICT);
    session.setIgnoreArtifactDescriptorRepositories(ignoreArtifactDescriptorRepositories);
    workspaceReader.ifPresent(session::setWorkspaceReader);
    userProperties.ifPresent(properties -> session.setUserProperties(properties));
    system = repositorySystem;
  }

  private RepositorySystem createRepositorySystem() {
    locator = new DefaultServiceLocator();
    locator.addService(RepositoryConnectorFactory.class, BasicRepositoryConnectorFactory.class);
    locator.addService(TransporterFactory.class, FileTransporterFactory.class);
    locator.addService(RepositorySystem.class, DefaultRepositorySystem.class);
    locator.addService(TransporterFactory.class, HttpTransporterFactory.class);
    locator.addService(VersionResolver.class, DefaultVersionResolver.class);
    locator.addService(VersionRangeResolver.class, DefaultVersionRangeResolver.class);
    locator.addService(ArtifactDescriptorReader.class, DefaultArtifactDescriptorReader.class);
    locator.setErrorHandler(new DefaultServiceLocator.ErrorHandler() {

      @Override
      public void serviceCreationFailed(Class<?> type, Class<?> impl, Throwable exception) {
        LOGGER.warn(exception.getMessage());
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug(exception.getMessage(), exception);
        }
      }
    });

    // Customization of Aether Services
    loadAetherServiceRegister().ifPresent(aetherServiceRegister -> aetherServiceRegister.registerServices(locator));

    return locator.getService(RepositorySystem.class);
  }

  /**
   * Use the ServiceLoader mechanism to load (if present) a {@link AetherServiceRegister}.
   *
   * @return {@link AetherServiceRegister}
   */
  private Optional<AetherServiceRegister> loadAetherServiceRegister() {
    for (AetherServiceRegister serviceRegister : ServiceLoader.load(AetherServiceRegister.class)) {
      return of(serviceRegister);
    }
    return empty();
  }

  public DefaultRepositorySystemSession getSession() {
    return session;
  }

  public RepositorySystem getSystem() {
    return system;
  }

  public ModelResolver createModelResolver(List<RemoteRepository> repositories) {
    return new DefaultModelResolver(session, locator.getService(ArtifactResolver.class),
                                    locator.getService(VersionRangeResolver.class),
                                    getRemoteRepositoryManager(),
                                    repositories);
  }

  private static DefaultRepositorySystemSession newDefaultRepositorySystemSession(boolean noUpdateSnapshots,
                                                                                  boolean updateSnapshots,
                                                                                  Consumer<DefaultRepositorySystemSession> sessionConfigurator,
                                                                                  String globalChecksumPolicy) {
    final DefaultRepositorySystemSession session = MavenRepositorySystemUtils.newSession();
    session.setResolutionErrorPolicy(new SimpleResolutionErrorPolicy(CACHE_NOT_FOUND));
    session.setCache(new DefaultRepositoryCache());
    if (noUpdateSnapshots) {
      session.setUpdatePolicy(UPDATE_POLICY_NEVER);
    } else if (updateSnapshots) {
      session.setUpdatePolicy(UPDATE_POLICY_ALWAYS);
    } else {
      session.setUpdatePolicy(null);
    }
    if (globalChecksumPolicy != null) {
      session.setChecksumPolicy(globalChecksumPolicy);
    }
    session.setTransferListener(new LoggingTransferListener());
    sessionConfigurator.accept(session);
    return session;
  }

  public void setWorkspaceReader(WorkspaceReader workspaceReader) {
    this.session.setWorkspaceReader(workspaceReader);
  }

  public RemoteRepositoryManager getRemoteRepositoryManager() {
    return locator.getService(RemoteRepositoryManager.class);
  }

  private static final class LoggingTransferListener implements TransferListener {

    @Override
    public void transferSucceeded(TransferEvent event) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Transfer {} for '{}' from {}", event.getType(), event.getResource().getResourceName(),
                     event.getResource().getRepositoryUrl());
      }
    }

    @Override
    public void transferStarted(TransferEvent event) throws TransferCancelledException {
      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace("Transfer {} for '{}'", event.getType(), event.getResource().getResourceName());
      }
    }

    @Override
    public void transferProgressed(TransferEvent event) throws TransferCancelledException {
      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace("Transfer {} for '{}' ({}/{})", event.getType(), event.getResource().getResourceName(),
                     byteCountToDisplaySize(event.getTransferredBytes()),
                     byteCountToDisplaySize(event.getResource().getContentLength()));
      }
    }

    @Override
    public void transferInitiated(TransferEvent event) throws TransferCancelledException {
      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace("Transfer {} for '{}'", event.getType(), event.getResource().getResourceName());
      }
    }

    @Override
    public void transferFailed(TransferEvent event) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Transfer {} for '{}' ({})", event.getType(), event.getResource(), event.getException());
      }
    }

    @Override
    public void transferCorrupted(TransferEvent event) throws TransferCancelledException {
      LOGGER.warn("Transfer {} for '{}' ({})", event.getType(), event.getResource(), event.getException());
    }

  }

}
