/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Optional.ofNullable;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public class MavenCommandLineParser {

  private MavenCommandLineParser() {}

  public static MavenArguments parseMavenArguments(MavenEnvironmentVariables mavenEnvironmentVariables) {
    MavenArguments mavenArguments = new MavenArguments();
    JCommander.newBuilder()
        .addObject(mavenArguments)
        .acceptUnknownOptions(true)
        .allowParameterOverwriting(true)
        .build()
        .parse(getMavenCmdLineArgs(mavenEnvironmentVariables));
    return mavenArguments;
  }

  private static String[] getMavenCmdLineArgs(MavenEnvironmentVariables mavenEnvironmentVariables) {
    String mavenCmdLineArgs = mavenEnvironmentVariables.getMavenCmdLineArgsEnv();
    if (mavenCmdLineArgs != null) {
      return mavenCmdLineArgs.trim().split(" ");
    }
    return new String[0];
  }

  /**
   * POJO to be created by JCommander when parsing the command line arguments from Maven.
   * It supports as Maven command line duplicate entries for settings and global settings which in that case uses the first
   * one.
   */
  public static class MavenArguments {

    @Parameter(names = {"-s", "--settings"}, description = "User settings")
    private List<File> settings = new ArrayList<>();

    @Parameter(names = {"-gs", "--global-settings"}, description = "Global settings")
    private List<File> globalSettings = new ArrayList<>();

    @Parameter(names = {"-C", "--strict-checksums"}, description = "Strict checksums")
    private boolean strictChecksums = false;

    @Parameter(names = {"-c", "--lax-checksums"}, description = "Lax checksums")
    private boolean laxChecksums = false;

    public Optional<File> getSettings() {
      return ofNullable(settings.size() >= 1 ? settings.get(0) : null);
    }

    public Optional<File> getGlobalSettings() {
      return ofNullable(globalSettings.size() >= 1 ? globalSettings.get(0) : null);
    }

    public boolean isStrictChecksums() {
      return strictChecksums;
    }

    public boolean isLaxChecksums() {
      return laxChecksums;
    }
  }


}
