/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Collections.emptyList;
import static org.eclipse.aether.util.StringUtils.isEmpty;
import static org.eclipse.aether.util.graph.transformer.ConflictResolver.NODE_DATA_WINNER;
import static org.mule.maven.client.internal.AetherMavenClient.MULE_PLUGIN_CLASSIFIER;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.BundleScope;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.graph.DependencyFilter;
import org.eclipse.aether.graph.DependencyNode;

/**
 * Utility class that converts a dependency node into a BundleDependency by also converting the transitive dependencies and
 * caching the results for future usages. Since Mule Plugins will have it's dependencies isolated from each other and the
 * deployable artifact itself, the cache should consider that to avoid returning an already cached dependency from a different
 * plugin.
 */
public class BundleDependencyHelper {

  private static final String ROOT_CONTEXT = "ROOT";

  private Map<String, Map<BundleDescriptor, BundleDependency>> contexts = new HashMap<>();

  public BundleDependency getBundleDependency(DependencyNode node, DependencyFilter transitiveDependencyFilter) {
    Map<BundleDescriptor, BundleDependency> context = contexts.computeIfAbsent(ROOT_CONTEXT, k -> new HashMap<>());
    return getBundleDependency(node, context, transitiveDependencyFilter);
  }

  private BundleDependency getBundleDependency(DependencyNode node, Map<BundleDescriptor, BundleDependency> context,
                                               DependencyFilter transitiveDependencyFilter) {
    DependencyNode effectiveNodeDependency = node;
    if (node.getData().containsKey(NODE_DATA_WINNER)) {
      effectiveNodeDependency = (DependencyNode) node.getData().get(NODE_DATA_WINNER);
    }
    Artifact artifact = effectiveNodeDependency.getArtifact();
    BundleDescriptor bundleDescriptor = artifactToBundleDescriptor(artifact).build();
    Map<BundleDescriptor, BundleDependency> realContext =
        MULE_PLUGIN_CLASSIFIER.equals(artifact.getClassifier()) ? contexts.get(ROOT_CONTEXT) : context;
    BundleDependency cachedValue = realContext.get(bundleDescriptor);
    if (cachedValue != null) {
      return cachedValue;
    }
    String scope = effectiveNodeDependency.getDependency().getScope();


    BundleScope bundleScope = StringUtils.isEmpty(scope) ? BundleScope.COMPILE : BundleScope.valueOf(scope.toUpperCase());
    BundleDependency.Builder bundleDependencyBuilder =
        artifactToBundleDependency(effectiveNodeDependency.getArtifact(), bundleScope.name());
    Map<BundleDescriptor, BundleDependency> nextContext = getContext(effectiveNodeDependency, context);
    effectiveNodeDependency.getChildren()
        .stream()
        .map(c -> c.getData().containsKey(NODE_DATA_WINNER) ? (DependencyNode) c.getData().get(NODE_DATA_WINNER) : c)
        .filter(children -> transitiveDependencyFilter.accept(children, emptyList()))
        .forEach(childNode -> bundleDependencyBuilder
            .addTransitiveDependency(getBundleDependency(childNode, nextContext, transitiveDependencyFilter)));
    BundleDependency bundleDependency = bundleDependencyBuilder.build();
    realContext.put(bundleDependency.getDescriptor(), bundleDependency);
    return bundleDependencyBuilder.build();
  }

  static BundleDependency.Builder artifactToBundleDependency(Artifact artifact, String scope) {
    final BundleDescriptor.Builder bundleDescriptorBuilder = artifactToBundleDescriptor(artifact);

    BundleScope bundleScope = StringUtils.isEmpty(scope) ? BundleScope.COMPILE : BundleScope.valueOf(scope.toUpperCase());
    BundleDependency.Builder builder = new BundleDependency.Builder()
        .setDescriptor(bundleDescriptorBuilder.build())
        .setScope(bundleScope);
    if (artifact.getFile() != null) {
      builder.setBundleUri(artifact.getFile().toURI());
    }
    return builder;
  }

  static BundleDescriptor.Builder artifactToBundleDescriptor(Artifact artifact) {
    final BundleDescriptor.Builder bundleDescriptorBuilder = new BundleDescriptor.Builder()
        .setArtifactId(artifact.getArtifactId())
        .setGroupId(artifact.getGroupId())
        .setVersion(artifact.getVersion())
        .setBaseVersion(artifact.getBaseVersion())
        .setType(artifact.getExtension());

    String classifier = artifact.getClassifier();
    if (!isEmpty(classifier)) {
      bundleDescriptorBuilder.setClassifier(classifier);
    }

    return bundleDescriptorBuilder;
  }

  private Map<BundleDescriptor, BundleDependency> getContext(DependencyNode node,
                                                             Map<BundleDescriptor, BundleDependency> parentContext) {
    // Returns the context for the resolution of dependencies. Each MULE_PLUGIN will have a new context to cache it's dependencies
    // and avoid computing duplicates multiple times.
    if (MULE_PLUGIN_CLASSIFIER.equals(node.getArtifact().getClassifier())) {
      String key = node.getArtifact().getGroupId() + ":" + node.getArtifact().getArtifactId();
      return contexts.computeIfAbsent(key, k -> new HashMap<>());
    }
    return parentContext;
  }
}
