/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.stream.Collectors.toList;

import com.google.common.collect.ImmutableList;

import java.util.List;

import org.eclipse.aether.repository.RemoteRepository;

/**
 * Utility class that merges a list of {@link org.eclipse.aether.repository.RemoteRepository Remote Repostiories} by identifier.
 */
public class RemoteRepositoriesMerger {

  /**
   * Merges the two lists of remote repositories by their IDs. Considering the fist to be the dominant list, therefore starting by
   * that one and adding missing repositories from the {@code recessive} list.
   *
   * @param dominant  original list of {@link RemoteRepository repositories} that should have authentication already set.
   * @param recessive list of {@link RemoteRepository repositories} to be added if there is not already one in dominant list for
   *                  the same ID.
   * @return merged {@link List} of {@link RemoteRepository}.
   */
  public List<RemoteRepository> merge(List<RemoteRepository> dominant, List<RemoteRepository> recessive) {
    ImmutableList.Builder<RemoteRepository> mergedRepositories = ImmutableList.builder();
    mergedRepositories.addAll(dominant);

    List<String> dominantRepositoriesIds = dominant.stream().map(remoteRepository -> remoteRepository.getId()).collect(toList());

    recessive.stream().forEach(remoteRepository -> {
      if (!dominantRepositoriesIds.contains(remoteRepository.getId())) {
        mergedRepositories.add(remoteRepository);
      }
    });

    return mergedRepositories.build();
  }

}
