/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.dependency;

import static org.mule.maven.client.internal.MuleMavenClient.MULE_PLUGIN_CLASSIFIER;

import org.eclipse.aether.collection.DependencyCollectionContext;
import org.eclipse.aether.collection.DependencySelector;
import org.eclipse.aether.graph.Dependency;

/**
 * {@link DependencySelector} that knows how to handle exclusions from mule plugins.
 */
public class PluginAwareExclusionDependencySelector implements DependencySelector {

  private DependencySelector delegateSelector;
  private boolean isPluginContext = false;

  public PluginAwareExclusionDependencySelector(DependencySelector delegateSelector) {
    this.delegateSelector = delegateSelector;
  }

  public PluginAwareExclusionDependencySelector(DependencySelector delegateSelector, boolean isPluginContext) {
    this.delegateSelector = delegateSelector;
    this.isPluginContext = isPluginContext;
  }

  @Override
  public boolean selectDependency(Dependency dependency) {
    if (isPluginContext) {
      if (!(MULE_PLUGIN_CLASSIFIER.equals(dependency.getArtifact().getClassifier()))) {
        return true;
      }
    }
    return delegateSelector.selectDependency(dependency);
  }

  @Override
  public DependencySelector deriveChildSelector(DependencyCollectionContext context) {
    // If the context is for a mule plugin, then return a selector that can handle logic for selecting plugin exclusions,
    // otherwise, delegate to the default behaviour.
    boolean isPluginContext = context.getDependency() != null
        && MULE_PLUGIN_CLASSIFIER.equals(context.getDependency().getArtifact().getClassifier());
    DependencySelector dependencySelector = delegateSelector.deriveChildSelector(context);
    if (dependencySelector.equals(delegateSelector) && isPluginContext == this.isPluginContext) {
      return this;
    }
    return new PluginAwareExclusionDependencySelector(dependencySelector, isPluginContext);
  }

  @Override
  public int hashCode() {
    return delegateSelector.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof PluginAwareExclusionDependencySelector)) {
      return false;
    }
    return delegateSelector.equals(((PluginAwareExclusionDependencySelector) obj).delegateSelector);
  }

}
