/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.version;

import static java.lang.String.format;

import org.mule.maven.client.api.VersionUtils;

import java.util.List;

import org.eclipse.aether.version.InvalidVersionSpecificationException;

public class DefaultVersionUtils implements VersionUtils {

  /**
   * Determines whether the specified version satisfies this constraint. In more detail, a version satisfies this constraint if it
   * matches its version range or if this constraint has no version range and the specified version equals the version recommended
   * by the constraint.
   * 
   * @param version     The version to test, must not be {@code null}.
   * @param constraints The versions to test against, must not be {@code null}.
   * @return {@code true} if the specified version satisfies any of the constraints, {@code false} otherwise.
   */
  @Override
  public boolean containsVersion(String version, List<String> constraints) throws IllegalArgumentException {
    MavenVersion thisArtifactCoordinatesVersion;
    try {
      thisArtifactCoordinatesVersion = new MavenVersion(version);
    } catch (InvalidVersionSpecificationException e) {
      throw new IllegalArgumentException(format("Error parsing version %s", version), e);
    }

    return constraints
        .stream()
        .map(constraint -> {
          try {
            return new MavenVersionConstraintParser(constraint);
          } catch (InvalidVersionSpecificationException e) {
            throw new IllegalArgumentException(format("Error parsing constraint %s", constraint), e);
          }
        })
        .anyMatch(mavenVersionConstraintParser -> mavenVersionConstraintParser.containsVersion(thisArtifactCoordinatesVersion));
  }
}
