/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.repository;

import org.mule.maven.client.internal.wagon.SimplePlexusWagonConfigurator;
import org.mule.maven.client.internal.wagon.SimplePlexusWagonProvider;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.impl.ArtifactResolver;
import org.eclipse.aether.impl.MetadataResolver;
import org.eclipse.aether.impl.OfflineController;
import org.eclipse.aether.impl.RemoteRepositoryFilterManager;
import org.eclipse.aether.impl.RemoteRepositoryManager;
import org.eclipse.aether.impl.RepositoryConnectorProvider;
import org.eclipse.aether.impl.RepositoryEventDispatcher;
import org.eclipse.aether.impl.UpdateCheckManager;
import org.eclipse.aether.impl.UpdatePolicyAnalyzer;
import org.eclipse.aether.impl.VersionRangeResolver;
import org.eclipse.aether.impl.VersionResolver;
import org.eclipse.aether.spi.connector.checksum.ChecksumPolicyProvider;
import org.eclipse.aether.spi.connector.transport.TransporterFactory;
import org.eclipse.aether.spi.io.FileProcessor;
import org.eclipse.aether.spi.resolution.ArtifactResolverPostProcessor;
import org.eclipse.aether.spi.synccontext.SyncContextFactory;
import org.eclipse.aether.supplier.RepositorySystemSupplier;
import org.eclipse.aether.transport.file.FileTransporterFactory;
import org.eclipse.aether.transport.http.ChecksumExtractor;
import org.eclipse.aether.transport.wagon.WagonTransporterFactory;

/**
 * {@link MuleRepositorySystemManager} implementation to be used when running outside a Maven context.
 *
 * @since 2.0
 */
public class StandaloneMuleRepositorySystemManager extends RepositorySystemSupplier implements MuleRepositorySystemManager {

  private final RepositorySystem repositorySystem;
  private ArtifactResolver artifactResolver;
  private VersionRangeResolver versionRangeResolver;
  private RemoteRepositoryManager remoteRepositoryManager;

  public StandaloneMuleRepositorySystemManager() {
    repositorySystem = this.get();
  }

  @Override
  protected Map<String, TransporterFactory> getTransporterFactories(Map<String, ChecksumExtractor> extractors) {
    HashMap<String, TransporterFactory> result = new HashMap<>();
    result.put("file", new FileTransporterFactory());
    result.put("wagon", new WagonTransporterFactory(new SimplePlexusWagonProvider(), new SimplePlexusWagonConfigurator()));
    return result;
  }

  @Override
  protected ArtifactResolver getArtifactResolver(FileProcessor fileProcessor, RepositoryEventDispatcher repositoryEventDispatcher,
                                                 VersionResolver versionResolver, UpdateCheckManager updateCheckManager,
                                                 RepositoryConnectorProvider repositoryConnectorProvider,
                                                 RemoteRepositoryManager remoteRepositoryManager,
                                                 SyncContextFactory syncContextFactory, OfflineController offlineController,
                                                 Map<String, ArtifactResolverPostProcessor> artifactResolverPostProcessors,
                                                 RemoteRepositoryFilterManager remoteRepositoryFilterManager) {
    artifactResolver =
        super.getArtifactResolver(fileProcessor, repositoryEventDispatcher, versionResolver, updateCheckManager,
                                  repositoryConnectorProvider, remoteRepositoryManager, syncContextFactory, offlineController,
                                  artifactResolverPostProcessors, remoteRepositoryFilterManager);
    return artifactResolver;
  }

  @Override
  protected VersionRangeResolver getVersionRangeResolver(MetadataResolver metadataResolver, SyncContextFactory syncContextFactory,
                                                         RepositoryEventDispatcher repositoryEventDispatcher) {
    versionRangeResolver = super.getVersionRangeResolver(metadataResolver, syncContextFactory, repositoryEventDispatcher);
    return versionRangeResolver;
  }

  @Override
  protected RemoteRepositoryManager getRemoteRepositoryManager(UpdatePolicyAnalyzer updatePolicyAnalyzer,
                                                               ChecksumPolicyProvider checksumPolicyProvider) {
    remoteRepositoryManager = super.getRemoteRepositoryManager(updatePolicyAnalyzer, checksumPolicyProvider);
    return remoteRepositoryManager;
  }

  @Override
  public RepositorySystem getRepositorySystem() {
    return repositorySystem;
  }

  @Override
  public ArtifactResolver getArtifactResolver() {
    return artifactResolver;
  }

  @Override
  public VersionRangeResolver getVersionRangeResolver() {
    return versionRangeResolver;
  }

  @Override
  public RemoteRepositoryManager getRemoteRepositoryManager() {
    return remoteRepositoryManager;
  }

  @Override
  public void close() {
    repositorySystem.shutdown();
  }

}
