/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.lang.Boolean.getBoolean;
import static java.lang.System.getProperties;
import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.ServiceLoader.load;

import static org.apache.commons.io.FileUtils.byteCountToDisplaySize;
import static org.eclipse.aether.repository.RepositoryPolicy.UPDATE_POLICY_ALWAYS;
import static org.eclipse.aether.repository.RepositoryPolicy.UPDATE_POLICY_NEVER;
import static org.eclipse.aether.resolution.ArtifactDescriptorPolicy.STRICT;
import static org.eclipse.aether.resolution.ResolutionErrorPolicy.CACHE_NOT_FOUND;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.dependency.PluginAwareExclusionDependencySelector;
import org.mule.maven.client.internal.repository.MuleRepositorySystemManager;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Consumer;

import org.apache.maven.repository.internal.MavenRepositorySystemUtils;
import org.eclipse.aether.DefaultRepositoryCache;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.collection.DependencyGraphTransformer;
import org.eclipse.aether.collection.DependencySelector;
import org.eclipse.aether.internal.impl.synccontext.named.NameMappers;
import org.eclipse.aether.named.providers.FileLockNamedLockFactory;
import org.eclipse.aether.repository.AuthenticationSelector;
import org.eclipse.aether.repository.LocalRepository;
import org.eclipse.aether.repository.MirrorSelector;
import org.eclipse.aether.repository.ProxySelector;
import org.eclipse.aether.repository.WorkspaceReader;
import org.eclipse.aether.transfer.TransferCancelledException;
import org.eclipse.aether.transfer.TransferEvent;
import org.eclipse.aether.transfer.TransferListener;
import org.eclipse.aether.util.graph.selector.AndDependencySelector;
import org.eclipse.aether.util.graph.selector.ExclusionDependencySelector;
import org.eclipse.aether.util.graph.selector.OptionalDependencySelector;
import org.eclipse.aether.util.graph.selector.ScopeDependencySelector;
import org.eclipse.aether.util.graph.transformer.ChainedDependencyGraphTransformer;
import org.eclipse.aether.util.graph.transformer.ConflictResolver;
import org.eclipse.aether.util.graph.transformer.JavaDependencyContextRefiner;
import org.eclipse.aether.util.graph.transformer.JavaScopeDeriver;
import org.eclipse.aether.util.graph.transformer.JavaScopeSelector;
import org.eclipse.aether.util.graph.transformer.NearestVersionSelector;
import org.eclipse.aether.util.graph.transformer.SimpleOptionalitySelector;
import org.eclipse.aether.util.repository.SimpleResolutionErrorPolicy;
import org.slf4j.Logger;

public class MuleMavenRepositoryStateFactory implements AutoCloseable {

  private static final Logger LOGGER = getLogger(MuleMavenRepositoryStateFactory.class);

  public static final String MULE_MAVEN_CLIENT_CONCURRENT_LOCAL_REPOSITORY_PROPERTY =
      "mule.maven.client.concurrent.local.repository";

  // Maven Resolver lock properties' names
  private static final String LOCK_FACTORY_KEY = "aether.syncContext.named.factory";
  private static final String LOCK_NAME_MAPPER_KEY = "aether.syncContext.named.nameMapper";

  private final boolean isConcurrentLocalRepositoryEnabled = isConcurrentLocalRepositoryEnabled();

  private final MuleRepositorySystemManager muleRepositorySystemManager = MuleRepositorySystemManager.getManager();

  private boolean isConcurrentLocalRepositoryEnabled() {
    return getBoolean(MULE_MAVEN_CLIENT_CONCURRENT_LOCAL_REPOSITORY_PROPERTY) || loadConcurrentLocalRepository()
        .map(MuleMavenConcurrentLocalRepository::isConcurrentLocalRepositoryEnabled).orElse(false);
  }

  public MuleMavenRepositoryState createMavenRepositoryState(File localRepositoryLocation,
                                                             MuleMavenResolutionContext muleMavenResolutionContext,
                                                             MavenConfiguration mavenConfiguration,
                                                             Optional<Consumer<DefaultRepositorySystemSession>> sessionConfigurator,
                                                             Optional<WorkspaceReader> workspaceReader) {

    return createMavenRepositoryState(localRepositoryLocation, workspaceReader,
                                      muleMavenResolutionContext.getAuthenticatorSelector(),
                                      muleMavenResolutionContext.getProxySelector(),
                                      muleMavenResolutionContext.getMirrorSelector(),
                                      muleMavenResolutionContext.getServerConfigurations(),
                                      mavenConfiguration.getForcePolicyUpdateNever(),
                                      mavenConfiguration.getForcePolicyUpdateAlways(),
                                      mavenConfiguration.getOfflineMode(),
                                      mavenConfiguration.getIgnoreArtifactDescriptorRepositories(),
                                      mavenConfiguration.getUserProperties(),
                                      sessionConfigurator.orElse(session -> {
                                      }),
                                      mavenConfiguration.getGlobalChecksumPolicy());
  }

  public MuleMavenRepositoryState createMavenRepositoryState(File localRepositoryFolder,
                                                             Optional<WorkspaceReader> workspaceReader,
                                                             Optional<AuthenticationSelector> authenticationSelector,
                                                             Optional<ProxySelector> proxySelector,
                                                             Optional<MirrorSelector> mirrorSelector,
                                                             Map<String, Object> serverConfigurations,
                                                             boolean noSnapshotUpdates, boolean offline,
                                                             boolean ignoreArtifactDescriptorRepositories,
                                                             Optional<Properties> userProperties,
                                                             Consumer<DefaultRepositorySystemSession> sessionConfigurator) {
    return createMavenRepositoryState(localRepositoryFolder, workspaceReader, offline, ignoreArtifactDescriptorRepositories,
                                      authenticationSelector, proxySelector, mirrorSelector, serverConfigurations,
                                      noSnapshotUpdates, false, userProperties,
                                      sessionConfigurator,
                                      null);
  }

  public MuleMavenRepositoryState createMavenRepositoryState(File localRepositoryFolder,
                                                             Optional<WorkspaceReader> workspaceReader,
                                                             Optional<AuthenticationSelector> authenticationSelector,
                                                             Optional<ProxySelector> proxySelector,
                                                             Optional<MirrorSelector> mirrorSelector,
                                                             Map<String, Object> serverConfigurations,
                                                             boolean noSnapshotUpdates, boolean updateSnapshots,
                                                             boolean offline,
                                                             boolean ignoreArtifactDescriptorRepositories,
                                                             Optional<Properties> userProperties,
                                                             Consumer<DefaultRepositorySystemSession> sessionConfigurator,
                                                             String globalChecksumPolicy) {
    return createMavenRepositoryState(localRepositoryFolder, workspaceReader, offline, ignoreArtifactDescriptorRepositories,
                                      authenticationSelector, proxySelector, mirrorSelector, serverConfigurations,
                                      noSnapshotUpdates, updateSnapshots,
                                      userProperties,
                                      sessionConfigurator, globalChecksumPolicy);
  }

  public MuleMavenRepositoryState createMavenRepositoryState(File localRepositoryFolder,
                                                             Optional<WorkspaceReader> workspaceReader,
                                                             Optional<AuthenticationSelector> authenticationSelector,
                                                             Optional<ProxySelector> proxySelector,
                                                             Optional<MirrorSelector> mirrorSelector,
                                                             boolean noSnapshotUpdates, boolean updateSnapshots,
                                                             boolean offline,
                                                             boolean ignoreArtifactDescriptorRepositories,
                                                             Optional<Properties> userProperties,
                                                             Consumer<DefaultRepositorySystemSession> sessionConfigurator,
                                                             String globalChecksumPolicy) {
    return createMavenRepositoryState(localRepositoryFolder, workspaceReader, offline, ignoreArtifactDescriptorRepositories,
                                      authenticationSelector, proxySelector, mirrorSelector, emptyMap(), noSnapshotUpdates,
                                      updateSnapshots,
                                      userProperties,
                                      sessionConfigurator, globalChecksumPolicy);
  }

  private MuleMavenRepositoryState createMavenRepositoryState(File localRepositoryFolder,
                                                              Optional<WorkspaceReader> workspaceReader, boolean offline,
                                                              boolean ignoreArtifactDescriptorRepositories,
                                                              Optional<AuthenticationSelector> authenticationSelector,
                                                              Optional<ProxySelector> proxySelector,
                                                              Optional<MirrorSelector> mirrorSelector,
                                                              Map<String, Object> serverConfigurations,
                                                              boolean noSnapshotUpdates,
                                                              boolean updateSnapshots,
                                                              Optional<Properties> userProperties,
                                                              Consumer<DefaultRepositorySystemSession> sessionConfigurator,
                                                              String globalChecksumPolicy) {
    DefaultRepositorySystemSession repositorySystemSession =
        newDefaultRepositorySystemSession(noSnapshotUpdates, updateSnapshots, sessionConfigurator, globalChecksumPolicy,
                                          serverConfigurations);
    RepositorySystem repositorySystem = muleRepositorySystemManager.getRepositorySystem();

    repositorySystemSession.setSystemProperties(getProperties());
    repositorySystemSession.setLocalRepositoryManager(repositorySystem
        .newLocalRepositoryManager(repositorySystemSession, new LocalRepository(localRepositoryFolder)));
    repositorySystemSession.setOffline(offline);
    repositorySystemSession.setConfigProperty(ConflictResolver.CONFIG_PROP_VERBOSE, true);
    proxySelector.ifPresent(repositorySystemSession::setProxySelector);
    mirrorSelector.ifPresent(repositorySystemSession::setMirrorSelector);
    authenticationSelector.ifPresent(repositorySystemSession::setAuthenticationSelector);

    DependencyGraphTransformer transformer =
        new ConflictResolver(new MuleVersionSelector(new NearestVersionSelector()), new JavaScopeSelector(),
                             new SimpleOptionalitySelector(), new JavaScopeDeriver());

    List<DependencyGraphTransformer> graphTransformers = new ArrayList<>();
    graphTransformers.add(new OneInstancePerNodeGraphTransformer());
    graphTransformers.add(new MuleDomainDependencyGraphTransformer());
    graphTransformers.add(new ApiDependencyGraphTransformer());
    graphTransformers.add(new MulePluginDependencyGraphTransformer());
    graphTransformers.add(transformer);
    graphTransformers.add(new JavaDependencyContextRefiner());
    graphTransformers.add(new ArtifactRestorerTransformer());

    repositorySystemSession.setDependencyGraphTransformer(new ChainedDependencyGraphTransformer(graphTransformers
        .toArray(new DependencyGraphTransformer[0])));

    // This is how the default dependency selector is created, except from our own implementation.
    DependencySelector dependencySelector = new AndDependencySelector(
                                                                      new ScopeDependencySelector("test", "provided"),
                                                                      new OptionalDependencySelector(),
                                                                      new PluginAwareExclusionDependencySelector(new ExclusionDependencySelector()));

    repositorySystemSession.setDependencySelector(dependencySelector);

    repositorySystemSession.setArtifactDescriptorPolicy((session, request) -> STRICT);
    repositorySystemSession.setIgnoreArtifactDescriptorRepositories(ignoreArtifactDescriptorRepositories);
    workspaceReader.ifPresent(repositorySystemSession::setWorkspaceReader);
    userProperties.ifPresent(repositorySystemSession::setUserProperties);

    if (isConcurrentLocalRepositoryEnabled) {
      repositorySystemSession.setConfigProperty(LOCK_FACTORY_KEY, FileLockNamedLockFactory.NAME);
      repositorySystemSession.setConfigProperty(LOCK_NAME_MAPPER_KEY, NameMappers.FILE_GAV_NAME);
    }

    return new DefaultMuleMavenRepositoryState(repositorySystemSession, repositorySystem,
                                               muleRepositorySystemManager.getArtifactResolver(),
                                               muleRepositorySystemManager.getVersionRangeResolver(),
                                               muleRepositorySystemManager.getRemoteRepositoryManager());
  }

  private static DefaultRepositorySystemSession newDefaultRepositorySystemSession(boolean noUpdateSnapshots,
                                                                                  boolean updateSnapshots,
                                                                                  Consumer<DefaultRepositorySystemSession> sessionConfigurator,
                                                                                  String globalChecksumPolicy,
                                                                                  Map<String, Object> serverConfigurations) {
    final DefaultRepositorySystemSession session = MavenRepositorySystemUtils.newSession();
    session.setResolutionErrorPolicy(new SimpleResolutionErrorPolicy(CACHE_NOT_FOUND));
    session.setCache(new DefaultRepositoryCache());
    if (noUpdateSnapshots) {
      session.setUpdatePolicy(UPDATE_POLICY_NEVER);
    } else if (updateSnapshots) {
      session.setUpdatePolicy(UPDATE_POLICY_ALWAYS);
    } else {
      session.setUpdatePolicy(null);
    }
    if (globalChecksumPolicy != null) {
      session.setChecksumPolicy(globalChecksumPolicy);
    }
    session.setTransferListener(new LoggingTransferListener());
    session.setConfigProperties(serverConfigurations);
    sessionConfigurator.accept(session);
    return session;
  }

  @Override
  public void close() throws Exception {
    muleRepositorySystemManager.close();
  }

  private static final class LoggingTransferListener implements TransferListener {

    @Override
    public void transferSucceeded(TransferEvent event) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Transfer {} for '{}' from {}", event.getType(), event.getResource().getResourceName(),
                     event.getResource().getRepositoryUrl());
      }
    }

    @Override
    public void transferStarted(TransferEvent event) throws TransferCancelledException {
      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace("Transfer {} for '{}'", event.getType(), event.getResource().getResourceName());
      }
    }

    @Override
    public void transferProgressed(TransferEvent event) throws TransferCancelledException {
      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace("Transfer {} for '{}' ({}/{})", event.getType(), event.getResource().getResourceName(),
                     byteCountToDisplaySize(event.getTransferredBytes()),
                     byteCountToDisplaySize(event.getResource().getContentLength()));
      }
    }

    @Override
    public void transferInitiated(TransferEvent event) throws TransferCancelledException {
      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace("Transfer {} for '{}'", event.getType(), event.getResource().getResourceName());
      }
    }

    @Override
    public void transferFailed(TransferEvent event) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Transfer {} for '{}' ({})", event.getType(), event.getResource(), event.getException());
      }
    }

    @Override
    public void transferCorrupted(TransferEvent event) throws TransferCancelledException {
      LOGGER.warn("Transfer {} for '{}' ({})", event.getType(), event.getResource(), event.getException());
    }

  }

  private static Optional<MuleMavenConcurrentLocalRepository> loadConcurrentLocalRepository() {
    for (MuleMavenConcurrentLocalRepository serviceRegister : load(MuleMavenConcurrentLocalRepository.class,
                                                                   MuleMavenRepositoryStateFactory.class.getClassLoader())) {
      return of(serviceRegister);
    }
    return empty();
  }

}
