/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.maven.client.internal.RemoteRepositoryModelResolver;

import java.util.List;

import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.impl.RemoteRepositoryManager;
import org.eclipse.aether.repository.RemoteRepository;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class RemoteRepositoryModelResolverTestCase {

  private static final String MULE_ID = "mule";
  private static final String MULE_PUBLIC_URL = "https://repository.mulesoft.org/nexus/content/repositories/public";

  private static final String OTHER_ID = "other-repository";
  private static final String OTHER_URL = "https://other.repository.com/relese";

  private static final String SPRING_ID = "spring-release";
  private static final String SPRING_RELEASE_URL = "https://repo.spring.io/list/release/";

  @Test
  public void remoteRepositoriesShouldGoFirst() {
    RemoteRepository otherRepository = new RemoteRepository.Builder(OTHER_ID, "default", OTHER_URL).build();
    RemoteRepository mulePublicRepository = new RemoteRepository.Builder(MULE_ID, "default", MULE_PUBLIC_URL).build();
    RemoteRepository springRelease = new RemoteRepository.Builder(SPRING_ID, "default", SPRING_RELEASE_URL).build();

    List<RemoteRepository> dominantRepositories = unmodifiableList(asList(otherRepository, mulePublicRepository));

    List<RemoteRepository> recessiveRepositories = singletonList(springRelease);

    List<RemoteRepository> aggregatedRepositories =
        unmodifiableList(concat(dominantRepositories.stream(), recessiveRepositories.stream()).collect(toList()));

    RemoteRepositoryManager remoteRepositoryManager = mock(RemoteRepositoryManager.class);
    RepositorySystemSession repositorySystemSession = mock(RepositorySystemSession.class);
    when(remoteRepositoryManager
        .aggregateRepositories(repositorySystemSession, dominantRepositories, recessiveRepositories, true))
            .thenReturn(aggregatedRepositories);

    RemoteRepositoryModelResolver remoteRepositoryModelResolver =
        new RemoteRepositoryModelResolver(remoteRepositoryManager, repositorySystemSession);
    List<RemoteRepository> resolved =
        remoteRepositoryModelResolver.resolveRepositories(dominantRepositories,
                                                          recessiveRepositories);

    assertThat(resolved.size(), equalTo(3));
    assertThat(resolved.get(0).getId(), is(OTHER_ID));
    assertThat(resolved.get(1).getId(), is(MULE_ID));
    assertThat(resolved.get(2).getId(), is(SPRING_ID));

    verify(remoteRepositoryManager).aggregateRepositories(repositorySystemSession, dominantRepositories, recessiveRepositories,
                                                          true);
  }

}
