/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.internal.util.VersionChecker.areCompatibleVersions;
import static org.mule.maven.client.internal.util.VersionChecker.getHighestVersion;
import static org.mule.maven.client.internal.util.VersionChecker.isHighestVersion;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class VersionCheckerTestCase {

  @Rule
  public ExpectedException expectedException = none();

  @Test
  public void wrongVersion() {
    expectedException.expect(IllegalArgumentException.class);
    expectedException
        .expectMessage(containsString("Unable to parse version a, version is not following semantic versioning"));
    areCompatibleVersions("a", "1.0.0");
  }

  @Test
  public void compatibleVersions() {
    assertThat(areCompatibleVersions("1", "1"), is(true));
    assertThat(areCompatibleVersions("1", "1.4"), is(true));
    assertThat(areCompatibleVersions("1.2", "1"), is(true));
    assertThat(areCompatibleVersions("1.2", "1.2"), is(true));
    assertThat(areCompatibleVersions("11.2.33", "11.2.55"), is(true));
    assertThat(areCompatibleVersions("11.2.33-rc.1", "11.2"), is(true));
  }

  @Test
  public void incompatibleVersions() {
    assertThat(areCompatibleVersions("1", "2"), is(false));
    assertThat(areCompatibleVersions("2.1", "1.2"), is(false));
    assertThat(areCompatibleVersions("1.2.4", "5"), is(false));
  }

  @Test
  public void highestVersion() {
    assertThat(getHighestVersion("1.1.0-beta", "1.1.0-beta.1"), is("1.1.0-beta.1"));
    assertThat(getHighestVersion("1.1.0-beta.1", "1.1.0-beta"), is("1.1.0-beta.1"));
    assertThat(getHighestVersion("1.1.0-beta.1", "1.1.0-beta.2"), is("1.1.0-beta.2"));
    assertThat(getHighestVersion("1.1.0-beta.1", "1.2.0-beta"), is("1.2.0-beta"));
    assertThat(getHighestVersion("1.1.0-FD.1", "1.1.0-FD.10"), is("1.1.0-FD.10"));
    assertThat(getHighestVersion("1.1.0-FD.1", "1.1.0-beta.2"), is("1.1.0-FD.1"));
    assertThat(getHighestVersion("1.1", "1.2"), is("1.2"));
    assertThat(getHighestVersion("2.1", "1.2"), is("2.1"));
    assertThat(getHighestVersion("3.1", "0.2"), is("3.1"));
    assertThat(getHighestVersion("1.1", "1.1.2"), is("1.1.2"));
    assertThat(getHighestVersion("1.1.2", "1.1.2.3"), is("1.1.2.3"));
  }

  @Test
  public void checkHighestVersion() {
    assertThat(isHighestVersion("1.1.0-beta", "1.1.0-beta.1"), is(false));
    assertThat(isHighestVersion("1.1.0-beta.1", "1.1.0-beta"), is(true));
    assertThat(isHighestVersion("1.1.0-beta.1", "1.1.0-beta.2"), is(false));
    assertThat(isHighestVersion("1.1.0-beta.1", "1.2.0-beta"), is(false));
    assertThat(isHighestVersion("1.1.0-FD.1", "1.1.0-FD.10"), is(false));
    assertThat(isHighestVersion("1.1.0-FD.1", "1.1.0-beta.2"), is(true));
    assertThat(isHighestVersion("1.1", "1.2"), is(false));
    assertThat(isHighestVersion("2.1", "1.2"), is(true));
    assertThat(isHighestVersion("3.1", "0.2"), is(true));
    assertThat(isHighestVersion("1.1", "1.1.2"), is(false));
    assertThat(isHighestVersion("1.1.2", "1.1.2.3"), is(false));
  }

}
