/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.lang.System.getProperty;

import org.mule.maven.client.api.EnvironmentConfiguration;

/**
 * Implementation of the {@link EnvironmentConfiguration} interface for Mule environments.
 *
 * <p>
 * This class provides configuration specific to Mule environments, including determining if the environment is FIPS (Federal
 * Information Processing Standards) compliant.
 * </p>
 */
public class MuleEnvironmentConfiguration implements EnvironmentConfiguration {

  /**
   * System property key for the Mule security model.
   */
  public static final String MULE_SECURITY_MODEL_PROPERTY = "mule.security.model";

  /**
   * Expected value for the FIPS 140-2 security model.
   */
  public static final String FIPS_140_2_SECURITY_MODEL = "fips140-2";

  private static EnvironmentConfiguration INSTANCE = new MuleEnvironmentConfiguration();
  private final boolean fipsEnvironment;

  /**
   * Private constructor to initialize the FIPS environment configuration.
   *
   * <p>
   * The FIPS compliance is determined based on the system property {@link #MULE_SECURITY_MODEL_PROPERTY}.
   * </p>
   */
  private MuleEnvironmentConfiguration() {
    this.fipsEnvironment = FIPS_140_2_SECURITY_MODEL.equals(getProperty(MULE_SECURITY_MODEL_PROPERTY));
  }

  /**
   * Retrieves the singleton instance of {@link MuleEnvironmentConfiguration}.
   *
   * @return the singleton instance of {@link MuleEnvironmentConfiguration}.
   */
  public static EnvironmentConfiguration getInstance() {
    return INSTANCE;
  }

  /**
   * {@inheritDoc}
   *
   * <p>
   * Checks if the current environment is FIPS compliant based on the system property {@link #MULE_SECURITY_MODEL_PROPERTY}.
   * </p>
   *
   * @return {@code true} if the environment is FIPS compliant, {@code false} otherwise.
   */
  @Override
  public boolean isFipsEnvironment() {
    return fipsEnvironment;
  }

  public static void resetConfiguration() {
    INSTANCE = new MuleEnvironmentConfiguration();
  }
}
