/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Collections.singletonList;

import static org.codehaus.plexus.PlexusConstants.SCANNING_INDEX;

import java.net.MalformedURLException;

import org.apache.maven.model.Dependency;
import org.apache.maven.model.Parent;
import org.apache.maven.model.resolution.ModelResolver;
import org.apache.maven.model.resolution.UnresolvableModelException;
import org.apache.maven.repository.internal.MavenRepositorySystemUtils;
import org.codehaus.plexus.ContainerConfiguration;
import org.codehaus.plexus.PlexusTestCase;
import org.codehaus.plexus.component.repository.exception.ComponentLookupException;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.impl.ArtifactResolver;
import org.eclipse.aether.impl.RemoteRepositoryManager;
import org.eclipse.aether.impl.VersionRangeResolver;
import org.eclipse.aether.repository.LocalRepository;
import org.eclipse.aether.repository.RemoteRepository;

/**
 * Just like {@link DefaultModelResolver} was copied from {@link org.apache.maven.repository.internal.DefaultModelResolver}, this
 * test case has been copied from {@link org.apache.maven.repository.internal.DefaultModelResolverTest}.
 */
public class DefaultModelResolverTestCase extends PlexusTestCase {

  protected RepositorySystem system;

  protected RepositorySystemSession session;

  @Override
  protected void customizeContainerConfiguration(ContainerConfiguration containerConfiguration) {
    super.customizeContainerConfiguration(containerConfiguration);
    containerConfiguration.setAutoWiring(true);
    containerConfiguration.setClassPathScanning(SCANNING_INDEX);
  }

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    system = lookup(RepositorySystem.class);
    session = newMavenRepositorySystemSession(system);
  }

  @Override
  protected void tearDown() throws Exception {
    session = null;
    system = null;
    super.tearDown();
  }

  public static RepositorySystemSession newMavenRepositorySystemSession(RepositorySystem system) {
    DefaultRepositorySystemSession session = MavenRepositorySystemUtils.newSession();

    LocalRepository localRepo = new LocalRepository("target/local-repo");
    session.setLocalRepositoryManager(system.newLocalRepositoryManager(session, localRepo));

    return session;
  }

  public static RemoteRepository newTestRepository() throws MalformedURLException {
    return new RemoteRepository.Builder(
                                        "repo",
                                        "default",
                                        getTestFile("target/test-classes/repo").toURI().toURL().toString())
                                            .build();
  }

  public void testResolveParentThrowsUnresolvableModelExceptionWhenNotFound() throws Exception {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("0");

    try {
      this.newModelResolver().resolveModel(parent);
      fail("Expected 'UnresolvableModelException' not thrown.");
    } catch (final UnresolvableModelException e) {
      assertNotNull(e.getMessage());
      assertTrue(e.getMessage().contains("Could not find artifact ut.simple:artifact:pom:0 in repo"));
    }
  }

  public void testResolveParentThrowsUnresolvableModelExceptionWhenNoMatchingVersionFound() throws Exception {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("[2.0,2.1)");

    try {
      this.newModelResolver().resolveModel(parent);
      fail("Expected 'UnresolvableModelException' not thrown.");
    } catch (final UnresolvableModelException e) {
      assertEquals("No versions matched the requested range '[2.0,2.1)'", e.getMessage());
    }
  }

  public void testResolveParentThrowsUnresolvableModelExceptionWhenUsingRangesWithoutUpperBound() throws Exception {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("[1.0,)");

    try {
      this.newModelResolver().resolveModel(parent);
      fail("Expected 'UnresolvableModelException' not thrown.");
    } catch (final UnresolvableModelException e) {
      assertEquals("The requested version range '[1.0,)' does not specify an upper bound", e.getMessage());
    }
  }

  public void testResolveParentSuccessfullyResolvesExistingParentWithoutRange() throws Exception {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("1.0");

    assertNotNull(this.newModelResolver().resolveModel(parent));
    assertEquals("1.0", parent.getVersion());
  }

  public void testResolveParentSuccessfullyResolvesExistingParentUsingHighestVersion() throws Exception {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("(,2.0)");

    assertNotNull(this.newModelResolver().resolveModel(parent));
    assertEquals("1.0", parent.getVersion());
  }

  public void testResolveDependencyThrowsUnresolvableModelExceptionWhenNotFound() throws Exception {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("0");

    try {
      this.newModelResolver().resolveModel(dependency);
      fail("Expected 'UnresolvableModelException' not thrown.");
    } catch (final UnresolvableModelException e) {
      assertNotNull(e.getMessage());
      assertTrue(e.getMessage().contains("Could not find artifact ut.simple:artifact:pom:0 in repo"));
    }
  }

  public void testResolveDependencyThrowsUnresolvableModelExceptionWhenNoMatchingVersionFound() throws Exception {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("[2.0,2.1)");

    try {
      this.newModelResolver().resolveModel(dependency);
      fail("Expected 'UnresolvableModelException' not thrown.");
    } catch (final UnresolvableModelException e) {
      assertEquals("No versions matched the requested dependency version range '[2.0,2.1)'", e.getMessage());
    }
  }

  public void testResolveDependencyThrowsUnresolvableModelExceptionWhenUsingRangesWithoutUpperBound()
      throws Exception {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("[1.0,)");

    try {
      this.newModelResolver().resolveModel(dependency);
      fail("Expected 'UnresolvableModelException' not thrown.");
    } catch (final UnresolvableModelException e) {
      assertEquals(
                   "The requested dependency version range '[1.0,)' does not specify an upper bound", e.getMessage());
    }
  }

  public void testResolveDependencySuccessfullyResolvesExistingDependencyWithoutRange() throws Exception {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("1.0");

    assertNotNull(this.newModelResolver().resolveModel(dependency));
    assertEquals("1.0", dependency.getVersion());
  }

  public void testResolveDependencySuccessfullyResolvesExistingDependencyUsingHighestVersion() throws Exception {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("(,2.0)");

    assertNotNull(this.newModelResolver().resolveModel(dependency));
    assertEquals("1.0", dependency.getVersion());
  }

  private ModelResolver newModelResolver() throws ComponentLookupException, MalformedURLException {
    return new DefaultModelResolver(
                                    this.session,
                                    lookup(ArtifactResolver.class),
                                    lookup(VersionRangeResolver.class),
                                    lookup(RemoteRepositoryManager.class),
                                    singletonList(newTestRepository()));
  }
}
