/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static java.lang.String.format;
import static java.lang.System.getProperties;
import static java.util.Collections.unmodifiableList;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.maven.model.Activation;
import org.apache.maven.model.ActivationOS;
import org.apache.maven.model.ActivationProperty;
import org.apache.maven.model.Profile;
import org.apache.maven.model.Repository;
import org.apache.maven.model.building.ModelProblemCollector;
import org.apache.maven.model.profile.DefaultProfileActivationContext;
import org.apache.maven.model.profile.DefaultProfileSelector;
import org.apache.maven.model.profile.ProfileActivationContext;
import org.apache.maven.model.profile.ProfileSelector;
import org.apache.maven.model.profile.activation.JdkVersionProfileActivator;
import org.apache.maven.model.profile.activation.OperatingSystemProfileActivator;
import org.apache.maven.model.profile.activation.PropertyProfileActivator;
import org.apache.maven.settings.Settings;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.repository.RepositoryPolicy;

public class MavenModelUtils {

  public static ProfileSelector getProfileSelector() {
    DefaultProfileSelector profileSelector = new DefaultProfileSelector();
    profileSelector.addProfileActivator(new JdkVersionProfileActivator());
    profileSelector.addProfileActivator(new OperatingSystemProfileActivator());
    profileSelector.addProfileActivator(new PropertyProfileActivator());
    return profileSelector;
  }

  public static RemoteRepository convertModelRepository(Repository modelRepository) {
    RemoteRepository.Builder builder = new RemoteRepository.Builder(modelRepository.getId(), "default", modelRepository.getUrl());
    if (modelRepository.getSnapshots() != null) {
      org.apache.maven.model.RepositoryPolicy snapshotPolicy = modelRepository.getSnapshots();
      builder.setSnapshotPolicy(new RepositoryPolicy(snapshotPolicy.isEnabled(), snapshotPolicy.getUpdatePolicy(),
                                                     snapshotPolicy.getChecksumPolicy()));
    }

    if (modelRepository.getReleases() != null) {
      org.apache.maven.model.RepositoryPolicy releasesPolicy = modelRepository.getReleases();
      builder.setReleasePolicy(new RepositoryPolicy(releasesPolicy.isEnabled(), releasesPolicy.getUpdatePolicy(),
                                                    releasesPolicy.getChecksumPolicy()));
    }

    return builder.build();
  }


  public static Profile toModelProfile(org.apache.maven.settings.Profile profile) {
    Profile p = new org.apache.maven.model.Profile();
    org.apache.maven.settings.Activation activation = profile.getActivation();
    if (activation != null) {
      Activation a = new Activation();
      a.setActiveByDefault(activation.isActiveByDefault());
      a.setJdk(activation.getJdk());
      a.setOs(toModelActivationOS(activation.getOs()));
      a.setProperty(toModelActivationProperty(activation.getProperty()));
      if (activation.getFile() != null) {
        throw new UnsupportedOperationException(
                                                format("Error while resolving dependencies, profile '%s' resolved from effective settings has defined a profile activation by file which is not supported",
                                                       profile.getId()));
      }

      p.setActivation(a);
    }
    p.setSource(profile.getSourceLevel());
    p.setId(profile.getId());
    return p;
  }

  private static ActivationOS toModelActivationOS(org.apache.maven.settings.ActivationOS activationOS) {
    if (activationOS == null) {
      return null;
    }
    ActivationOS activation = new ActivationOS();
    activation.setArch(activationOS.getArch());
    activation.setFamily(activationOS.getFamily());
    activation.setName(activationOS.getName());
    activation.setVersion(activationOS.getVersion());
    return activation;
  }

  private static ActivationProperty toModelActivationProperty(org.apache.maven.settings.ActivationProperty activationProperty) {
    if (activationProperty == null) {
      return null;
    }
    ActivationProperty activation = new ActivationProperty();
    activation.setName(activationProperty.getName());
    activation.setValue(activationProperty.getValue());
    return activation;
  }

  public static ProfileActivationContext getProfileActivationContext(Settings mavenSettings,
                                                                     List<String> activeProfiles,
                                                                     List<String> inactiveProfiles,
                                                                     Properties userProperties) {
    DefaultProfileActivationContext context = new DefaultProfileActivationContext();
    List<String> activeProfilesListBuilder = new ArrayList<>();
    if (activeProfiles != null) {
      activeProfilesListBuilder.addAll(activeProfiles);
    }
    activeProfilesListBuilder.addAll(mavenSettings.getActiveProfiles());
    context.setActiveProfileIds(unmodifiableList(activeProfilesListBuilder));

    if (inactiveProfiles != null) {
      context.setInactiveProfileIds(inactiveProfiles);
    }

    context.setSystemProperties(getProperties());
    if (userProperties != null) {
      context.setUserProperties(userProperties);
    }

    // TODO: support project properties (will be calculated from Model when reading pom.xml files)
    // context.setProjectProperties();
    return context;
  }

  public static ModelProblemCollector getModelProblemCollector() {
    return (req) -> {
    };
  }
}
