/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.maven.artifact.ArtifactUtils.key;
import static org.eclipse.aether.util.artifact.ArtifactIdUtils.toId;
import static org.mule.maven.client.internal.ArtifactRestorerTransformer.ORIGINAL_ARTIFACT_KEY;

import java.util.Optional;

import org.apache.maven.artifact.ArtifactUtils;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.graph.DependencyNode;
import org.eclipse.aether.util.artifact.ArtifactIdUtils;

/**
 * Mark all dependencies that belong to an specific type of parent to prevent the Maven conflict resolution from removing
 * replicates when they belong to different parents.
 *
 * @since 1.5.0
 */
public abstract class ParentContextDependencyGraphTransformer extends GraphCycleAvoidingTransformer {

  @Override
  protected void transformNode(DependencyNode node) {
    Optional<DependencyNode> parent = empty();
    if (isSelectableParent(node)) {
      parent = of(node);
    } else if (node.getData().containsKey(getAncestorKey())) {
      parent = of((DependencyNode) node.getData().get(getAncestorKey()));
    }
    for (DependencyNode child : node.getChildren()) {
      if (isChildNode(child)) {
        markChildNode(child, parent);
      }
    }
  }

  protected void markChildNode(DependencyNode childNode, Optional<DependencyNode> parent) {
    parent.ifPresent(parentNode -> {
      Artifact nodeArtifact = childNode.getArtifact();
      if (!childNode.getData().containsKey(ORIGINAL_ARTIFACT_KEY)) {
        childNode.setData(ORIGINAL_ARTIFACT_KEY, nodeArtifact);
        String newArtifactId = nodeArtifact.getArtifactId() + "<-" + toId(parentNode.getArtifact());
        Artifact newNodeArtifact = new DefaultArtifact(
                                                       nodeArtifact.getGroupId(),
                                                       newArtifactId,
                                                       nodeArtifact.getClassifier(),
                                                       nodeArtifact.getExtension(),
                                                       nodeArtifact.getVersion(),
                                                       nodeArtifact.getProperties(),
                                                       nodeArtifact.getFile());
        childNode.setArtifact(newNodeArtifact);
        childNode.setData(getAncestorKey(), parentNode);
      }
    });
  }

  /**
   * Return {@code true} if the node should be selected as parent and all children should be considered in it's context/
   *
   * @param dependencyNode the node to evaluate as parent.
   * @return true if the node should be considered as context parent
   */
  protected abstract boolean isSelectableParent(DependencyNode dependencyNode);

  /**
   * Returns if the node should be marked as a child node. That means that it belongs to a parent's context
   *
   * @param childNode the node to evaluate as child node
   * @return true if the node belongs to a parent context
   */
  protected abstract boolean isChildNode(DependencyNode childNode);

  /**
   * Key to store ancestors of this node that define it's context
   *
   * @return the key used to store node ancestors
   */
  protected abstract String getAncestorKey();

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    return null != obj && this.getClass().equals(obj.getClass());
  }

  @Override
  public int hashCode() {
    return this.getClass().hashCode();
  }

}
