/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static org.mule.maven.client.internal.util.MavenModelUtils.getModelProblemCollector;
import static org.mule.maven.client.internal.util.MavenModelUtils.getProfileActivationContext;
import static org.mule.maven.client.internal.util.MavenModelUtils.getProfileSelector;

import static java.lang.String.format;
import static java.lang.String.join;
import static java.lang.System.getProperty;
import static java.lang.System.getProperties;
import static java.util.Collections.emptyMap;
import static java.util.Collections.unmodifiableList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

import static org.eclipse.aether.repository.RepositoryPolicy.CHECKSUM_POLICY_WARN;
import static org.eclipse.aether.repository.RepositoryPolicy.UPDATE_POLICY_DAILY;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.BadMavenConfigurationException;
import org.mule.maven.client.api.EnvironmentConfiguration;
import org.mule.maven.client.api.model.Authentication;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.util.MavenModelUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;

import org.apache.maven.model.Model;
import org.apache.maven.model.building.DefaultModelProcessor;
import org.apache.maven.model.io.DefaultModelReader;
import org.apache.maven.model.locator.DefaultModelLocator;
import org.apache.maven.model.superpom.DefaultSuperPomProvider;
import org.apache.maven.settings.Activation;
import org.apache.maven.settings.Profile;
import org.apache.maven.settings.Repository;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.apache.maven.settings.building.DefaultSettingsBuilder;
import org.apache.maven.settings.building.DefaultSettingsBuilderFactory;
import org.apache.maven.settings.building.DefaultSettingsBuildingRequest;
import org.apache.maven.settings.building.SettingsBuildingException;
import org.apache.maven.settings.building.SettingsBuildingRequest;
import org.apache.maven.settings.building.SettingsBuildingResult;
import org.apache.maven.settings.crypto.DefaultSettingsDecryptionRequest;
import org.apache.maven.settings.crypto.SettingsDecryptionResult;
import org.codehaus.plexus.configuration.PlexusConfiguration;
import org.codehaus.plexus.configuration.xml.XmlPlexusConfiguration;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.eclipse.aether.repository.AuthenticationSelector;
import org.eclipse.aether.repository.MirrorSelector;
import org.eclipse.aether.repository.Proxy;
import org.eclipse.aether.repository.ProxySelector;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.repository.RepositoryPolicy;
import org.eclipse.aether.util.repository.AuthenticationBuilder;
import org.eclipse.aether.util.repository.DefaultAuthenticationSelector;
import org.eclipse.aether.util.repository.DefaultMirrorSelector;
import org.eclipse.aether.util.repository.DefaultProxySelector;
import org.slf4j.Logger;
import org.sonatype.plexus.components.cipher.DefaultPlexusCipher;
import org.sonatype.plexus.components.sec.dispatcher.DefaultSecDispatcher;

/**
 * Represents the context for resolving artifacts using Aether.
 */
public class MuleMavenResolutionContext {

  private static final Logger LOGGER = getLogger(MuleMavenResolutionContext.class);
  public static final String AETHER_CONNECTOR_WAGON_CONFIG = "aether.connector.wagon.config.";

  // A breaking change on the maven-resolver code generated the existence of two configuration properties
  public static final String AETHER_CHECKSUMS_ALGORITHMS_LEGACY_CONFIG = "aether.checksums.algorithms";
  public static final String AETHER_CHECKSUMS_ALGORITHMS_CURRENT_CONFIG = "aether.layout.maven2.checksumAlgorithms";

  public static final String FIPS_COMPLIANT_ALGORITHMS = "SHA-256,SHA-512";
  private final EnvironmentConfiguration environmentConfiguration;

  private List<RemoteRepository> remoteRepositories = new ArrayList<>();
  private File localRepositoryLocation;
  private Optional<AuthenticationSelector> authenticationSelector = empty();
  private Optional<ProxySelector> proxySelector = empty();
  private DefaultMirrorSelector mirrorSelector = new DefaultMirrorSelector();
  private Optional<Properties> userProperties = empty();

  private List<String> superPomModelRepositoriesId = new ArrayList<>();
  private Map<String, Object> serverConfigurations = new HashMap<>();

  public Map<String, Object> getServerConfigurations() {
    return serverConfigurations;
  }

  public MuleMavenResolutionContext(MavenConfiguration mavenConfiguration, EnvironmentConfiguration environmentConfiguration) {
    this.environmentConfiguration = environmentConfiguration;
    Model superPomModel = new DefaultSuperPomProvider().setModelProcessor(new DefaultModelProcessor()
        .setModelLocator(new DefaultModelLocator()).setModelReader(new DefaultModelReader())).getSuperModel("4.0.0");
    superPomModel.getRepositories().stream().forEach(repository -> superPomModelRepositoriesId.add(repository.getId()));
    resolveMavenConfiguration(mavenConfiguration);
  }

  public MuleMavenResolutionContext(MavenConfiguration mavenConfiguration) {
    this(mavenConfiguration, MuleEnvironmentConfiguration.getInstance());
  }

  public File getLocalRepositoryLocation() {
    return localRepositoryLocation;
  }

  public List<RemoteRepository> getRemoteRepositories() {
    return remoteRepositories;
  }

  private void resolveMavenConfiguration(MavenConfiguration mavenConfiguration) {
    localRepositoryLocation = mavenConfiguration.getLocalMavenRepositoryLocation();

    Optional<Settings> mavenSettingsOptional =
        getMavenSettings(mavenConfiguration.getSettingsSecurityLocation(),
                         mavenConfiguration.getUserSettingsLocation(),
                         mavenConfiguration.getGlobalSettingsLocation());
    mavenSettingsOptional.ifPresent(mavenSettings -> {
      createAuthenticatorSelector(mavenConfiguration, mavenSettings);
      createProxySelector(mavenSettings);
      copyMirrorSelectorsFrom(mavenSettings);
    });

    // W-12337231: Block http repositories. More info in https://maven.apache.org/docs/3.8.1/release-notes.html
    mirrorSelector.add("maven-default-http-blocker", "http://0.0.0.0/", "", false, true,
                       "external:http:*", "");

    remoteRepositories =
        collectRepositoriesFromConfiguration(mavenConfiguration, authenticationSelector, proxySelector, of(mirrorSelector));

    mavenSettingsOptional.ifPresent(mavenSettings -> {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Configuring optional Maven settings...");
      }
      addRepositoriesFromMavenSettings(mavenConfiguration, mavenSettings, authenticationSelector, proxySelector,
                                       of(mirrorSelector));

      String localRepository = mavenSettings.getLocalRepository();
      if (localRepositoryLocation == null && localRepository != null) {
        File localRepositoryFile = new File(localRepository);
        if (!localRepositoryFile.isDirectory() || !localRepositoryFile.exists()) {
          throw new BadMavenConfigurationException(format(
                                                          "Local repository location %s resolved from maven settings file does not exists or is not a directory",
                                                          localRepository));
        }
        localRepositoryLocation = localRepositoryFile;
      }
    });

    userProperties = mavenConfiguration.getUserProperties();

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Using {}", toString());
    }
  }

  private List<RemoteRepository> collectRepositoriesFromConfiguration(MavenConfiguration mavenConfiguration,
                                                                      Optional<AuthenticationSelector> authenticationSelectorOptional,
                                                                      Optional<ProxySelector> proxySelectorOptional,
                                                                      Optional<MirrorSelector> mirrorSelector) {
    List<RemoteRepository> collectedRepositories = new ArrayList<>();
    List<org.mule.maven.client.api.model.RemoteRepository> clientConfiguredRepositories =
        mavenConfiguration.getMavenRemoteRepositories();
    clientConfiguredRepositories.stream().forEachOrdered(explicitRemoteRepository -> {
      RemoteRepository.Builder aetherRepoBuilder =
          new RemoteRepository.Builder(explicitRemoteRepository.getId(), "default", explicitRemoteRepository.getUrl().toString());
      if (explicitRemoteRepository.getAuthentication().isPresent()) {
        Authentication authentication = explicitRemoteRepository.getAuthentication().get();
        aetherRepoBuilder.setAuthentication(new AuthenticationBuilder().addUsername(authentication.getUsername())
            .addPassword(authentication.getPassword()).build());
      }
      aetherRepoBuilder = processRemoteRepository(authenticationSelectorOptional, proxySelectorOptional,
                                                  mirrorSelector, aetherRepoBuilder);
      if (explicitRemoteRepository.getReleasePolicy().isPresent()) {
        org.mule.maven.client.api.model.RepositoryPolicy releasePolicy = explicitRemoteRepository.getReleasePolicy().get();
        aetherRepoBuilder.setReleasePolicy(new RepositoryPolicy(releasePolicy.isEnabled(), releasePolicy.getUpdatePolicy(),
                                                                releasePolicy.getChecksumPolicy()));
      }

      if (explicitRemoteRepository.getSnapshotPolicy().isPresent()) {
        org.mule.maven.client.api.model.RepositoryPolicy snapshotPolicy = explicitRemoteRepository.getSnapshotPolicy().get();
        aetherRepoBuilder.setSnapshotPolicy(new RepositoryPolicy(snapshotPolicy.isEnabled(), snapshotPolicy.getUpdatePolicy(),
                                                                 snapshotPolicy.getChecksumPolicy()));
      }

      collectedRepositories.add(aetherRepoBuilder.build());
    });

    return unmodifiableList(collectedRepositories);
  }

  private RemoteRepository.Builder selectAuthenticator(RemoteRepository.Builder aetherRepoBuilder,
                                                       Optional<AuthenticationSelector> authenticationSelectorOptional) {
    if (authenticationSelectorOptional.isPresent()) {
      RemoteRepository prototypeRepository = aetherRepoBuilder.build();
      org.eclipse.aether.repository.Authentication authentication =
          authenticationSelectorOptional.get().getAuthentication(prototypeRepository);
      if (authentication != null) {
        aetherRepoBuilder = new RemoteRepository.Builder(prototypeRepository);
        aetherRepoBuilder.setAuthentication(authentication);
      }
    }
    return aetherRepoBuilder;
  }

  private void addRepositoriesFromMavenSettings(MavenConfiguration mavenConfiguration, Settings mavenSettings,
                                                Optional<AuthenticationSelector> authenticationSelectorOptional,
                                                Optional<ProxySelector> proxySelectorOptional,
                                                Optional<MirrorSelector> mirrorSelectorOptional) {

    List<org.apache.maven.model.Profile> profiles =
        mavenSettings.getProfiles().stream().map(MavenModelUtils::toModelProfile).collect(toList());

    List<String> activeProfiles =
        getProfileSelector()
            .getActiveProfiles(profiles,
                               getProfileActivationContext(mavenSettings, mavenConfiguration.getActiveProfiles().orElse(null),
                                                           mavenConfiguration.getInactiveProfiles().orElse(null),
                                                           mavenConfiguration.getUserProperties().orElse(null)),
                               getModelProblemCollector())
            .stream().map(profile -> profile.getId()).collect(toList());

    List<RemoteRepository> remoteRepositoriesFromSettings = new ArrayList<>();
    for (String profileName : mavenSettings.getProfilesAsMap().keySet()) {
      Profile profile = mavenSettings.getProfilesAsMap().get(profileName);
      if (mavenSettings.getActiveProfiles().contains(profileName)
          || ofNullable(profile.getActivation()).map(Activation::isActiveByDefault).orElse(true)
          || activeProfiles.contains(profileName)) {
        List<Repository> repositories = profile.getRepositories();
        for (Repository repo : repositories) {
          RemoteRepository.Builder remoteRepo = new RemoteRepository.Builder(repo.getId(), "default", repo.getUrl());
          remoteRepo = processRemoteRepository(authenticationSelectorOptional, proxySelectorOptional,
                                               mirrorSelectorOptional, remoteRepo);
          if (repo.getSnapshots() != null) {
            remoteRepo
                .setSnapshotPolicy(new RepositoryPolicy(repo.getSnapshots().isEnabled(),
                                                        toUpdatePolicy(repo.getSnapshots().getUpdatePolicy()),
                                                        toChecksumPolicy(repo.getSnapshots().getChecksumPolicy())));

          }
          if (repo.getReleases() != null) {
            remoteRepo
                .setReleasePolicy(new RepositoryPolicy(repo.getReleases().isEnabled(),
                                                       toUpdatePolicy(repo.getReleases().getUpdatePolicy()),
                                                       toChecksumPolicy(repo.getReleases().getChecksumPolicy())));
          }
          remoteRepositoriesFromSettings.add(remoteRepo.build());
        }
      }
    }
    remoteRepositories =
        new RemoteRepositoriesMerger().merge(remoteRepositories, unmodifiableList(remoteRepositoriesFromSettings));
  }

  private String toChecksumPolicy(String releasesPolicy) {
    return releasesPolicy != null ? releasesPolicy : CHECKSUM_POLICY_WARN;
  }

  private String toUpdatePolicy(String snapshotsPolicy) {
    return snapshotsPolicy != null ? snapshotsPolicy : UPDATE_POLICY_DAILY;
  }

  private RemoteRepository.Builder processRemoteRepository(Optional<AuthenticationSelector> authenticationSelectorOptional,
                                                           Optional<ProxySelector> proxySelectorOptional,
                                                           Optional<MirrorSelector> mirrorSelectorOptional,
                                                           RemoteRepository.Builder aetherRepoBuilder) {
    aetherRepoBuilder = selectAuthenticator(aetherRepoBuilder, authenticationSelectorOptional);

    if (proxySelectorOptional.isPresent()) {
      RemoteRepository prototypeRepository = aetherRepoBuilder.build();
      Proxy proxy = proxySelectorOptional.get().getProxy(prototypeRepository);
      if (proxy != null) {
        aetherRepoBuilder = new RemoteRepository.Builder(prototypeRepository);
        aetherRepoBuilder.setProxy(proxy);
      }
    }
    if (mirrorSelectorOptional.isPresent()) {
      RemoteRepository prototypeRepository = aetherRepoBuilder.build();
      RemoteRepository mirror = mirrorSelectorOptional.get().getMirror(prototypeRepository);
      if (mirror != null) {
        aetherRepoBuilder = new RemoteRepository.Builder(mirror);
        aetherRepoBuilder = selectAuthenticator(aetherRepoBuilder, authenticationSelectorOptional);
      }
    }
    return aetherRepoBuilder;
  }

  private void configureServer(Server server) {
    if (environmentConfiguration.isFipsEnvironment()) {
      serverConfigurations.put(AETHER_CHECKSUMS_ALGORITHMS_LEGACY_CONFIG, FIPS_COMPLIANT_ALGORITHMS);
      serverConfigurations.put(AETHER_CHECKSUMS_ALGORITHMS_CURRENT_CONFIG, FIPS_COMPLIANT_ALGORITHMS);
    }

    Object configuration = server.getConfiguration();
    PlexusConfiguration config = null;
    if (configuration instanceof PlexusConfiguration) {
      config = (PlexusConfiguration) configuration;
    } else if (configuration instanceof Xpp3Dom) {
      config = new XmlPlexusConfiguration((Xpp3Dom) configuration);
    } else if (configuration == null) {
      return;
    } else {
      LOGGER.warn("Unexpected configuration type: " + configuration.getClass().getName());
    }
    serverConfigurations.put(AETHER_CONNECTOR_WAGON_CONFIG + server.getId(), config);
  }

  private void createAuthenticatorSelector(MavenConfiguration mavenConfiguration, Settings mavenSettings) {
    DefaultAuthenticationSelector defaultAuthenticationSelector = new DefaultAuthenticationSelector();
    mavenSettings.getServers().stream()
        .forEach(server -> {
          defaultAuthenticationSelector.add(server.getId(), new AuthenticationBuilder()
              .addUsername(server.getUsername())
              .addPassword(server.getPassword())
              .addPrivateKey(server.getPrivateKey(), server.getPassphrase())
              .build());
          configureServer(server);
        });

    // In addition to the serverIds defined on settings file we do append the ones from remote repositories declared
    // through the MavenConfiguration API
    mavenConfiguration.getMavenRemoteRepositories().stream()
        .filter(remoteRepository -> remoteRepository.getAuthentication().isPresent())
        .forEach(remoteRepository -> defaultAuthenticationSelector.add(remoteRepository.getId(), new AuthenticationBuilder()
            .addUsername(remoteRepository.getAuthentication().get().getUsername())
            .addPassword(remoteRepository.getAuthentication().get().getPassword())
            .build()));

    this.authenticationSelector = of(defaultAuthenticationSelector);
  }

  private void createProxySelector(Settings mavenSettings) {
    DefaultProxySelector defaultProxySelector = new DefaultProxySelector();
    mavenSettings.getProxies().stream().filter(proxy -> proxy.isActive())
        .forEach(proxy -> defaultProxySelector.add(
                                                   new Proxy(proxy.getProtocol(), proxy.getHost(), proxy.getPort(),
                                                             new AuthenticationBuilder()
                                                                 .addUsername(proxy.getUsername())
                                                                 .addPassword(proxy.getPassword())
                                                                 .build()),
                                                   proxy.getNonProxyHosts()));
    this.proxySelector = of(defaultProxySelector);
  }

  private void copyMirrorSelectorsFrom(Settings mavenSettings) {
    mavenSettings.getMirrors().stream()
        .forEachOrdered(mirror ->
        // Repository manager flag is set to false
        // Maven does not support specifying it in the settings.xml
        mirrorSelector.add(mirror.getId(), mirror.getUrl(), mirror.getLayout(), false, mirror.getMirrorOf(),
                           mirror.getMirrorOfLayouts()));
  }

  private Optional<Settings> getMavenSettings(final Optional<File> secureSettingsFile, Optional<File> userSettingsFile,
                                              Optional<File> globalSettingsFile) {
    if (!userSettingsFile.isPresent() && !globalSettingsFile.isPresent()) {
      return empty();
    }
    try {
      SettingsBuildingRequest settingsBuildingRequest = new DefaultSettingsBuildingRequest();
      settingsBuildingRequest.setSystemProperties(getProperties());
      userSettingsFile.ifPresent(settingsBuildingRequest::setUserSettingsFile);
      globalSettingsFile.ifPresent(settingsBuildingRequest::setGlobalSettingsFile);

      DefaultSettingsBuilderFactory mvnSettingBuilderFactory = new DefaultSettingsBuilderFactory();
      DefaultSettingsBuilder settingsBuilder = mvnSettingBuilderFactory.newInstance();
      SettingsBuildingResult settingsBuildingResult = settingsBuilder.build(settingsBuildingRequest);

      Settings settings = settingsBuildingResult.getEffectiveSettings();

      secureSettingsFile.ifPresent(secureSettings -> {
        DefaultSettingsDecrypter settingsDecrypter =
            MavenClientSettingsDecryptorFactory.newInstance(secureSettings.getAbsolutePath());

        SettingsDecryptionResult result = settingsDecrypter.decrypt(new DefaultSettingsDecryptionRequest(settings));
        settings.setServers(result.getServers());
        settings.setProxies(result.getProxies());
      });

      return of(settings);
    } catch (SettingsBuildingException e) {
      throw new RuntimeException(e);
    }
  }

  public List<RemoteRepository> getRemoteRepositoriesWithoutSuperPomModelRepositories() {
    return getRemoteRepositories().stream()
        .filter(remoteRepository -> !superPomModelRepositoriesId.contains(remoteRepository.getId())).collect(toList());
  }

  static class MavenClientSecDispatcher extends DefaultSecDispatcher {

    public MavenClientSecDispatcher(String configurationFilePath) {
      super(new DefaultPlexusCipher(), emptyMap(), configurationFilePath);
    }
  }

  static class MavenClientSettingsDecryptorFactory {

    public static DefaultSettingsDecrypter newInstance(String secureSettingsFile) {
      MavenClientSecDispatcher secDispatcher = new MavenClientSecDispatcher(secureSettingsFile);

      DefaultSettingsDecrypter decrypter = new DefaultSettingsDecrypter(secDispatcher);
      return decrypter;
    }
  }

  public Optional<AuthenticationSelector> getAuthenticatorSelector() {
    return this.authenticationSelector;
  }

  public Optional<ProxySelector> getProxySelector() {
    return this.proxySelector;
  }

  public Optional<MirrorSelector> getMirrorSelector() {
    return of(this.mirrorSelector);
  }

  public Optional<Properties> getUserProperties() {
    return this.userProperties;
  }

  @Override
  public String toString() {
    return "AetherResolutionContext{" +
        "remoteRepositories=" + repositoriesToString() +
        ", localMavenRepositoryLocation=" + localRepositoryLocation.getAbsolutePath() +
        '}';
  }

  private String repositoriesToString() {
    return join(",\n", remoteRepositories.stream().map(RemoteRepository::toString).collect(toList()));
  }

}
