/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;

import org.mule.maven.client.internal.MuleMavenClient;
import org.mule.maven.pom.parser.api.model.MavenPomModel;

import java.net.URL;

import java.io.File;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

public class PomModelTestCase {

  private static final String APP_SUFFIX = "-1.0.0-mule-application";
  private static final String APP_NAME = "test-app";
  private static final String EXPLODED_APP_FILE = APP_NAME + APP_SUFFIX;
  private static final String COMPRESSED_APP_FILE = EXPLODED_APP_FILE + ".jar";

  private static final String APP_WITH_PARENT_NAME = APP_NAME + "-with-parent-pom";
  private static final String EXPLODED_APP_WITH_PARENT_FILE = APP_WITH_PARENT_NAME + APP_SUFFIX;
  private static final String COMPRESSED_APP_WITH_PARENT_FILE = EXPLODED_APP_WITH_PARENT_FILE + ".jar";

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Rule
  public TemporaryFolder repositoryFolder = new TemporaryFolder();

  @Rule
  public ExpectedException expectedException = none();

  private MuleMavenClient mavenClient;

  @Before
  public void setUp() throws Exception {
    mavenClient = new MuleMavenClient(
                                      newMavenConfigurationBuilder()
                                          .localMavenRepositoryLocation(repositoryFolder.getRoot())
                                          .remoteRepository(newRemoteRepositoryBuilder()
                                              .id("maven-central")
                                              .url(new URL("https://repo.maven.apache.org/maven2/")).build())
                                          .build());

  }

  @Test
  public void getPomModelFromJarWithNoTemporaryFolderResolvedInMemory() throws Exception {
    File jarFile = new File(this.getClass().getClassLoader().getResource(COMPRESSED_APP_FILE).toURI());
    MavenPomModel model = mavenClient.getEffectiveModel(jarFile, empty());
    assertThat(model.getArtifactId(), is(APP_NAME));
  }

  @Test
  public void getPomModelFromJarWithTemporaryFolderResolvedInDisk() throws Exception {
    File jarFile = new File(this.getClass().getClassLoader().getResource(COMPRESSED_APP_FILE).toURI());
    File folder = temporaryFolder.newFolder();
    MavenPomModel model = mavenClient.getEffectiveModel(jarFile, of(folder));
    assertThat(model.getArtifactId(), is(APP_NAME));
    assertThat(FileUtils.readFileToString(folder.listFiles()[0]), containsString("<artifactId>" + APP_NAME + "</artifactId>"));
  }

  @Test
  public void getPomModelFromExplodedApp() throws Exception {
    File app = new File(this.getClass().getClassLoader().getResource(EXPLODED_APP_FILE).toURI());
    MavenPomModel model = mavenClient.getEffectiveModel(app, empty());
    assertThat(model.getArtifactId(), is(APP_NAME));
  }

  @Test
  public void getPomModelFromExplodedAppWithParentPom() throws Exception {
    File app = new File(this.getClass().getClassLoader().getResource(EXPLODED_APP_WITH_PARENT_FILE).toURI());
    MavenPomModel model = mavenClient.getEffectiveModel(app, empty());
    assertThat(model.getArtifactId(), is(APP_WITH_PARENT_NAME));
  }

  @Test
  public void getPomModelFromCompressedAppWithParentPom() throws Exception {
    File jarFile = new File(this.getClass().getClassLoader().getResource(COMPRESSED_APP_WITH_PARENT_FILE).toURI());
    expectedException.expect(RuntimeException.class);
    expectedException
        .expectMessage("problem was encountered while building the effective model for test:" + APP_WITH_PARENT_NAME + ":1.0.0");
    mavenClient.getEffectiveModel(jarFile, empty());
  }

  @Test
  public void getPomModelFromCompressedAppWithParentPomAndTemporaryFolder() throws Exception {
    File jarFile = new File(this.getClass().getClassLoader().getResource(COMPRESSED_APP_WITH_PARENT_FILE).toURI());
    File folder = temporaryFolder.newFolder();
    expectedException.expect(RuntimeException.class);
    expectedException
        .expectMessage("problem was encountered while building the effective model for test:" + APP_WITH_PARENT_NAME + ":1.0.0");
    mavenClient.getEffectiveModel(jarFile, of(folder));
  }


}
