/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static org.mule.maven.client.api.model.RepositoryPolicy.CHECKSUM_POLICY_FAIL;
import static org.mule.maven.client.api.model.RepositoryPolicy.CHECKSUM_POLICY_WARN;
import static org.mule.maven.client.internal.MavenCommandLineParser.parseMavenArguments;
import static org.mule.maven.client.internal.MavenEnvironmentVariables.MAVEN_CMD_LINE_ARGS;

import static java.io.File.separator;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.SettingsSupplierFactory;
import org.mule.maven.client.api.model.MavenConfiguration.MavenConfigurationBuilder;
import org.mule.maven.client.internal.MavenCommandLineParser.MavenArguments;

import java.io.File;
import java.util.Optional;

import org.slf4j.Logger;

/**
 * Provides a set of suppliers to be used to resolve maven settings files.
 *
 * @since 1.0
 */
public class DefaultSettingsSupplierFactory implements SettingsSupplierFactory {

  private static final Logger LOGGER = getLogger(DefaultSettingsSupplierFactory.class);

  private static final String SETTINGS_XML = "settings.xml";
  private static final String SETTINGS_SECURITY_XML = "settings-security.xml";
  private static final String MAVEN_GLOBAL_SETTINGS_PATH = "conf" + separator + SETTINGS_XML;

  public static final String MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY = "settings.security";

  private final MavenEnvironmentVariables mavenEnvironmentVariables;
  private final MavenArguments mavenArguments;

  public DefaultSettingsSupplierFactory(MavenEnvironmentVariables mavenEnvironmentVariables) {
    requireNonNull(mavenEnvironmentVariables, "mavenEnvironmentVariables cannot be null");

    this.mavenArguments = parseMavenArguments(mavenEnvironmentVariables);
    this.mavenEnvironmentVariables = mavenEnvironmentVariables;
  }

  @Override
  public Optional<File> environmentGlobalSettingsSupplier() {
    if (mavenArguments.getGlobalSettings().isPresent()) {
      final File file = mavenArguments.getGlobalSettings().get();
      if (file.exists()) {
        logGlobalSettings(file, MAVEN_CMD_LINE_ARGS);
        return of(mavenArguments.getGlobalSettings().get());
      }
    }

    File globalSettingsFile = mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(GLOBAL_SETTINGS_SYSTEM_PROPERTY);
    if (globalSettingsFile != null) {
      logGlobalSettings(globalSettingsFile, GLOBAL_SETTINGS_SYSTEM_PROPERTY);
      return of(globalSettingsFile);
    }

    String mavenHome = mavenEnvironmentVariables.getM2HomeEnv();
    mavenHome = mavenHome != null ? mavenHome : mavenEnvironmentVariables.getMavenHomeEnv();
    mavenHome = mavenHome != null ? mavenHome : mavenEnvironmentVariables.getMavenHomeProperty();
    if (mavenHome != null) {
      File globalSettings = new File(mavenHome, MAVEN_GLOBAL_SETTINGS_PATH);
      if (globalSettings.exists()) {
        logGlobalSettings(globalSettings, mavenHome);
        return of(globalSettings);
      }
    }
    return empty();
  }

  @Override
  public Optional<File> environmentUserSettingsSupplier() {
    if (mavenArguments.getSettings().isPresent()) {
      final File file = mavenArguments.getSettings().get();
      if (file.exists()) {
        logUserSettings(file, MAVEN_CMD_LINE_ARGS);
        return of(file);
      }
    }

    File userSettingsFile = mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(USER_SETTINGS_SYSTEM_PROPERTY);
    if (userSettingsFile != null) {
      logUserSettings(userSettingsFile, USER_SETTINGS_SYSTEM_PROPERTY);
      return of(userSettingsFile);
    }

    File userM2Folder = new File(mavenEnvironmentVariables.getUserHome(), ".m2");
    if (userM2Folder.exists()) {
      final File file = new File(userM2Folder, SETTINGS_XML);
      if (file.exists()) {
        logUserSettings(file, userM2Folder.getAbsolutePath());
        return of(file);
      }
    }
    return empty();
  }

  @Override
  public Optional<File> environmentSettingsSecuritySupplier() {
    File settingsSecurityFile = mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY);
    if (settingsSecurityFile != null) {
      logSettingsSecurity(settingsSecurityFile, MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY);
      return of(settingsSecurityFile);
    }

    File userM2Folder = new File(mavenEnvironmentVariables.getUserHome(), ".m2");
    if (userM2Folder.exists()) {
      final File file = new File(userM2Folder, SETTINGS_SECURITY_XML);
      if (file.exists()) {
        logUserSettings(file, userM2Folder.getAbsolutePath());
        return of(file);
      }
    }
    return empty();
  }

  @Override
  public MavenConfigurationBuilder addToMavenConfig(MavenConfigurationBuilder mavenConfigurationBuilder) {
    final Optional<File> globalSettings = environmentGlobalSettingsSupplier();
    final Optional<File> userSettings = environmentUserSettingsSupplier();
    final Optional<File> settingsSecurity = environmentSettingsSecuritySupplier();

    if (globalSettings.isPresent()) {
      mavenConfigurationBuilder.globalSettingsLocation(globalSettings.get());
    } else {
      LOGGER
          .info("Maven global settings couldn't be found, M2_HOME environment variable has to be set in order to use global settings (if needed)");
    }

    if (userSettings.isPresent()) {
      mavenConfigurationBuilder.userSettingsLocation(userSettings.get());
    } else {
      LOGGER.info("Maven user settings couldn't be found, this could cause a wrong resolution for dependencies");
    }

    if (settingsSecurity.isPresent()) {
      mavenConfigurationBuilder.settingsSecurityLocation(settingsSecurity.get());
    } else {
      LOGGER.info("Maven settings security couldn't be found");
    }

    if (mavenArguments.isStrictChecksums()) {
      mavenConfigurationBuilder.globalChecksumPolicy(CHECKSUM_POLICY_FAIL);
    }

    if (mavenArguments.isLaxChecksums()) {
      mavenConfigurationBuilder.globalChecksumPolicy(CHECKSUM_POLICY_WARN);
    }

    return null;
  }

  private void logSettingsSecurity(File settingsSecurity, String location) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Resolved Maven settings security '{}' from '{}'", settingsSecurity.getAbsolutePath(), location);
    }
  }

  private void logUserSettings(File userSettings, String location) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Resolved Maven user settings '{}' from '{}'", userSettings.getAbsolutePath(), location);
    }
  }

  private void logGlobalSettings(File globalSettings, String location) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Resolved Maven global settings '{}' from '{}'", globalSettings.getAbsolutePath(), location);
    }
  }

}
