/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.nio.file.Files.createTempDirectory;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

import java.io.File;
import java.io.IOException;
import java.util.concurrent.Callable;
import java.util.function.Supplier;

import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.ExternalResource;

public class DefaultLocalRepositorySupplierFactoryTestCase {

  private static final File tmpDir1;
  private static final File tmpDir2;
  private static final File repositoryInsideTmpDir2;

  static {
    try {
      tmpDir1 = createTempDirectory("fixedFolder").toFile();
      tmpDir2 = createTempDirectory("fixedFolder").toFile();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
    File m2 = new File(tmpDir2, ".m2");
    m2.mkdir();
    repositoryInsideTmpDir2 = new File(m2, "repository");
    repositoryInsideTmpDir2.mkdir();
  }

  @ClassRule
  public static ExternalResource USER_HOME_PROP = new SystemProperty("user.home", tmpDir2.getAbsolutePath());

  private final DefaultLocalRepositorySupplierFactory factory = new DefaultLocalRepositorySupplierFactory();

  @Test
  public void fixedFolderSupplier() {
    Supplier<File> fixedSupplier1 = factory.fixedFolderSupplier(tmpDir1);
    Supplier<File> fixedSupplier2 = factory.fixedFolderSupplier(tmpDir2);
    assertThat(fixedSupplier1.get(), is(tmpDir1));
    assertThat(fixedSupplier2.get(), is(tmpDir2));
  }

  @Test
  public void environmentMavenRepositorySupplierDefault() throws Throwable {
    callWithProperty("localRepository", null, () -> {
      Supplier<File> envSupplier = factory.environmentMavenRepositorySupplier();
      assertThat(envSupplier.get(), is(repositoryInsideTmpDir2));
      return null;
    });
  }

  @Test
  public void environmentMavenRepositorySupplierFromSystemProperty() throws Throwable {
    callWithProperty("localRepository", tmpDir1.getAbsolutePath(), () -> {
      Supplier<File> envSupplier = factory.environmentMavenRepositorySupplier();
      assertThat(envSupplier.get(), is(tmpDir1));
      return null;
    });
  }

  @Test
  public void composeSuppliers() {
    Supplier<File> composedSupplier = factory.composeSuppliers(() -> null, factory.fixedFolderSupplier(tmpDir1));
    assertThat(composedSupplier.get(), is(tmpDir1));
  }

  private static class SystemProperty extends ExternalResource {

    private final String name;
    private final String value;
    private String oldValue;

    private SystemProperty(String name, String value) {
      this.name = name;
      this.value = value;
    }

    @Override
    protected void before() {
      if (value == null) {
        System.clearProperty(name);
      } else {
        oldValue = System.setProperty(name, value);
      }
    }

    @Override
    protected void after() {
      if (oldValue == null) {
        System.clearProperty(name);
      } else {
        System.setProperty(name, oldValue);
      }
    }
  }

  private static <V> V callWithProperty(String name, String value, Callable<V> callable) throws Throwable {
    SystemProperty property = new SystemProperty(name, value);
    property.before();
    try {
      return callable.call();
    } catch (Exception e) {
      throw new Exception("Callable threw an exception during its execution.", e);
    } finally {
      property.after();
    }
  }
}
