/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.stream.Collectors.toList;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;

import org.hamcrest.CoreMatchers;
import org.junit.Test;

import java.util.List;

import com.google.common.collect.ImmutableList;

public abstract class AbstractOnlineMavenClientTestCase extends AbstractMavenClientTestCase {

  private static final String CONNECTORS_VERSION = "1.0.0-SNAPSHOT";

  @Override
  protected void beforeTest() throws Exception {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration());
  }

  protected abstract MavenConfiguration createMavenConfiguration() throws Exception;

  @Test
  public void resolveExistentBundleDescriptor() {
    String groupId = "org.mule";
    String artifactId = "mule-core";
    String version = "3.3.0";
    String type = "jar";
    BundleDependency coreBundle = mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder().setGroupId(groupId)
        .setArtifactId(artifactId)
        .setVersion(version)
        .setType(type).build());
    assertThat(coreBundle.getDescriptor().getGroupId(), is(groupId));
    assertThat(coreBundle.getDescriptor().getArtifactId(), is(artifactId));
    assertThat(coreBundle.getDescriptor().getVersion(), is(version));
    assertThat(coreBundle.getDescriptor().getType(), is(type));
    assertThat(coreBundle.getDescriptor().getClassifier().isPresent(), is(false));
  }

  @Test
  public void resolveNotExistentBundleDescriptor() {
    String groupId = "org.mule";
    String artifactId = "mule-core-fake";
    String version = "3.3.0";
    String type = "jar";
    expectedException.expect(BundleDependenciesResolutionException.class);
    mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder().setGroupId(groupId)
        .setArtifactId(artifactId)
        .setVersion(version)
        .setType(type).build());
  }

  @Test
  public void resolvePluginDependencies() {
    BundleDescriptor emailBundleDescriptor = new BundleDescriptor.Builder().setGroupId("org.mule.connectors")
        .setArtifactId("mule-email-connector")
        .setVersion(CONNECTORS_VERSION)
        .setClassifier("mule-plugin").build();
    BundleDescriptor httpBundleDescriptor = new BundleDescriptor.Builder().setGroupId("org.mule.connectors")
        .setArtifactId("mule-http-connector")
        .setVersion(CONNECTORS_VERSION)
        .setClassifier("mule-plugin").build();
    BundleDescriptor socketsBundleDescriptor = new BundleDescriptor.Builder().setGroupId("org.mule.connectors")
        .setArtifactId("mule-sockets-connector")
        .setVersion(CONNECTORS_VERSION)
        .setClassifier("mule-plugin").build();
    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(ImmutableList.<BundleDescriptor>builder()
            .add(httpBundleDescriptor)
            .add(emailBundleDescriptor)
            .build());
    assertThat(bundleDependencies.stream().map(BundleDependency::getDescriptor)
        .map(bundleDescriptor -> new BundleDescriptor.Builder()
            .setGroupId(bundleDescriptor.getGroupId())
            .setArtifactId(bundleDescriptor.getArtifactId())
            .setVersion(CONNECTORS_VERSION)
            .setType(bundleDescriptor.getType())
            .setClassifier(bundleDescriptor.getClassifier().orElse(null))
            .setBaseVersion(bundleDescriptor.getBaseVersion())
            .build())
        .collect(toList()),
               CoreMatchers.hasItems(httpBundleDescriptor, socketsBundleDescriptor, emailBundleDescriptor));
  }


}
