/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.lang.String.format;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mule.maven.client.internal.DefaultSettingsSupplierFactory.GLOBAL_SETTINGS_SYSTEM_PROPERTY;
import static org.mule.maven.client.internal.DefaultSettingsSupplierFactory.USER_SETTINGS_SYSTEM_PROPERTY;
import static org.mule.maven.client.test.AllureConstants.MavenClient.MAVEN_CLIENT;
import static org.mule.maven.client.test.AllureConstants.MavenClient.SettingsSupplierFactoryStory.SETTINGS_SUPPLIER_FACTORY_STORY;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;

import java.io.File;
import java.io.IOException;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;

@Features(MAVEN_CLIENT)
@Stories(SETTINGS_SUPPLIER_FACTORY_STORY)
@RunWith(MockitoJUnitRunner.class)
public class DefaultSettingsSupplierFactoryTestCase {

  private static final String GS_S_COMMAND_LINE =
      "mvn -f pom.xml -s %s -gs %s  -Dbranch_wsc=master -Dbranch=master -Dextra_goals= -Dmaven.repo.local=/scratch/jenkins/workspace/.repository test -T 2 -Djava.net.preferIPv4Stack=true -fn -surefire.rerunFailingTestsCount=0";
  private static final String LONG_GS_S_COMMAND_LINE =
      "mvn -f pom.xml --settings %s --global-settings %s -Dbranch_wsc=master -Dbranch=master -Dextra_goals= -Dmaven.repo.local=/scratch/jenkins/workspace/.repository test -T 2 -Djava.net.preferIPv4Stack=true -fn -surefire.rerunFailingTestsCount=0";

  private File userSettings;
  private File globalSettings;

  @Mock
  private MavenEnvironmentVariables mavenEnvironmentVariables;

  @Before
  public void before() throws IOException {
    userSettings = File.createTempFile("settings", ".xml");
    globalSettings = File.createTempFile("globalSettings", ".xml");
  }

  @Test
  public void globalSettingsShortCmdParam() {
    assertCommandLineGlobalSettings(GS_S_COMMAND_LINE);
  }

  @Test
  public void globalSettingsLongCmdParam() {
    assertCommandLineGlobalSettings(LONG_GS_S_COMMAND_LINE);
  }

  @Test
  public void userSettingsShortCmdParam() {
    assertCommandLineSettings(GS_S_COMMAND_LINE);
  }

  @Test
  public void userSettingsLongCmdParam() {
    assertCommandLineSettings(LONG_GS_S_COMMAND_LINE);
  }

  @Test
  public void globalSettingsAsSystemProperty() throws Exception {
    assertSystemPropertyGlobalSettings(globalSettings);
  }

  @Test
  public void userSettingsAsSystemProperty() throws Exception {
    assertUserSystemPropertySettings(userSettings);
  }

  private void assertCommandLineGlobalSettings(String commandLine) {
    String formattedCommandLine = format(commandLine, userSettings.getAbsolutePath(), globalSettings.getAbsolutePath());

    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn(formattedCommandLine);
    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentGlobalSettingsSupplier().get().getAbsolutePath(),
               is(globalSettings.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
  }

  private void assertCommandLineSettings(String commandLine) {
    String formattedCommandLine = format(commandLine, userSettings.getAbsolutePath(), globalSettings.getAbsolutePath());

    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn(formattedCommandLine);
    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentUserSettingsSupplier().get().getAbsolutePath(),
               is(userSettings.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
  }

  private void assertSystemPropertyGlobalSettings(File settingsFile) throws Exception {
    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn("");
    when(mavenEnvironmentVariables.getSettingsFileAsSystemOrEnvProperty(GLOBAL_SETTINGS_SYSTEM_PROPERTY))
        .thenReturn(settingsFile);

    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentGlobalSettingsSupplier().get().getAbsolutePath(),
               is(settingsFile.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
    verify(mavenEnvironmentVariables).getSettingsFileAsSystemOrEnvProperty(GLOBAL_SETTINGS_SYSTEM_PROPERTY);
  }

  private void assertUserSystemPropertySettings(File settingsFile) throws Exception {
    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn("");
    when(mavenEnvironmentVariables.getSettingsFileAsSystemOrEnvProperty(USER_SETTINGS_SYSTEM_PROPERTY)).thenReturn(settingsFile);

    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentUserSettingsSupplier().get().getAbsolutePath(),
               is(settingsFile.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
    verify(mavenEnvironmentVariables).getSettingsFileAsSystemOrEnvProperty(USER_SETTINGS_SYSTEM_PROPERTY);
  }

}
