/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.client.internal.DefaultLocalRepositorySupplierFactory;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.net.URL;

/**
 * Helper class to work with tests using mule maven client
 *
 * @since 1.0
 */
public class MavenTestHelper {

  private static final String MULESOFT_PUBLIC_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/public/";
  private static final String MULESOFT_PRIVATE_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/private/";
  private static Logger LOGGER = LoggerFactory.getLogger(MavenTestHelper.class);

  /**
   * Creates a default maven configuration for enterprise edition.
   *
   * @return a new maven configuration based on the user environment.
   */
  public static MavenConfiguration createDefaultEnterpriseMavenConfiguration() throws IOException {
    MavenConfiguration mavenConfiguration = createDefaultEnterpriseMavenConfigurationBuilder().build();
    LOGGER.info("Using MavenConfiguration {}", mavenConfiguration);
    return mavenConfiguration;
  }

  /**
   * Creates a default maven configuration for community edition.
   *
   * @return a new maven configuration based on the user environment.
   */
  public static MavenConfiguration createDefaultCommunityMavenConfiguration() throws IOException {
    MavenConfiguration mavenConfiguration = createDefaultCommunityMavenConfigurationBuilder().build();
    LOGGER.info("Using MavenConfiguration {}", mavenConfiguration);
    return mavenConfiguration;
  }

  /**
   * Creates a maven config builder with the enterprise default settings.
   *
   * @return a new maven configuration builder based on the user environment.
   */
  public static MavenConfiguration.MavenConfigurationBuilder createDefaultEnterpriseMavenConfigurationBuilder()
      throws IOException {
    return createDefaultMavenConfigurationBuilder(true);
  }

  /**
   * Creates a maven config builder with the community default settings.
   *
   * @return a new maven configuration builder based on the user environment.
   */
  public static MavenConfiguration.MavenConfigurationBuilder createDefaultCommunityMavenConfigurationBuilder()
      throws IOException {
    return createDefaultMavenConfigurationBuilder(false);
  }

  /**
   * @return the local repository folder discovered for the environment.
   */
  public static File getLocalRepositoryFolder() {
    return new DefaultLocalRepositorySupplierFactory().environmentMavenRepositorySupplier().get();
  }

  /**
   * Creates a maven config builder with the default settings.
   *
   * @return a new maven configuration builder based on the user environment.
   */
  private static MavenConfiguration.MavenConfigurationBuilder createDefaultMavenConfigurationBuilder(boolean enterprise)
      throws IOException {
    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder =
        newMavenConfigurationBuilder().withForcePolicyUpdateNever(true);

    final File localMavenRepository = getLocalRepositoryFolder();
    mavenConfigurationBuilder.withLocalMavenRepositoryLocation(localMavenRepository);

    final DefaultSettingsSupplierFactory settingsSupplierFactory =
        new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());

    mavenConfigurationBuilder.withRemoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
        .withId("mulesoft-public")
        .withUrl(new URL(MULESOFT_PUBLIC_REPOSITORY))
        .build());
    if (enterprise) {
      mavenConfigurationBuilder.withRemoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
          .withId("mulesoft-private")
          .withUrl(new URL(MULESOFT_PRIVATE_REPOSITORY))
          .build());
    }
    settingsSupplierFactory.environmentUserSettingsSupplier().ifPresent(mavenConfigurationBuilder::withUserSettingsLocation);
    settingsSupplierFactory.environmentGlobalSettingsSupplier().ifPresent(mavenConfigurationBuilder::withGlobalSettingsLocation);
    return mavenConfigurationBuilder;
  }
}
