/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;

import org.eclipse.aether.transfer.TransferCancelledException;
import org.eclipse.aether.transfer.TransferEvent;
import org.eclipse.aether.transfer.TransferListener;
import org.hamcrest.Matcher;
import org.junit.Test;
import org.junit.internal.matchers.ThrowableCauseMatcher;
import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.AetherMavenClient;

import java.net.MalformedURLException;
import java.net.URL;

public class SystemConfiguratorTestCase extends AbstractMavenClientTestCase {

  private AetherMavenClient aetherClient;
  private TestTransferListener transferListener;

  @Override
  protected void beforeTest() throws Exception {
    mavenClient = aetherClient = new AetherMavenClient(createMavenConfiguration());
    TestTransferListener transferListener = new TestTransferListener();
    configureTransferListener(transferListener);
  }

  protected MavenConfiguration createMavenConfiguration() throws MalformedURLException {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(repositoryFolder.getRoot())
        .remoteRepository(newRemoteRepositoryBuilder()
            .id("maven-central")
            .url(new URL("https://repo.maven.apache.org/maven2/")).build())
        .build();
  }

  private void configureTransferListener(TestTransferListener transferListener) {
    aetherClient.setSessionConfigurator(session -> session.setTransferListener(transferListener));
    this.transferListener = transferListener;
  }

  @Test
  public void successfulTransferListenerLoggingTest() {
    BundleDescriptor bundleDescriptor = getExistentBundleDescriptor();
    aetherClient.resolveBundleDescriptor(bundleDescriptor);

    assertThat(transferListener.isInitiated(), is(true));
    assertThat(transferListener.isStarted(), is(true));
    assertThat(transferListener.isProgressed(), is(true));
    assertThat(transferListener.isSucceeded(), is(true));

    assertThat(transferListener.isCorrupted(), is(false));
    assertThat(transferListener.isFailed(), is(false));
  }

  private BundleDescriptor getExistentBundleDescriptor() {
    return new BundleDescriptor.Builder()
        .setGroupId("org.apache.maven")
        .setArtifactId("maven-core")
        .setVersion("3.5.0")
        .build();
  }

  @Test
  public void failedTransferListenerLoggingTest() {
    BundleDescriptor bundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.apache.maven")
        .setArtifactId("maven-core-should-not-exist")
        .setVersion("3.5.0")
        .build();

    try {
      aetherClient.resolveBundleDescriptor(bundleDescriptor);
      fail("the sentence above should have failed");
    } catch (BundleDependenciesResolutionException e) {
      // a resolution exception should have been raised
    }

    assertThat(transferListener.isInitiated(), is(true));
    assertThat(transferListener.isFailed(), is(true));

    assertThat(transferListener.isStarted(), is(false));
    assertThat(transferListener.isProgressed(), is(false));
    assertThat(transferListener.isSucceeded(), is(false));
    assertThat(transferListener.isCorrupted(), is(false));
  }

  @Test
  public void cancelTransferTest() {
    BundleDescriptor bundleDescriptor = getExistentBundleDescriptor();

    TransferCancelledException cancelledException = new TransferCancelledException();
    configureTransferListener(new TestTransferListener() {

      @Override
      public void transferInitiated(TransferEvent event) throws TransferCancelledException {
        throw cancelledException;
      }
    });

    expectedException.expect(transitiveCause(is(cancelledException)));
    aetherClient.resolveBundleDescriptor(bundleDescriptor);
  }

  private class TestTransferListener implements TransferListener {

    boolean initiated = false;
    boolean started = false;
    boolean progressed = false;
    boolean corrupted = false;
    boolean succeeded = false;
    boolean failed = false;

    @Override
    public void transferInitiated(TransferEvent event) throws TransferCancelledException {
      initiated = true;
    }

    @Override
    public void transferStarted(TransferEvent event) throws TransferCancelledException {
      started = true;
    }

    @Override
    public void transferProgressed(TransferEvent event) throws TransferCancelledException {
      progressed = true;
    }

    @Override
    public void transferCorrupted(TransferEvent event) throws TransferCancelledException {
      corrupted = true;
    }

    @Override
    public void transferSucceeded(TransferEvent event) {
      succeeded = true;
    }

    @Override
    public void transferFailed(TransferEvent event) {
      failed = true;
    }

    public boolean isInitiated() {
      return initiated;
    }

    public boolean isStarted() {
      return started;
    }

    public boolean isProgressed() {
      return progressed;
    }

    public boolean isCorrupted() {
      return corrupted;
    }

    public boolean isSucceeded() {
      return succeeded;
    }

    public boolean isFailed() {
      return failed;
    }
  }

  static <T extends Throwable> ThrowableCauseMatcher<T> transitiveCause(Matcher<T> causeMatcher) {
    return new TransitiveCauseMatcher<>(causeMatcher);
  }

  static class TransitiveCauseMatcher<T extends Throwable> extends ThrowableCauseMatcher<T> {


    public TransitiveCauseMatcher(Matcher<? extends Throwable> causeMatcher) {
      super(causeMatcher);
    }

    @Override
    protected boolean matchesSafely(T item) {
      boolean result = false;
      T current = item;
      while (current != null && !result) {
        result |= super.matchesSafely(current);
        current = (T) current.getCause();
      }
      return result;
    }
  }
}
