/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static com.google.common.io.Files.createTempDir;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.OfflineStory.OFFLINE_STORY;
import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;

import io.qameta.allure.Story;
import org.eclipse.aether.resolution.ArtifactResolutionException;
import org.junit.Test;

@Story(OFFLINE_STORY)
public class OffLineTestCase extends AbstractMavenClientTestCase {

  private String groupId = "org.mule";
  private String artifactId = "mule-core";
  private String version = "3.3.0";
  private String type = "jar";

  @Override
  protected void beforeTest() throws Exception {}

  private MavenConfiguration createMavenConfiguration(File localRepository) throws MalformedURLException {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .offlineMode(true)
        .remoteRepository(newRemoteRepositoryBuilder()
            .id("maven-central")
            .url(new URL("https://repo.maven.apache.org/maven2/")).build())
        .remoteRepository(newRemoteRepositoryBuilder()
            .id("mulesoft-public")
            .url(new URL("https://repository.mulesoft.org/nexus/content/repositories/public/")).build())
        .build();
  }

  @Test
  public void resolveRemoteBundleDescriptor() throws MalformedURLException {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration(createTempDir()));

    expectedException.expect(BundleDependenciesResolutionException.class);
    expectedException.expectCause(instanceOf(ArtifactResolutionException.class));
    expectedException.expectMessage("offline mode");
    mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder().setGroupId(groupId)
        .setArtifactId(artifactId)
        .setVersion(version)
        .setType(type).build());
  }

  @Test
  public void resolveInstalledLocalBundleDescriptor() throws MalformedURLException {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration(repositoryFolder.getRoot()));

    repositoryFolder.addArtifacts(artifact(artifactId).groupId(groupId).version(version).type(type).build());

    BundleDependency resolved = mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder().setGroupId(groupId)
        .setArtifactId(artifactId)
        .setVersion(version)
        .setType(type).build());

    final BundleDescriptor descriptor = resolved.getDescriptor();
    assertThat(descriptor.getGroupId(), equalTo(groupId));
    assertThat(descriptor.getArtifactId(), equalTo(artifactId));
    assertThat(descriptor.getVersion(), equalTo(version));
    assertThat(descriptor.getType(), equalTo(type));
  }

}
