/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.stream.Collectors.toList;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.hasItem;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.BundleScope.COMPILE;
import static org.mule.maven.client.api.model.BundleScope.PROVIDED;
import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.exception.IncompatibleMulePluginVersionResolutionException;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.BundleScope;
import org.mule.maven.client.api.model.MavenConfiguration;

import com.google.common.collect.ImmutableList;

import java.util.Arrays;
import java.util.List;

import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.junit.Test;

public abstract class AbstractOnlineMavenClientTestCase extends AbstractMavenClientTestCase {

  @Override
  protected void beforeTest() throws Exception {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration());
  }

  protected abstract MavenConfiguration createMavenConfiguration() throws Exception;

  @Test
  public void resolveExistentBundleDescriptor() {
    String artifactId = "artifact";
    String type = "jar";
    repositoryFolder.addArtifacts(artifact(artifactId).classifier(MULE_PLUGIN).type(type).build());
    BundleDependency coreBundle = mavenClient.resolveBundleDescriptor(getDescriptor(artifactId));

    assertThat(coreBundle.getDescriptor().getGroupId(), is(GROUP_ID));
    assertThat(coreBundle.getDescriptor().getArtifactId(), is(artifactId));
    assertThat(coreBundle.getDescriptor().getVersion(), is(VERSION));
    assertThat(coreBundle.getDescriptor().getType(), is(type));
    assertThat(coreBundle.getDescriptor().getClassifier().isPresent(), is(true));
    assertThat(coreBundle.getDescriptor().getClassifier().get(), is(MULE_PLUGIN));
  }

  @Test
  public void resolveNotExistentBundleDescriptor() {
    expectedException.expect(BundleDependenciesResolutionException.class);
    mavenClient.resolveBundleDescriptor(getDescriptor("artifact-does-not-exist"));
  }

  @Test
  public void resolvePluginDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("mule-http-connector").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("mule-sockets-connector")
                                                        .classifier(MULE_PLUGIN))
                                      .build());
    BundleDescriptor socketsBundleDescriptor = getDescriptor("mule-sockets-connector");
    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(ImmutableList.<BundleDescriptor>builder()
            .add(httpBundleDescriptor)
            .build());
    List<BundleDescriptor> result = bundleDependencies.stream().map(BundleDependency::getDescriptor)
        .map(bundleDescriptor -> new BundleDescriptor.Builder()
            .setGroupId(bundleDescriptor.getGroupId())
            .setArtifactId(bundleDescriptor.getArtifactId())
            .setVersion(VERSION)
            .setType(bundleDescriptor.getType())
            .setClassifier(bundleDescriptor.getClassifier().orElse(null))
            .build())
        .collect(toList());
    assertThat(result.size(), is(2));
    assertThat(result, contains(httpBundleDescriptor, socketsBundleDescriptor));
  }

  @Test
  public void resolvePluginDependenciesFromNonPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("library")
                                      .dependencies(
                                                    dependency("connector").classifier("mule-plugin"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, getDescriptor("library", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, hasItem("connector"));
  }

  @Test
  public void resolvePluginWithDependencyThatDependsOnPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier("mule-plugin").dependencies(
                                                                                                 dependency("library-a")
                                                                                                     .dependencies(
                                                                                                                   dependency("connector-b")
                                                                                                                       .classifier("mule-plugin")
                                                                                                                       .dependencies(
                                                                                                                                     dependency("library-b"))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, getDescriptor("connector-a"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, contains("library-a", "connector-b"));
    assertThat(result, not(hasItem("library-b")));
  }

  @Test
  public void resolvePluginWithDependencyThatDependsOnPluginProvided() {
    repositoryFolder.addArtifacts(artifact("connector-a").classifier("mule-plugin")
        .dependencies(
                      dependency("library-a"),
                      dependency("library-b").scope("provided"),
                      dependency("library-c").dependencies(dependency("library-c1").scope("provided")),
                      dependency("library-d").scope("provided").dependencies(dependency("library-d1")),
                      dependency("library-e").scope("test").dependencies(dependency("test-library-e1").scope("test")),
                      dependency("library-f").scope("test").dependencies(dependency("test-library-f1")))
        .build());

    List<BundleDependency> classpath =
        mavenClient.resolveBundleDescriptorDependencies(false, true, getDescriptor("connector-a"));
    assertThat(classpath, hasSize(4));
    assertThat(classpath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(classpath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-c", COMPILE, true));
    assertThat(classpath.get(2), BundleDependencyMatcher.bundleDependencyMatcher("library-b", PROVIDED, false));
    assertThat(classpath.get(3), BundleDependencyMatcher.bundleDependencyMatcher("library-d", PROVIDED, false));

    List<BundleDependency> testClasspath =
        mavenClient.resolveBundleDescriptorDependencies(true, true, getDescriptor("connector-a"));
    assertThat(testClasspath, hasSize(7));
    assertThat(testClasspath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-e", COMPILE, true));
    assertThat(testClasspath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-f", COMPILE, true));
    assertThat(testClasspath.get(2), BundleDependencyMatcher.bundleDependencyMatcher("test-library-f1", COMPILE, true));
    assertThat(testClasspath.get(3), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(testClasspath.get(4), BundleDependencyMatcher.bundleDependencyMatcher("library-c", COMPILE, true));
    assertThat(testClasspath.get(5), BundleDependencyMatcher.bundleDependencyMatcher("library-b", PROVIDED, false));
    assertThat(testClasspath.get(6), BundleDependencyMatcher.bundleDependencyMatcher("library-d", PROVIDED, false));
  }

  @Test
  public void resolveChainOfPluginDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("artifact").dependencies(
                                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                                        .dependencies(
                                                                                      dependency("connector-b")
                                                                                          .classifier(MULE_PLUGIN).dependencies(
                                                                                                                                dependency("connector-c")
                                                                                                                                    .classifier(MULE_PLUGIN))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, getDescriptor("artifact", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, contains("connector-a", "connector-b", "connector-c"));
  }

  @Test
  public void resolvePluginDependenciesDoesNotIncludeTransitiveDependenciesOfOtherPlugins() {
    repositoryFolder.addArtifacts(artifact("connector-a").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("connector-b")
                          .classifier(MULE_PLUGIN)
                          .dependencies(
                                        dependency("library")))
        .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector-a");
    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(true, httpBundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, hasItem("connector-b"));
    assertThat(result, not(hasItem("library")));
  }

  @Test
  public void resolveNonPluginDependenciesWithoutTestDependencies() {
    repositoryFolder.addArtifacts(artifact("connector").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("library-a")
                          .dependencies(dependency("library-a-dep")),
                      dependency("test-library")
                          .scope("test"))
        .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, httpBundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat("Direct non-plugin dependencies are not returned", result, hasItem("library-a"));
    assertThat("Direct non-plugin dependencies' dependencies are not returned", result,
               hasItem("library-a-dep"));
    assertThat("Test dependencies are returned (they shouldn't)", result, not(hasItem("test-library")));
  }

  @Test
  public void resolveNonPluginDependenciesWithRuntimeDependencies() {
    repositoryFolder.addArtifacts(artifact("connector").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("library-a")
                          .dependencies(dependency("library-a-dep")),
                      dependency("runtime-library")
                          .scope("runtime"))
        .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, httpBundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat("Direct non-plugin dependencies are not returned", result, hasItem("library-a"));
    assertThat("Direct non-plugin dependencies' dependencies are not returned", result,
               hasItem("library-a-dep"));
    assertThat("Runtime dependencies are not returned", result, hasItem("runtime-library"));
  }

  @Test
  public void resolveNonPluginDependenciesWithTestDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-a-dep")),
                                                    dependency("test-library")
                                                        .scope("test"))
                                      .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, httpBundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    // direct dependencies
    assertThat("Direct non-plugin dependencies are not returned", result, hasItem("library-a"));
    assertThat("Direct non-plugin dependencies' dependencies are not returned", result,
               hasItem("library-a-dep"));
    assertThat("Test dependencies are not returned", result, hasItem("test-library"));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("connector").version("1.1").classifier(MULE_PLUGIN).build(),
                                  artifact("connector").version("1.0").classifier(MULE_PLUGIN).build());

    BundleDescriptor connector10 = getDescriptor("connector"); // 1.0
    BundleDescriptor connector11 = getDescriptor("connector", "1.1", true);

    List<String> result =
        mavenClient.resolvePluginBundleDescriptorsDependencies(Arrays.asList(connector11, connector10)).stream()
            .map(dep -> dep.getDescriptor())
            .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());
    assertThat(result.size(), is(1));
    assertThat(result, hasItem("connector:1.1"));

    result = mavenClient.resolvePluginBundleDescriptorsDependencies(Arrays.asList(connector10, connector11)).stream()
        .map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());
    assertThat(result.size(), is(1));
    assertThat(result, hasItem("connector:1.1"));
  }

  @Test
  public void resolutionFailsWithIncompatibleVersions() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("2.0")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.0").classifier(MULE_PLUGIN).build());

    expectedException.expect(RuntimeException.class);
    mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                           Arrays.asList(getDescriptor("connector-a"),
                                                                         getDescriptor("connector-b")));
  }

  @Test
  public void resolutionFailsWithIncompatibleVersions_1() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("2.0")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.0").classifier(MULE_PLUGIN).build());

    expectedException.expect(RuntimeException.class);
    mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                           Arrays.asList(getDescriptor("connector-b"),
                                                                         getDescriptor("connector-a")));
  }

  @Test
  public void resolutionFailsWithIncompatibleVersions_2() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .classifier(MULE_PLUGIN)
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-c")
                                                                                                                     .version("2.0")
                                                                                                                     .classifier(MULE_PLUGIN)))
                                      .build(),
                                  artifact("connector-c").version("1.0").classifier(MULE_PLUGIN).build());

    expectedException.expect(RuntimeException.class);
    mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                           Arrays.asList(getDescriptor("connector-a"),
                                                                         getDescriptor("connector-c")));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPluginsConsideringDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("1.1")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.0").classifier(MULE_PLUGIN).build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                               Arrays.asList(getDescriptor("connector-b"),
                                                                             getDescriptor("connector-a")));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:1.1", "connector-a:" + VERSION));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPluginsConsideringDependencies_1() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("1.0")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.1").classifier(MULE_PLUGIN).build());

    BundleDescriptor descriptorB = getDescriptor("connector-b", "1.1", true);
    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(Arrays.asList(descriptorB, getDescriptor("connector-a")));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:1.1", "connector-a:" + VERSION));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPluginsConsideringTransitiveDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .classifier(MULE_PLUGIN)
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-c")
                                                                                                                     .version("1.1")
                                                                                                                     .classifier(MULE_PLUGIN)))
                                      .build(),
                                  artifact("connector-c").version("1.0").classifier(MULE_PLUGIN).build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                               Arrays.asList(getDescriptor("connector-a"),
                                                                             getDescriptor("connector-c")));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:" + VERSION, "connector-c:1.1", "connector-a:" + VERSION));
  }

  @Test
  public void fromPluginsOnlyNestedPluginsShouldBeIncludedInDependencyGraph() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("connector-b")
                                                        .classifier(MULE_PLUGIN)
                                                        .dependencies(
                                                                      dependency("connector-c")
                                                                          .classifier(MULE_PLUGIN),
                                                                      dependency("third-party").version("1.0")),
                                                    dependency("internal-library").version("1.0")
                                                        .dependencies(
                                                                      dependency("third-party").version("1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:" + VERSION,
                                "connector-c:" + VERSION,
                                "internal-library:1.0",
                                "third-party:1.0"));
  }

  @Test
  public void pluginsDoNotParticipateOnMavenVersionResolution() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("connector-b")
                                                        .classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("third-party").version("2.0")),
                                                    dependency("internal-library").version("1.0")
                                                        .dependencies(dependency("third-party").version("1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:" + VERSION,
                                "internal-library:1.0",
                                "third-party:1.0"));
  }

  @Test
  public void pluginDependsOnJarThatDependsOnPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("connector-b").classifier(MULE_PLUGIN)
                                                            .dependencies(dependency("library-d").version("1.1"))),
                                                    dependency("library-b").version("1.0")
                                                        .dependencies(dependency("library-c").version("1.0")
                                                            .dependencies(dependency("library-d").version("1.0"))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-a:" + VERSION, "connector-b:" + VERSION,
                                "library-b:" + VERSION,
                                "library-c:" + VERSION,
                                "library-d:" + VERSION));
  }

  @Test
  public void applicationThatDependsOnPluginThatDependsOnJarThatDependsOnPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(
                                                                      dependency("library-a").version("1.1")
                                                                          .dependencies(dependency("connector-b")
                                                                              .classifier(MULE_PLUGIN)
                                                                              .dependencies(
                                                                                            dependency("library-d").version("1.1")
                                                                                                .dependencies(
                                                                                                              dependency("library-e")
                                                                                                                  .version("1.1")
                                                                                                                  .dependencies(
                                                                                                                                dependency("connector-d")
                                                                                                                                    .classifier(MULE_PLUGIN))),
                                                                                            dependency("connector-c")
                                                                                                .classifier(MULE_PLUGIN)))),
                                                    dependency("library-b").version("1.0")
                                                        .dependencies(
                                                                      dependency("library-c").version("1.0")
                                                                          .dependencies(
                                                                                        dependency("library-d").version("1.0"))),
                                                    dependency("library-a").version("1.0"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION,
                                "connector-b:" + VERSION,
                                "connector-d:" + VERSION,
                                "connector-c:" + VERSION,
                                "library-b:1.0",
                                "library-c:1.0",
                                "library-d:1.0",
                                "library-a:1.0"));
  }

  @Test
  public void appWithPluginsWithSimpleCyclicDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)),
                                                    dependency("connector-c").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-a").classifier(MULE_PLUGIN)))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION,
                                "connector-c:" + VERSION));
  }

  @Test
  public void resolvePluginRootNodeShouldNotRemoveDirectDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a")
                                      .dependencies(
                                                    dependency("connector-b").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)),
                                                    dependency("library-1"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:" + VERSION,
                                "connector-c:" + VERSION,
                                "library-1:" + VERSION));
  }

  @Test
  public void appWithPluginsWithComplexCyclicDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .dependencies(
                                                                          dependency("connector-d").classifier(MULE_PLUGIN)
                                                                              .dependencies(dependency("connector-a")
                                                                                  .classifier(MULE_PLUGIN)))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION,
                                "connector-c:" + VERSION,
                                "connector-d:" + VERSION));
  }

  @Test
  public void appWithMultipleVersionsOfPlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")),
                                                    dependency("connector-b").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION,
                                "connector-b:" + VERSION,
                                "connector-c:1.1.0"));
  }

  @Test
  public void appWithIncompatibleVersionsOfPlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")),
                                                    dependency("connector-b").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("2.1.0")))
                                      .build());

    expectedException.expect(IncompatibleMulePluginVersionResolutionException.class);
    expectedException
        .expectMessage(containsString("Incompatible Mule plugins versions among: org.mule:connector-c:jar:mule-plugin:2.1.0, org.mule:connector-c:jar:mule-plugin:1.0.2"));

    mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-1:" + VERSION,
                                "connector-a:" + VERSION,
                                "connector-c:1.1.0"));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependenciesDifferentOrder() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0")),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-1:" + VERSION,
                                "connector-c:1.1.0",
                                "connector-a:" + VERSION));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependenciesTwoLevels() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.3.0"))),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.1.0"))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-1:" + VERSION,
                                "connector-c:1.1.0",
                                "connector-d:2.3.0",
                                "connector-a:" + VERSION));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependenciesTwoLevelsAndRootDefinitionOrSameConnector() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.1.0"))),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.3.0"))),
                                                    dependency("connector-d").classifier(MULE_PLUGIN).version("2.2.0"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-1:" + VERSION,
                                "connector-c:1.1.0",
                                "connector-a:" + VERSION,
                                "connector-d:2.2.0"));
  }

  @Test
  public void appSamePluginDefinedBothTimesMavenTakesLastDefinitionOfDependency() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-1").classifier(MULE_PLUGIN)
                                                        .version("1.1.0"),
                                                    dependency("connector-1").classifier(MULE_PLUGIN)
                                                        .version("1.0.2"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app", false));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-1:1.0.2"));
  }

  private static class BundleDependencyMatcher extends BaseMatcher<BundleDependency> {

    private String groupId;
    private String artifactId;
    private String version;
    private BundleScope scope;
    private boolean hasUri;

    static BundleDependencyMatcher bundleDependencyMatcher(String artifactId, BundleScope scope, boolean hasUri) {
      return new BundleDependencyMatcher(null, artifactId, null, scope, hasUri);
    }

    private BundleDependencyMatcher(String groupId, String artifactId, String version,
                                    BundleScope scope, boolean hasUri) {
      this.groupId = groupId;
      this.artifactId = artifactId;
      this.version = version;
      this.scope = scope;
      this.hasUri = hasUri;
    }

    @Override
    public boolean matches(Object o) {
      BundleDependency bundleDependency = (BundleDependency) o;
      final BundleDescriptor bundleDescriptor = bundleDependency.getDescriptor();
      boolean matches = true;
      if (groupId != null) {
        matches &= bundleDescriptor.getGroupId().equals(groupId);
      }
      if (artifactId != null) {
        matches &= bundleDescriptor.getArtifactId().equals(artifactId);
      }
      if (version != null) {
        matches &= bundleDescriptor.getVersion().equals(version);
      }
      if (hasUri) {
        matches &= bundleDependency.getBundleUri() != null;
      } else {
        matches &= bundleDependency.getBundleUri() == null;
      }
      return matches && bundleDependency.getScope().equals(scope);
    }

    @Override
    public void describeTo(Description description) {
      description.appendText("BundleDependency does not match.");

    }
  }
}
