/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.lang.String.format;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;

import org.eclipse.aether.resolution.ArtifactResolutionException;
import org.junit.Test;

public class ActivationTestCase extends AbstractMavenClientTestCase {

  private static final String GROUP_ID = "org.mule";
  private static final String ARTIFACT_ID = "mule-core";
  private static final String VERSION = "3.3.0";
  private static final String TYPE = "jar";
  private static final String JDK = "jdk";
  private static final String OS = "os";
  private static final String PROPERTY = "property";

  @Override
  protected void beforeTest() throws Exception {

  }

  private MavenConfiguration.MavenConfigurationBuilder getMavenConfigurationBuilder(String activation, boolean active,
                                                                                    String pomFile) {
    URL settingsUrl =
        getClass().getClassLoader().getResource(format("activation-poms/active-by-%s/%s", activation, pomFile));
    return newMavenConfigurationBuilder().localMavenRepositoryLocation(repositoryFolder.getRoot())
        .globalSettingsLocation(toFile(settingsUrl));
  }

  private void test(String activation, boolean active, String pomFile) {
    if (!active) {
      expectedException.expect(BundleDependenciesResolutionException.class);
      expectedException.expectCause(instanceOf(ArtifactResolutionException.class));
    }
    File localRepository;
    try {
      localRepository = temporaryFolder.newFolder();
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
    mavenClient = mavenClientProvider.createMavenClient(getMavenConfigurationBuilder(activation, active, pomFile)
        .localMavenRepositoryLocation(localRepository).build());
    BundleDependency resolved = mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder().setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .setType(TYPE).build());

    BundleDescriptor descriptor = resolved.getDescriptor();
    assertThat(descriptor.getGroupId(), equalTo(GROUP_ID));
    assertThat(descriptor.getArtifactId(), equalTo(ARTIFACT_ID));
    assertThat(descriptor.getVersion(), equalTo(VERSION));
    assertThat(descriptor.getType(), equalTo(TYPE));
  }

  @Test
  public void activeByExistentProperty() {
    System.setProperty("someProperty", "true");
    try {
      test(PROPERTY, true, "pom.xml");
    } finally {
      System.getProperties().remove("someProperty");
    }
  }

  @Test
  public void inactiveByNonExistentProperty() {
    test(PROPERTY, false, "pom.xml");
  }

  @Test
  public void activationByLinuxOS() {
    test(OS, isOS("linux"), "pom-linux.xml");
  }

  @Test
  public void activationByMacOS() {
    test(OS, isOS("mac"), "pom-mac.xml");
  }

  @Test
  public void activationByWindowsOS() {
    test(OS, isOS("windows"), "pom-windows.xml");
  }

  private boolean isOS(String os) {
    return System.getProperty("os.name").toLowerCase().indexOf(os) >= 0;
  }

  @Test
  public void activeByCorrectJDK() {
    test(JDK, true, "pom-active.xml");
  }

  @Test
  public void inactiveByOldJDK() {
    test(JDK, false, "pom-inactive.xml");
  }
}
