/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProjectVersion;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProperty;

import java.io.File;
import java.net.URISyntaxException;
import java.util.function.Supplier;

import org.junit.Test;

public class MavenTestUtilsTestCase {

  private static final String SOME_PROPERTY_KEY = "someProperty";
  private static final String SOME_PROPERTY_VALUE = "someValue";

  private static Supplier<File> buildPomFolderFinder(String folderName) {
    return () -> {
      try {
        return new File(MavenTestUtilsTestCase.class.getClassLoader().getResource(folderName).toURI());
      } catch (URISyntaxException e) {
        throw new RuntimeException(e);
      }
    };
  }

  private Supplier<File> pomWithProperties = buildPomFolderFinder("pom-with-properties");

  @Test
  public void mavenProperty() {
    assertThat(getMavenProperty(SOME_PROPERTY_KEY, pomWithProperties), is(SOME_PROPERTY_VALUE));
  }

  @Test
  public void getPlaceholderMavenProperty() {
    assertThat(getMavenProperty("somePlaceholderProperty", pomWithProperties), is(SOME_PROPERTY_VALUE));
  }

  @Test
  public void getOverriddenMavenProperty() {
    setProperty(SOME_PROPERTY_KEY, "fromSysProp");
    try {
      assertThat(getMavenProperty(SOME_PROPERTY_KEY, pomWithProperties), is("fromSysProp"));
    } finally {
      clearProperty(SOME_PROPERTY_KEY);
    }
  }

  @Test
  public void getMavenVersion() {
    assertThat(getMavenProjectVersion(pomWithProperties), is("2.1"));
  }

  @Test
  public void getPlaceholderMavenVersion() {
    assertThat(getMavenProjectVersion(buildPomFolderFinder("pom-with-placeholder-version")), is(SOME_PROPERTY_VALUE));
  }

  @Test
  public void getParentMavenVersion() {
    assertThat(getMavenProjectVersion(buildPomFolderFinder("pom-with-no-version")), is("1.2"));
  }

}
