/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.toFile;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.MAVEN_CLIENT;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.test.ArtifactCreator;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import io.qameta.allure.Story;
import org.hamcrest.Matchers;
import org.junit.Test;

@Story(MAVEN_CLIENT)
public class TransitiveDependencyInMulePluginTestCase extends AbstractMavenClientTestCase {

  @Override
  protected void beforeTest() {}

  private MavenConfiguration createMavenConfiguration(File localRepository) throws MalformedURLException {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .remoteRepository(newRemoteRepositoryBuilder()
            .id("maven-central")
            .url(new URL("https://repo.maven.apache.org/maven2/")).build())
        .ignoreArtifactDescriptorRepositories(false)
        .build();
  }

  @Test
  public void pluginClassLoaderWasNotHandlingCorrectlyConflictingTransitiveDependencies() throws IOException {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration(repositoryFolder.getRoot()));

    ArtifactCreator.DependencyBuilder xmlsec = dependency("xmlsec").groupId("org.apache.santuario").version("2.1.2");
    ArtifactCreator.DependencyBuilder wss4jwssecuritydom =
        dependency("wss4j-ws-security-dom").groupId("org.apache.wss4j").version("2.2.3");
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("mule-wss-module").classifier(MULE_PLUGIN)
                                                                   .dependencies(
                                                                                 xmlsec,
                                                                                 wss4jwssecuritydom))
                                      .build());

    // We will remove those artifacts from local repository as we want them to be downloaded from Maven central in order
    // to collect their transitive dependencies
    repositoryFolder.removeArtifact(xmlsec.build());
    repositoryFolder.removeArtifact(xmlsec.type("pom").build());
    repositoryFolder.removeArtifact(wss4jwssecuritydom.build());
    repositoryFolder.removeArtifact(wss4jwssecuritydom.type("pom").build());

    File appFile = toFile(mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId("app")
        .setVersion(VERSION)
        .setType("pom")
        .build()).getBundleUri().toURL());
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(appFile,
                                                false,
                                                false,
                                                of(repositoryFolder.getRoot()),
                                                empty());

    List<String> artifactIds = new ArrayList<>();
    bundleDependencies.stream().forEach(bundleDependency -> collectArtifactIds(artifactIds, bundleDependency));

    // ParentContextDependencyGraphTransformer was setting an artifactId wrong for xmlsec when it came from a transitive dependency from wss4j-ws-security-dom therefore there were two versions
    assertThat(artifactIds.stream().filter(artifactId -> artifactId.startsWith("org.apache.santuario:xmlsec")).collect(toList()),
               Matchers.contains("org.apache.santuario:xmlsec:2.1.2"));
  }

  private void collectArtifactIds(List<String> artifactIds, BundleDependency bundleDependency) {
    BundleDescriptor bundleDependencyDescriptor = bundleDependency.getDescriptor();
    String artifactId = bundleDependencyDescriptor.getGroupId() + ":" + bundleDependencyDescriptor.getArtifactId() + ":"
        + bundleDependencyDescriptor.getVersion();
    if (!artifactIds.contains(artifactId)) {
      artifactIds.add(artifactId);
    }
    bundleDependency.getTransitiveDependencies().stream()
        .forEach(transitiveDependency -> collectArtifactIds(artifactIds, transitiveDependency));
  }

}
