/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.lang.String.format;
import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.lang3.SystemUtils.IS_OS_LINUX;
import static org.apache.commons.lang3.SystemUtils.IS_OS_MAC;
import static org.apache.commons.lang3.SystemUtils.IS_OS_WINDOWS;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.internal.util.MavenUtils.getPomModelFromFile;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.AetherRepositoryState;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Properties;

import com.google.common.collect.ImmutableList;
import org.apache.maven.model.Model;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.installation.InstallRequest;
import org.eclipse.aether.installation.InstallationException;
import org.junit.Test;

public class PomProfilesActivationTestCase extends AbstractMavenClientTestCase {

  private static final String GROUP_ID = "org.mule";
  private static final String ARTIFACT_ID = "mule-core";
  private static final String VERSION = "3.3.0";
  private static final String TYPE = "pom";

  private static final String SOME_PROPERTY = "someProperty";

  private MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder;
  private AetherRepositoryState aetherRepositoryState;

  @Override
  protected void beforeTest() throws Exception {
    File localMavenRepository = repositoryFolder.newFolder();

    mavenConfigurationBuilder = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localMavenRepository)
        .ignoreArtifactDescriptorRepositories(false);

    aetherRepositoryState =
        new AetherRepositoryState(localMavenRepository, empty(), empty(),
                                  empty(), empty(), false, false, false, empty(),
                                  session -> {
                                  });
  }

  @Test
  public void activateProfileByProperty() throws IOException {
    Properties userProperties = new Properties();
    userProperties.setProperty(SOME_PROPERTY, "true");

    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.userProperties(userProperties).build());

    BundleDescriptor bundleDescriptor =
        installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-property/pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(1));
    assertDescriptor(bundleDependencies.get(0).getDescriptor());
  }

  @Test
  public void activateProfileByJdk() throws IOException {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

    BundleDescriptor bundleDescriptor =
        installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-jdk/active-by-jdk-pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(1));
    assertDescriptor(bundleDependencies.get(0).getDescriptor());
  }

  @Test
  public void activateProfileByLinuxOs() throws IOException {
    activeProfileByOs("linux", IS_OS_LINUX);
  }

  @Test
  public void activateProfileByWindowsOs() throws IOException {
    activeProfileByOs("windows", IS_OS_WINDOWS);
  }

  @Test
  public void activateProfileByMacOs() throws IOException {
    activeProfileByOs("mac", IS_OS_MAC);
  }

  @Test
  public void fileActivationNotSupported() throws IOException {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

    expectedException
        .expectMessage("Error while resolving dependencies for org.mule.tests:pom-active-by-file:pom:1.0.0-SNAPSHOT due to profiles activation by file are not supported");
    expectedException.expect(UnsupportedOperationException.class);
    mavenClient
        .resolveArtifactDependencies(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-file/pom.xml")),
                                     false, false, empty(), empty());
    expectedException.reportMissingExceptionWithMessage("Should not be supported to activate profiles by file");
  }

  @Test
  public void inactivateProfileByJdk() throws IOException {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

    BundleDescriptor bundleDescriptor =
        installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-jdk/inactive-by-jdk-pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(0));
  }

  @Test
  public void activateProfilesExplicitlyAAreNotConsideredWhenResolvingBundleDescriptorDependencies() throws IOException {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .activeProfiles(ImmutableList.<String>builder().add("mule-core-dependency").build()).build());

    BundleDescriptor bundleDescriptor =
        installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-property/pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(0));
  }

  @Test
  public void activateProfileExplicitlyAreConsideredWhenResolvingArtifactDependencies() throws IOException {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .activeProfiles(ImmutableList.<String>builder().add("mule-core-dependency").build()).build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(toFile(getClass().getClassLoader()
            .getResource("activation-poms/active-by-property/pom.xml")),
                                                false, false, empty(), empty());
    assertThat(bundleDependencies, hasSize(1));
    assertDescriptor(bundleDependencies.get(0).getDescriptor());
  }

  @Test
  public void inactivateProfileExplicitlyAreNotConsideredWhenResolvingBundleDescriptorDependencies() throws IOException {
    Properties userProperties = new Properties();
    userProperties.setProperty(SOME_PROPERTY, "true");

    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .inactiveProfiles(ImmutableList.<String>builder().add("mule-core-dependency").build())
        .userProperties(userProperties)
        .build());

    BundleDescriptor bundleDescriptor =
        installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-property/pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(1));
  }

  @Test
  public void inactivateProfileExplicitlyAreConsideredWhenResolvingArtifactDependencies() throws IOException {
    Properties userProperties = new Properties();
    userProperties.setProperty(SOME_PROPERTY, "true");

    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .inactiveProfiles(ImmutableList.<String>builder().add("mule-core-dependency").build())
        .userProperties(userProperties)
        .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(toFile(getClass().getClassLoader()
            .getResource("activation-poms/active-by-property/pom.xml")),
                                                false, false, empty(), empty());
    assertThat(bundleDependencies, hasSize(0));
  }

  @Test
  public void activateProfileBySystemProperty() throws IOException {
    String oldValue = setProperty(SOME_PROPERTY, "true");
    try {
      mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

      BundleDescriptor bundleDescriptor =
          installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-property/pom.xml")));

      List<BundleDependency> bundleDependencies =
          mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
      assertThat(bundleDependencies, hasSize(1));
      assertDescriptor(bundleDependencies.get(0).getDescriptor());
    } finally {
      if (oldValue != null) {
        setProperty(SOME_PROPERTY, oldValue);
      } else {
        clearProperty(SOME_PROPERTY);
      }
    }
  }

  @Test
  public void activateProfileByPropertyWithParentPom() throws IOException {
    Properties userProperties = new Properties();
    userProperties.setProperty(SOME_PROPERTY, "true");

    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.userProperties(userProperties).build());

    installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/parent-pom/parent-pom.xml")));
    BundleDescriptor bundleDescriptor = installArtifact(toFile(getClass().getClassLoader()
        .getResource("activation-poms/active-by-property-with-parent-pom/pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(2));
    assertDescriptor(bundleDependencies.get(0).getDescriptor());
  }

  @Test
  public void noActivationProfile() {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .activeProfiles(ImmutableList.<String>builder().add("mule-core-dependency").build()).build());

    List<BundleDependency> bundleDependencies = mavenClient
        .resolveArtifactDependencies(toFile(getClass().getClassLoader().getResource("pom-with-no-profile-activation/pom.xml")),
                                     false, false, empty(), empty());
    assertThat(bundleDependencies, hasSize(1));
  }

  public void activeProfileByOs(String os, boolean shouldBeActivated) {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

    BundleDescriptor bundleDescriptor = installArtifact(toFile(getClass().getClassLoader()
        .getResource(format("activation-poms/active-by-os/active-by-%s-os-pom.xml", os))));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    if (shouldBeActivated) {
      assertThat(bundleDependencies, hasSize(1));
      assertDescriptor(bundleDependencies.get(0).getDescriptor());
    } else {
      assertThat(bundleDependencies, hasSize(0));
    }
  }

  private BundleDescriptor installArtifact(File artifactFile) {
    Model pomModel = getPomModelFromFile(artifactFile);
    InstallRequest request = new InstallRequest();
    request.addArtifact(new DefaultArtifact(pomModel.getGroupId(), pomModel.getArtifactId(), null, "pom", pomModel.getVersion(),
                                            emptyMap(), artifactFile));
    try {
      aetherRepositoryState.getSystem().install(aetherRepositoryState.getSession(), request);
      return new BundleDescriptor.Builder()
          .setGroupId(pomModel.getGroupId())
          .setArtifactId(pomModel.getArtifactId())
          .setVersion(pomModel.getVersion())
          .setType(pomModel.getPackaging())
          .build();
    } catch (InstallationException e) {
      throw new RuntimeException(e);
    }
  }

  private void assertDescriptor(BundleDescriptor descriptor) {
    assertThat(descriptor.getGroupId(), equalTo(GROUP_ID));
    assertThat(descriptor.getArtifactId(), equalTo(ARTIFACT_ID));
    assertThat(descriptor.getVersion(), equalTo(VERSION));
    assertThat(descriptor.getType(), equalTo(TYPE));
  }

}
