/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static com.google.common.base.Throwables.getRootCause;
import static org.apache.commons.io.FileUtils.toFile;
import static org.codehaus.plexus.util.FileUtils.copyDirectoryStructure;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsInstanceOf.instanceOf;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.RemoteRepositories.REMOTE_REPOSITORIES_PRECEDENCE;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;

import io.qameta.allure.Description;
import io.qameta.allure.Story;
import org.apache.http.client.HttpResponseException;
import org.junit.Test;

@Story(REMOTE_REPOSITORIES_PRECEDENCE)
public class RemoteRepositoriesPrecedenceTestCase extends AbstractMavenClientTestCase {

  private static final String MULE_CORE_EE_GAV = "com.mulesoft.mule:mule-core-ee:jar:3.3.0";

  @Override
  protected void beforeTest() throws Exception {}

  @Test
  @Description("Validates that if no settings neither remote repositories are provided dependencies could not be fetched")
  public void cannotResolveDependencyWithoutSettingsOrRemoteRepositories() throws Exception {
    expectedException.expect(BundleDependenciesResolutionException.class);
    expectedException.expectMessage(MULE_CORE_EE_GAV);
    resolveDependencies("pom-with-repositories-with-dependencies", null);
  }

  @Test
  @Description("Validates that if a settings is provided with correct serverIds the remote repositories declared in pom.xml are ignored therefore dependencies could not be fetched")
  public void failsToCollectDependenciesWhenSettingsHasNoRemoteRepositories()
      throws Exception {
    final String artifactId = "pom-with-repositories-with-dependencies";
    File settings = toFile(getClass().getClassLoader().getResource(artifactId + "/settings.xml"));

    expectedException.expect(BundleDependenciesResolutionException.class);
    expectedException.expectMessage(MULE_CORE_EE_GAV);
    resolveDependencies(artifactId, settings);
  }

  @Test
  @Description("Validates that if en empty settings is provided dependencies could be fetched from a remote repository declared on MavenConfiguration")
  public void resolvesDependencyUsingRemoteRepositoryFromMavenConfiguration() throws Exception {
    List<BundleDependency> dependencies = resolveDependencies("pom-with-repositories-with-public-dependencies", null,
                                                              new RemoteRepository.RemoteRepositoryBuilder()
                                                                  .id("mulesoft-public")
                                                                  .url(new URL("https://repository.mulesoft.org/nexus/content/repositories/public"))
                                                                  .build());
    assertThat(dependencies, hasSize(1));
    assertCoreDependency(dependencies.get(0));
  }

  @Test
  @Description("Validates that remote repositories declared to the MavenConfiguration have precedence over the ones on settings and ignored the ones from pom, therefore artifacts can be found")
  public void remoteRepositoriesOnMavenConfigurationHasPrecedenceOverSettings() throws Exception {
    final String artifactId = "pom-with-repositories-with-public-dependencies";
    File settings = toFile(getClass().getClassLoader().getResource(artifactId + "/wrong-server-urls-settings.xml"));
    List<BundleDependency> dependencies = resolveDependencies("pom-with-repositories-with-public-dependencies", settings,
                                                              new RemoteRepository.RemoteRepositoryBuilder()
                                                                  .id("mulesoft-public")
                                                                  .url(new URL("https://repository.mulesoft.org/nexus/content/repositories/public"))
                                                                  .build(),
                                                              new RemoteRepository.RemoteRepositoryBuilder()
                                                                  .id("central")
                                                                  .url(new URL("https://repo.maven.apache.org/maven2-wrong"))
                                                                  .build());
    assertThat(dependencies, hasSize(1));
    assertCoreDependency(dependencies.get(0));
  }

  @Test
  @Description("Validates that remote repositories declared on MavenConfiguration without authentication are fulfilled when server authentications from settings.xml")
  public void useSettingsServerForMavenConfigurationRemoteRepositoriesWhenNoAuthenticationIsProvided() throws Exception {
    final String artifactId = "pom-with-repositories-with-dependencies";
    File settings = toFile(getClass().getClassLoader().getResource(artifactId + "/settings.xml"));

    try {
      resolveDependencies(artifactId, settings, new RemoteRepository.RemoteRepositoryBuilder()
          .id("mulesoft-private")
          .url(new URL("https://repository.mulesoft.org/nexus/content/repositories/private"))
          .build());
      fail("Should have thrown an error while resolving dependencies");
    } catch (Exception e) {
      final Throwable rootCause = getRootCause(e);
      assertThat(rootCause, instanceOf(HttpResponseException.class));
      assertThat(rootCause.getMessage(), containsString("Unauthorized"));
    }
  }

  private void assertCoreDependency(BundleDependency coreDependency) {
    assertThat(coreDependency.getDescriptor().getGroupId(), is("org.mule"));
    assertThat(coreDependency.getDescriptor().getArtifactId(), is("mule-core"));
    assertThat(coreDependency.getDescriptor().getVersion(), is("3.3.0"));
    assertThat(coreDependency.getDescriptor().getClassifier().isPresent(), is(false));
  }

  private List<BundleDependency> resolveDependencies(String artifactId, File settings, RemoteRepository... remoteRepositories)
      throws IOException {
    final File localMavenRepository = repositoryFolder.newFolder();
    copyDirectoryStructure(toFile(getClass().getClassLoader().getResource(artifactId + "/repository")),
                           localMavenRepository);

    final MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localMavenRepository);
    if (settings != null) {
      mavenConfigurationBuilder.userSettingsLocation(settings);
    }

    for (RemoteRepository remoteRepository : remoteRepositories) {
      mavenConfigurationBuilder.remoteRepository(remoteRepository);
    }

    MavenConfiguration mavenConfiguration = mavenConfigurationBuilder.build();
    mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration);

    final BundleDescriptor bundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.mule.tests")
        .setArtifactId(artifactId)
        .setVersion("1.0.0-SNAPSHOT")
        .setType("pom")
        .build();

    return getBundleDependencies(bundleDescriptor);
  }

  private List<BundleDependency> getBundleDependencies(BundleDescriptor bundleDescriptor) {
    BundleDependency pomBundle = mavenClient.resolveBundleDescriptor(bundleDescriptor);
    assertThat(pomBundle.getDescriptor().getGroupId(), is(bundleDescriptor.getGroupId()));
    assertThat(pomBundle.getDescriptor().getArtifactId(), is(bundleDescriptor.getArtifactId()));
    assertThat(pomBundle.getDescriptor().getVersion(), is(bundleDescriptor.getVersion()));
    assertThat(pomBundle.getDescriptor().getType(), is(bundleDescriptor.getType()));
    assertThat(pomBundle.getDescriptor().getClassifier().isPresent(), is(false));

    return mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
  }

}
