/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.Arrays.asList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.collection.IsIterableWithSize.iterableWithSize;
import static org.hamcrest.core.CombinableMatcher.both;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsCollectionContaining.hasItem;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.BundleScope.COMPILE;
import static org.mule.maven.client.api.model.BundleScope.PROVIDED;
import static org.mule.maven.client.api.model.BundleScope.RUNTIME;
import static org.mule.maven.client.api.model.BundleScope.SYSTEM;
import static org.mule.maven.client.api.model.BundleScope.TEST;
import static org.mule.maven.client.test.dependency.BundleDependencyMatcher.BundleDependencyMatcherBuilder.withArtifactId;
import static org.mule.maven.client.test.dependency.BundleDependencyMatcher.aBundleDependency;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.exception.IncompatibleMulePluginVersionResolutionException;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.test.ArtifactCreator;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.List;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.junit.Test;

public abstract class AbstractOnlineMavenClientTestCase extends AbstractMavenClientTestCase {

  protected static final String MULE_PLUGIN = "mule-plugin";
  private static final String MULE_DOMAIN = "mule-domain";

  @Override
  protected void beforeTest() throws Exception {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration());
  }

  protected abstract MavenConfiguration createMavenConfiguration() throws Exception;

  @Test
  public void resolveExistentBundleDescriptor() {
    String artifactId = "artifact";
    String type = "jar";
    repositoryFolder.addArtifacts(artifact(artifactId).classifier(MULE_PLUGIN).type(type).build());
    BundleDependency coreBundle = mavenClient.resolveBundleDescriptor(getDescriptor(artifactId, VERSION, type, MULE_PLUGIN));
    assertThat(coreBundle, is(aBundleDependency(withArtifactId(artifactId).groupId(GROUP_ID).version(VERSION).type(type).and()
        .classifier(MULE_PLUGIN))));
  }

  @Test
  public void resolveNotExistentBundleDescriptor() {
    expectedException.expect(BundleDependenciesResolutionException.class);
    mavenClient.resolveBundleDescriptor(getDescriptor("artifact-does-not-exist"));
  }

  @Test
  public void sameDependencyWithDifferentClassifierAllAreIncluded() {
    final String artifactName = "app";
    final String dependencyName = "dependency";
    final String classifierOne = "classifier-one";
    final String classifierTwo = "classifier-two";
    repositoryFolder.addArtifacts(
                                  artifact(artifactName).dependencies(
                                                                      dependency(dependencyName).classifier(classifierOne),
                                                                      dependency(dependencyName).classifier(classifierTwo))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor(artifactName));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(
                                                              withArtifactId(dependencyName)
                                                                  .classifier(classifierOne)
                                                                  .and()
                                                                  .withUri()),
                                            aBundleDependency(
                                                              withArtifactId(dependencyName)
                                                                  .classifier(classifierTwo)
                                                                  .and()
                                                                  .withUri())

    ));
  }

  @Test
  public void resolvePluginDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("mule-http-connector").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("mule-sockets-connector")
                                                        .classifier(MULE_PLUGIN))
                                      .build());
    BundleDescriptor socketsBundleDescriptor = getDescriptor("mule-sockets-connector");
    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(ImmutableList.<BundleDescriptor>builder()
            .add(httpBundleDescriptor)
            .build());
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId(httpBundleDescriptor.getArtifactId()).and()
                                                .version(httpBundleDescriptor.getVersion())),
                                            aBundleDependency(withArtifactId(socketsBundleDescriptor.getArtifactId()).and()
                                                .version(socketsBundleDescriptor.getVersion()))));
  }

  @Test
  public void mulePluginConflictResolutionUseSemVer() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("mule-sockets-connector").version("1.1.5")
                                                            .classifier(MULE_PLUGIN),
                                                                      dependency("library-a")),
                                                    dependency("mule-sockets-connector").classifier(MULE_PLUGIN).version("1.1.0"))
                                      .build());

    BundleDescriptor socketsBundleDescriptor = getDescriptor("mule-sockets-connector", "1.1.5");
    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, true, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(
                                                  withArtifactId(httpBundleDescriptor.getArtifactId())
                                                      .version(httpBundleDescriptor.getVersion()).scope(COMPILE).withUri().and()
                                                      .transitiveDependenciesThat(
                                                                                  both(
                                                                                       is(iterableWithSize(2)))
                                                                                           .and(
                                                                                                hasItem(
                                                                                                        aBundleDependency(
                                                                                                                          withArtifactId(socketsBundleDescriptor
                                                                                                                              .getArtifactId())
                                                                                                                                  .version(socketsBundleDescriptor
                                                                                                                                      .getVersion())
                                                                                                                                  .scope(COMPILE)
                                                                                                                                  .and()
                                                                                                                                  .withUri()))))),
                                aBundleDependency(
                                                  withArtifactId(socketsBundleDescriptor.getArtifactId())
                                                      .version(socketsBundleDescriptor.getVersion()).scope(COMPILE).and()
                                                      .withUri())));
  }

  @Test
  public void compileTransitiveDependencyFromTestIsWinner() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("artifact-a")
                                                        .dependencies(dependency("artifact-b")
                                                            .dependencies(dependency("artifact-c").version("1.0"))),
                                                    dependency("test-artifact-a").scope("test")
                                                        .dependencies(dependency("artifact-c").version("2.0")))
                                      .build());

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result.size(), is(3));

    assertThat(result, contains(
                                aBundleDependency(withArtifactId("artifact-a").version("1.0").scope(COMPILE).withUri().and()
                                    .transitiveDependenciesThat(
                                                                contains(
                                                                         aBundleDependency(withArtifactId("artifact-b")
                                                                             .version("1.0")
                                                                             .scope(COMPILE).withUri().and()
                                                                             .transitiveDependenciesThat(
                                                                                                         contains(
                                                                                                                  aBundleDependency(withArtifactId("artifact-c")
                                                                                                                      .version("2.0")
                                                                                                                      .scope(COMPILE)
                                                                                                                      .withUri()
                                                                                                                      .and()
                                                                                                                      .transitiveDependenciesThat(hasSize(0))))))))),
                                aBundleDependency(withArtifactId("artifact-b").version("1.0").scope(COMPILE).withUri().and()
                                    .transitiveDependenciesThat(contains(
                                                                         aBundleDependency(withArtifactId("artifact-c")
                                                                             .version("2.0")
                                                                             .scope(COMPILE).withUri().and()
                                                                             .transitiveDependenciesThat(hasSize(0)))))),
                                aBundleDependency(withArtifactId("artifact-c").version("2.0").scope(COMPILE).withUri().and()
                                    .transitiveDependenciesThat(hasSize(0)))));
  }

  @Test
  public void providedDependencyScopeShouldOverrideTransitiveDependenciesScopes() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("mule-sockets-connector")
                                                            .classifier(MULE_PLUGIN),
                                                                      dependency("library-a")),
                                                    dependency("mule-sockets-connector").classifier(MULE_PLUGIN)
                                                        .scope("provided"),
                                                    dependency("library-a").scope("provided"))
                                      .build());

    BundleDescriptor socketsBundleDescriptor = getDescriptor("mule-sockets-connector");
    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, true, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(withArtifactId(httpBundleDescriptor.getArtifactId()).scope(COMPILE).withUri()
                                    .and()
                                    .transitiveDependenciesThat(
                                                                contains(
                                                                         aBundleDependency(withArtifactId("library-a")
                                                                             .scope(COMPILE)
                                                                             .and().withUri())))),
                                aBundleDependency(withArtifactId(socketsBundleDescriptor.getArtifactId()).scope(PROVIDED).and()
                                    .withoutUri()),
                                aBundleDependency(withArtifactId("library-a").scope(PROVIDED).withoutUri())));
  }

  @Test
  public void providedDependencyScopeOverrideNotIncludedInTransitiveDependency() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("mule-sockets-connector")
                                                            .classifier(MULE_PLUGIN),
                                                                      dependency("library-a")),
                                                    dependency("mule-sockets-connector").classifier(MULE_PLUGIN)
                                                        .scope("provided"),
                                                    dependency("library-a").scope("provided"))
                                      .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(withArtifactId(httpBundleDescriptor.getArtifactId()).scope(COMPILE).withUri()
                                    .and()
                                    .transitiveDependenciesThat(
                                                                contains(
                                                                         aBundleDependency(withArtifactId("library-a")
                                                                             .scope(COMPILE)
                                                                             .and().withUri()))))));
  }

  @Test
  public void exclusionFromLibraryThatIsAMulePluginDependency() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-plugin-a").classifier(MULE_PLUGIN)
                                                        .dependencies(
                                                                      dependency("mule-plugin-b").classifier(MULE_PLUGIN)
                                                                          .dependencies(
                                                                                        dependency("library-a")
                                                                                            .dependencies(dependency("library-b")
                                                                                                .dependencies(dependency("library-c")))
                                                                                            .exclusions(exclusion("library-c")))))
                                      .build());

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(withArtifactId("mule-plugin-a")),
                                aBundleDependency(withArtifactId("mule-plugin-b").and().transitiveDependenciesThat(
                                                                                                                   contains(aBundleDependency(withArtifactId("library-a")
                                                                                                                       .and()
                                                                                                                       .transitiveDependenciesThat(
                                                                                                                                                   contains(aBundleDependency(withArtifactId("library-b")
                                                                                                                                                       .and()
                                                                                                                                                       .transitiveDependenciesThat(hasSize(0)))))))))));
  }

  @Test
  public void exclusionOfMulePlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("mule-sockets-connector")
                                                            .classifier(MULE_PLUGIN))
                                                        .exclusions(exclusion("mule-sockets-connector")))
                                      .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(withArtifactId(httpBundleDescriptor.getArtifactId()).scope(COMPILE).withUri()
                                    .and()
                                    .transitiveDependenciesThat(hasSize(0)))));
  }

  @Test
  public void exclusionOfArtifactsFromMulePluginsShouldNotWork() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("library-a"),
                                                                      dependency("library-b"))
                                                        .exclusions(exclusion("library-a")))
                                      .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(withArtifactId(httpBundleDescriptor.getArtifactId()).scope(COMPILE).withUri()
                                    .and()
                                    .transitiveDependenciesThat(
                                                                contains(
                                                                         aBundleDependency(withArtifactId("library-a")),
                                                                         aBundleDependency(withArtifactId("library-b")))))));
  }


  @Test
  public void exclusionOfArtifactsFromMulePluginsWorksOnLowerLevels() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-plugin-a").classifier(MULE_PLUGIN)
                                                        .dependencies(
                                                                      dependency("library-a").dependencies(
                                                                                                           dependency("library-a1")
                                                                                                               .dependencies(dependency("library-b1")
                                                                                                                   .dependencies(dependency("dependency")
                                                                                                                       .version("2.0.0"))),
                                                                                                           dependency("library-a2")
                                                                                                               .dependencies(dependency("dependency")
                                                                                                                   .version("1.0.0"))
                                                                                                               .exclusions(exclusion("dependency"))))
                                                        .exclusions(exclusion("dependency-a")))
                                      .build());

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(withArtifactId("mule-plugin-a")
                                    .and()
                                    .transitiveDependenciesThat(
                                                                contains(
                                                                         aBundleDependency(withArtifactId("library-a").and()
                                                                             .transitiveDependenciesThat(contains(
                                                                                                                  aBundleDependency(withArtifactId("library-a1")
                                                                                                                      .and()
                                                                                                                      .transitiveDependenciesThat(contains(
                                                                                                                                                           aBundleDependency(withArtifactId("library-b1")
                                                                                                                                                               .and()
                                                                                                                                                               .transitiveDependenciesThat(contains(
                                                                                                                                                                                                    aBundleDependency(withArtifactId("dependency")
                                                                                                                                                                                                        .and()
                                                                                                                                                                                                        .version("2.0.0")))))))),
                                                                                                                  aBundleDependency(withArtifactId("library-a2")
                                                                                                                      .and()
                                                                                                                      .transitiveDependenciesThat(hasSize(0)))))))))));
  }

  @Test
  public void exclusionOfDependenciesFromOtherDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("dependency-a").dependencies(
                                                                                            dependency("library-a"),
                                                                                            dependency("library-b"))
                                                        .exclusions(exclusion("library-a")))
                                      .build());

    BundleDescriptor dependencyBundleDescriptor = getDescriptor("dependency-a");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(
                                                  withArtifactId(dependencyBundleDescriptor.getArtifactId())
                                                      .scope(COMPILE).withUri()
                                                      .and()
                                                      .transitiveDependenciesThat(
                                                                                  contains(
                                                                                           aBundleDependency(withArtifactId("library-b"))))),
                                aBundleDependency(withArtifactId("library-b"))));

  }

  @Test
  public void pluginDeclaredAsProvidedShouldOverrideCompileWithGreaterVersion() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("plugin-a").classifier(MULE_PLUGIN).dependencies(
                                                                                                                dependency("provided-plugin")
                                                                                                                    .version("1.1.0")
                                                                                                                    .classifier(MULE_PLUGIN)),
                                                    dependency("provided-plugin")
                                                        .classifier(MULE_PLUGIN)
                                                        .version("1.0.0")
                                                        .scope("provided"))
                                      .build());

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(
                                                  withArtifactId("plugin-a")
                                                      .classifier(MULE_PLUGIN)
                                                      .and()
                                                      .scope(COMPILE).withUri())));
  }

  @Test
  public void pluginDeclaredAsProvidedShouldOverrideCompileWithSmallerVersion() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("plugin-a").classifier(MULE_PLUGIN).dependencies(
                                                                                                                dependency("provided-plugin")
                                                                                                                    .version("1.0.0")
                                                                                                                    .classifier(MULE_PLUGIN)),
                                                    dependency("provided-plugin")
                                                        .classifier(MULE_PLUGIN)
                                                        .version("1.1.0")
                                                        .scope("provided"))
                                      .build());

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(
                                                  withArtifactId("plugin-a")
                                                      .classifier(MULE_PLUGIN)
                                                      .withUri()
                                                      .scope(COMPILE).and().transitiveDependenciesThat(hasSize(0)))));
  }

  @Test
  public void resolveZipDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("zip-dependency-1").type("zip")
                                                        .dependencies(dependency("zip-dependency-2").type("zip")
                                                            .dependencies(dependency("library-jar-1")).type("jar")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("zip-dependency-1")),
                                            aBundleDependency(withArtifactId("zip-dependency-2")),
                                            aBundleDependency(withArtifactId("library-jar-1"))

    ));
  }

  @Test
  public void resolvePluginDependenciesFromNonPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("library")
                                      .dependencies(
                                                    dependency("connector").classifier(MULE_PLUGIN))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, getDescriptor("library"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies,
               contains(
                        aBundleDependency(withArtifactId("connector"))));
  }

  @Test
  public void resolveArtifactFromListOfDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("library-a")
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-b")
                                                                                                                     .classifier(MULE_PLUGIN)
                                                                                                                     .dependencies(
                                                                                                                                   dependency("library-b"))))
                                      .build());

    BundleDescriptor descriptor = getDescriptor("connector-a");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(Lists.newArrayList(descriptor),
                                                of(mavenClient.getMavenConfiguration().getLocalMavenRepositoryLocation()),
                                                empty());
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a")),
                                            aBundleDependency(withArtifactId("library-a")),
                                            aBundleDependency(withArtifactId("connector-b"))));
  }

  @Test
  public void resolvePluginWithDependencyThatDependsOnPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("library-a")
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-b")
                                                                                                                     .classifier(MULE_PLUGIN)
                                                                                                                     .dependencies(
                                                                                                                                   dependency("library-b"))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, getDescriptor("connector-a"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a")),
                                            aBundleDependency(withArtifactId("connector-b"))));
  }

  @Test
  public void resolvePluginWithDependencyThatDependsOnPluginProvided() {
    repositoryFolder.addArtifacts(artifact("connector-a").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("library-a"),
                      dependency("library-b").scope("provided"),
                      dependency("library-c").dependencies(dependency("library-c1").scope("provided")),
                      dependency("library-d").scope("provided").dependencies(dependency("library-d1")),
                      dependency("library-e").scope("test").dependencies(dependency("test-library-e1").scope("test")),
                      dependency("library-f").scope("test").dependencies(dependency("test-library-f1")))
        .build());

    List<BundleDependency> runtimeClasspath =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("connector-a"));
    checkNoDuplicatedInstances(runtimeClasspath);
    assertThat(runtimeClasspath, contains(
                                          aBundleDependency(withArtifactId("library-a").scope(COMPILE).withUri()),
                                          aBundleDependency(withArtifactId("library-c").scope(COMPILE).withUri())));

    List<BundleDependency> testClasspath =
        mavenClient.resolveBundleDescriptorDependencies(true, false, getDescriptor("connector-a"));
    checkNoDuplicatedInstances(testClasspath);
    assertThat(testClasspath, contains(
                                       aBundleDependency(withArtifactId("library-a").scope(COMPILE).withUri()),
                                       aBundleDependency(withArtifactId("library-c").scope(COMPILE).withUri()),
                                       aBundleDependency(withArtifactId("library-e").scope(TEST).withUri()),
                                       aBundleDependency(withArtifactId("library-f").scope(TEST).withUri()),
                                       aBundleDependency(withArtifactId("test-library-f1").scope(TEST).withUri())));

    List<BundleDependency> providedClasspath =
        mavenClient.resolveBundleDescriptorDependencies(false, true, getDescriptor("connector-a"));
    checkNoDuplicatedInstances(providedClasspath);
    assertThat(providedClasspath, contains(
                                           aBundleDependency(withArtifactId("library-a").scope(COMPILE).withUri()),
                                           aBundleDependency(withArtifactId("library-b").scope(PROVIDED)),
                                           aBundleDependency(withArtifactId("library-c").scope(COMPILE).withUri()),
                                           aBundleDependency(withArtifactId("library-d").scope(PROVIDED)),
                                           aBundleDependency(withArtifactId("library-d1").scope(PROVIDED))));
  }

  @Test
  public void scopeResolutions() throws IOException {
    repositoryFolder.addArtifacts(artifact("artifact").dependencies(
                                                                    dependency("library-a").scope("compile"),
                                                                    dependency("library-b").scope("provided"),
                                                                    dependency("library-c").scope("runtime"),
                                                                    dependency("library-d").scope("system")
                                                                        .systemPath(temporaryFolder.newFile()),
                                                                    dependency("test-library-a").scope("test"))
        .build());

    BundleDescriptor artifact = getDescriptor("artifact");
    List<BundleDependency> testClasspathPlusProvided =
        mavenClient.resolveBundleDescriptorDependencies(true, true, artifact);
    checkNoDuplicatedInstances(testClasspathPlusProvided);
    assertThat(testClasspathPlusProvided, contains(
                                                   aBundleDependency(withArtifactId("library-a").scope(COMPILE).and()
                                                       .withUri()),
                                                   aBundleDependency(withArtifactId("library-b").and().scope(PROVIDED)),
                                                   aBundleDependency(withArtifactId("library-c").scope(RUNTIME).and()
                                                       .withUri()),
                                                   aBundleDependency(withArtifactId("library-d").scope(SYSTEM).and()
                                                       .withUri()),
                                                   aBundleDependency(withArtifactId("test-library-a").scope(TEST).and()
                                                       .withUri())

    ));

    List<BundleDependency> testClasspath =
        mavenClient.resolveBundleDescriptorDependencies(true, false, artifact);
    checkNoDuplicatedInstances(testClasspath);
    assertThat(testClasspath, contains(
                                       aBundleDependency(withArtifactId("library-a").scope(COMPILE).and().withUri()),
                                       aBundleDependency(withArtifactId("library-c").scope(RUNTIME).and().withUri()),
                                       aBundleDependency(withArtifactId("library-d").scope(SYSTEM).and().withUri()),
                                       aBundleDependency(withArtifactId("test-library-a").scope(TEST).and().withUri())));

    List<BundleDependency> providedClasspath =
        mavenClient.resolveBundleDescriptorDependencies(false, true, artifact);
    checkNoDuplicatedInstances(providedClasspath);
    assertThat(providedClasspath, contains(
                                           aBundleDependency(withArtifactId("library-a").scope(COMPILE).and().withUri()),
                                           aBundleDependency(withArtifactId("library-b").and().scope(PROVIDED)),
                                           aBundleDependency(withArtifactId("library-c").scope(RUNTIME).and().withUri()),
                                           aBundleDependency(withArtifactId("library-d").scope(SYSTEM).and().withUri())));

    List<BundleDependency> runtimeClasspath =
        mavenClient.resolveBundleDescriptorDependencies(false, false, artifact);
    checkNoDuplicatedInstances(runtimeClasspath);
    assertThat(runtimeClasspath, contains(
                                          aBundleDependency(withArtifactId("library-a").scope(COMPILE).and().withUri()),
                                          aBundleDependency(withArtifactId("library-c").scope(RUNTIME).and().withUri()),
                                          aBundleDependency(withArtifactId("library-d").scope(SYSTEM).and().withUri())));
  }

  @Test
  public void resolveArtifactButNotPluginDependencies() {
    repositoryFolder.addArtifacts(artifact("dependency-a").dependencies(
                                                                        dependency("library-a").scope("provided"),
                                                                        dependency("library-b").scope("provided").dependencies(
                                                                                                                               dependency("library-b1")
                                                                                                                                   .scope("provided"),
                                                                                                                               dependency("library-b2")),
                                                                        dependency("connector-c").classifier(MULE_PLUGIN)
                                                                            .dependencies(
                                                                                          dependency("library-c1")),
                                                                        dependency("connector-d").classifier(MULE_PLUGIN),
                                                                        dependency("library-e").scope("test").dependencies(
                                                                                                                           dependency("library-e1")))
        .build());

    List<BundleDependency> classpath =
        mavenClient.resolveBundleDescriptorDependencies(true, true, getDescriptor("dependency-a"));
    checkNoDuplicatedInstances(classpath);
    assertThat(classpath, contains(
                                   aBundleDependency(withArtifactId("library-a").and().scope(PROVIDED)),
                                   aBundleDependency(withArtifactId("library-b").and().scope(PROVIDED)),
                                   aBundleDependency(withArtifactId("library-b2").and().scope(PROVIDED)),
                                   aBundleDependency(withArtifactId("connector-c").scope(COMPILE).classifier(MULE_PLUGIN).and()
                                       .withUri()),
                                   aBundleDependency(withArtifactId("connector-d").scope(COMPILE).classifier(MULE_PLUGIN).and()
                                       .withUri()),
                                   aBundleDependency(withArtifactId("library-e").scope(TEST).and().withUri()),
                                   aBundleDependency(withArtifactId("library-e1").scope(TEST).and().withUri())));
  }

  @Test
  public void resolveArtifactButNotIncludeOptionals() {
    repositoryFolder.addArtifacts(artifact("dependency-a").dependencies(
                                                                        dependency("library-a").optional("true"),
                                                                        dependency("library-b")
                                                                            .dependencies(dependency("library-b1")
                                                                                .optional("true"), dependency("library-b2")),
                                                                        dependency("connector-c").classifier(MULE_PLUGIN),
                                                                        dependency("library-d").scope("test")
                                                                            .dependencies(dependency("library-e1")
                                                                                .optional("true")),
                                                                        dependency("library-e")
                                                                            .dependencies(dependency("library-e1")
                                                                                .dependencies(dependency("library-e2"))))
        .build());

    List<BundleDependency> classpath =
        mavenClient.resolveBundleDescriptorDependencies(true, true, getDescriptor("dependency-a"));
    checkNoDuplicatedInstances(classpath);
    assertThat(classpath, contains(
                                   aBundleDependency(withArtifactId("library-a").scope(COMPILE).and().withUri()),
                                   aBundleDependency(withArtifactId("library-b").scope(COMPILE).and().withUri()),
                                   aBundleDependency(withArtifactId("library-b2").scope(COMPILE).and().withUri()),
                                   aBundleDependency(withArtifactId("connector-c").scope(COMPILE).classifier(MULE_PLUGIN).and()
                                       .withUri()),
                                   aBundleDependency(withArtifactId("library-d").scope(TEST).and().withUri()),
                                   aBundleDependency(withArtifactId("library-e").scope(COMPILE).and().withUri()),
                                   aBundleDependency(withArtifactId("library-e1").scope(COMPILE).and().withUri()),
                                   aBundleDependency(withArtifactId("library-e2").scope(COMPILE).and().withUri())));
  }

  @Test
  public void resolveChainOfPluginDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("artifact").dependencies(
                                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                                        .dependencies(
                                                                                      dependency("connector-b")
                                                                                          .classifier(MULE_PLUGIN).dependencies(
                                                                                                                                dependency("connector-c")
                                                                                                                                    .classifier(MULE_PLUGIN))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, getDescriptor("artifact"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a")),
                                            aBundleDependency(withArtifactId("connector-b")),
                                            aBundleDependency(withArtifactId("connector-c"))));
  }

  @Test
  public void resolveArtifactDependenciesFromPomFile() throws MalformedURLException {
    repositoryFolder.addArtifacts(
                                  artifact("artifact").dependencies(
                                                                    dependency("connector-a").classifier(MULE_PLUGIN))
                                      .build());

    BundleDependency bundleDependency = mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId("artifact")
        .setVersion(VERSION)
        .setType("pom")
        .build());
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(toFile(bundleDependency.getBundleUri().toURL()),
                                                false,
                                                false,
                                                of(repositoryFolder.getRoot()),
                                                empty());
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(aBundleDependency(withArtifactId("connector-a"))));
  }

  @Test
  public void resolvePluginDependenciesDoesNotIncludeTransitiveDependenciesOfOtherPlugins() {
    repositoryFolder.addArtifacts(artifact("connector-a").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("connector-b")
                          .classifier(MULE_PLUGIN)
                          .dependencies(
                                        dependency("library")))
        .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector-a");
    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(true, httpBundleDescriptor);
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-b"))));
  }

  @Test
  public void resolveNonPluginDependenciesWithoutTestDependencies() {
    repositoryFolder.addArtifacts(artifact("connector").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("library-a")
                          .dependencies(dependency("library-a-dep")),
                      dependency("test-library")
                          .scope("test"))
        .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, httpBundleDescriptor);
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a")),
                                            aBundleDependency(withArtifactId("library-a-dep"))));
  }

  @Test
  public void resolveNonPluginDependenciesWithRuntimeDependencies() {
    repositoryFolder.addArtifacts(artifact("connector").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("library-a")
                          .dependencies(dependency("library-a-dep")),
                      dependency("runtime-library")
                          .scope("runtime"))
        .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, httpBundleDescriptor);
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a")),
                                            aBundleDependency(withArtifactId("library-a-dep")),
                                            aBundleDependency(withArtifactId("runtime-library"))));
  }

  @Test
  public void resolveNonPluginDependenciesWithTestDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-a-dep")),
                                                    dependency("test-library")
                                                        .scope("test"))
                                      .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, httpBundleDescriptor);
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a")),
                                            aBundleDependency(withArtifactId("library-a-dep")),
                                            aBundleDependency(withArtifactId("test-library"))));
  }

  @Test
  public void pluginsWithLibraryDependenciesWithTransitiveDependencies() throws Exception {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                                                              dependency("library-a")
                                                                                                                                  .dependencies(
                                                                                                                                                dependency("transitive-library")
                                                                                                                                                    .version("1.0.0"))),
                                                               dependency("connector-b").classifier(MULE_PLUGIN).dependencies(
                                                                                                                              dependency("library-b")
                                                                                                                                  .dependencies(
                                                                                                                                                dependency("transitive-library")
                                                                                                                                                    .version("2.0.0"))))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    checkNoDuplicatedInstances(bundleDependencies);

    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a").and().transitiveDependenciesThat(
                                                                                                                             contains(
                                                                                                                                      aBundleDependency(withArtifactId("library-a")
                                                                                                                                          .and()
                                                                                                                                          .transitiveDependenciesThat(
                                                                                                                                                                      contains(
                                                                                                                                                                               aBundleDependency(withArtifactId("transitive-library")
                                                                                                                                                                                   .and()
                                                                                                                                                                                   .version("1.0.0")))))))),
                                            aBundleDependency(withArtifactId("connector-b").and().transitiveDependenciesThat(
                                                                                                                             contains(
                                                                                                                                      aBundleDependency(withArtifactId("library-b")
                                                                                                                                          .and()
                                                                                                                                          .transitiveDependenciesThat(
                                                                                                                                                                      contains(
                                                                                                                                                                               aBundleDependency(withArtifactId("transitive-library")
                                                                                                                                                                                   .and()
                                                                                                                                                                                   .version("2.0.0"))))))))));
  }

  @Test
  public void resolveTransitiveDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-a-dep")),
                                                    dependency("library-b")
                                                        .dependencies(dependency("library-b-dep1"),
                                                                      dependency("library-b-dep2")
                                                                          .dependencies(dependency("library-b-dep2-dep"))))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, bundleDescriptor);
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a").and().transitiveDependenciesThat(
                                                                                                                           contains(
                                                                                                                                    aBundleDependency(withArtifactId("library-a-dep"))))),
                                            aBundleDependency(withArtifactId("library-a-dep")),
                                            aBundleDependency(withArtifactId("library-b").and().transitiveDependenciesThat(
                                                                                                                           contains(
                                                                                                                                    aBundleDependency(withArtifactId("library-b-dep1")),
                                                                                                                                    aBundleDependency(withArtifactId("library-b-dep2"))))),
                                            aBundleDependency(withArtifactId("library-b-dep1")),
                                            aBundleDependency(withArtifactId("library-b-dep2").and().transitiveDependenciesThat(
                                                                                                                                contains(
                                                                                                                                         aBundleDependency(withArtifactId("library-b-dep2-dep"))))),
                                            aBundleDependency(withArtifactId("library-b-dep2-dep"))));
  }

  @Test
  public void resolveTransitiveDependenciesWithResolvedLibraryFromNonSharedBranch() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-dep").version("1.0")),
                                                    dependency("library-b")
                                                        .dependencies(dependency("library-dep").version("2.0")))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, bundleDescriptor);
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a").and().transitiveDependenciesThat(
                                                                                                                           contains(
                                                                                                                                    aBundleDependency(withArtifactId("library-dep"))))),
                                            aBundleDependency(withArtifactId("library-dep")),
                                            aBundleDependency(withArtifactId("library-b").and().transitiveDependenciesThat(
                                                                                                                           contains(
                                                                                                                                    aBundleDependency(withArtifactId("library-dep")))))));
  }

  @Test
  public void resolveTransitiveDependenciesWithCycles() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-b")),
                                                    dependency("library-b")
                                                        .dependencies(dependency("library-a")))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, bundleDescriptor);
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a").and()
                                                .transitiveDependenciesThat(hasSize(0))),
                                            aBundleDependency(withArtifactId("library-b").and().transitiveDependenciesThat(
                                                                                                                           contains(
                                                                                                                                    aBundleDependency(withArtifactId("library-a")))))));
  }

  @Test
  public void resolveTransitiveDependenciesWithCyclesOnNestedDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-c")
                                                            .dependencies(dependency("library-d"))),
                                                    dependency("library-b")
                                                        .dependencies(dependency("library-d")
                                                            .dependencies(dependency("library-c"))))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, bundleDescriptor);
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a").and().transitiveDependenciesThat(
                                                                                                                           contains(
                                                                                                                                    aBundleDependency(withArtifactId("library-c"))))),
                                            aBundleDependency(withArtifactId("library-c").and()
                                                .transitiveDependenciesThat(hasSize(0))),
                                            aBundleDependency(withArtifactId("library-b").and().transitiveDependenciesThat(
                                                                                                                           contains(
                                                                                                                                    aBundleDependency(withArtifactId("library-d"))))),
                                            aBundleDependency(withArtifactId("library-d").and().transitiveDependenciesThat(
                                                                                                                           contains(
                                                                                                                                    aBundleDependency(withArtifactId("library-c")))))));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("connector").version("1.1").classifier(MULE_PLUGIN).build(),
                                  artifact("connector").version("1.0").classifier(MULE_PLUGIN).build());

    BundleDescriptor connector10 = getDescriptor("connector"); // 1.0
    BundleDescriptor connector11 = getDescriptor("connector", "1.1");

    List<BundleDependency> result = mavenClient.resolvePluginBundleDescriptorsDependencies(asList(connector11, connector10));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(withArtifactId("connector").and().version("1.1"))));

    result = mavenClient.resolvePluginBundleDescriptorsDependencies(asList(connector10, connector11));
    checkNoDuplicatedInstances(result);
    assertThat(result, contains(
                                aBundleDependency(withArtifactId("connector").and().version("1.1"))));
  }

  @Test
  public void shouldAvoidResolvingDependenciesForPluginDiscardedAfterByDependencyResolutionConflictResolver() {
    repositoryFolder.addArtifacts(artifact("connector-a").version("1.1")
        .classifier(MULE_PLUGIN)
        .dependencies(dependency("connector-c")
            .version("2.0")
            .classifier(MULE_PLUGIN))
        .build(),
                                  artifact("connector-b").version("1.0")
                                      .classifier(MULE_PLUGIN)
                                      .dependencies(dependency("connector-c")
                                          .version("2.1")
                                          .classifier(MULE_PLUGIN))
                                      .build());

    // Should not resolve connector-c:2.0 mule-plugin.jar file as the conflict resolver defines the bigger version (semver) as
    // winner.
    repositoryFolder.removeArtifact(artifact("connector-c").version("2.0").classifier(MULE_PLUGIN).type("jar").build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                               asList(getDescriptor("connector-a", "1.1"),
                                                                      getDescriptor("connector-b", "1.0")));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a").and().version("1.1")),
                                            aBundleDependency(withArtifactId("connector-b").and().version("1.0")),
                                            aBundleDependency(withArtifactId("connector-c").and().version("2.1"))));
  }

  @Test
  public void shouldAvoidCollectDomainDependency() throws MalformedURLException {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a"),
                                                    dependency("domain").classifier(MULE_DOMAIN).scope("provided"))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    File artifactFile =
        toFile(mavenClient.resolveBundleDescriptor(bundleDescriptor).getBundleUri().toURL());

    repositoryFolder.removeArtifact(artifact("domain").type("pom").build());
    repositoryFolder.removeArtifact(artifact("domain").classifier(MULE_DOMAIN).type("jar").build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(artifactFile, false, false, of(repositoryFolder.getRoot()), empty());
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a"))));

    bundleDependencies =
        mavenClient.resolveArtifactDependencies(artifactFile, false, true, of(repositoryFolder.getRoot()), empty());
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a").scope(COMPILE).and().withUri()),
                                            aBundleDependency(withArtifactId("domain").and().scope(PROVIDED))));
  }

  @Test
  public void resolutionFailsWithIncompatibleVersions() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("2.0")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.0").classifier(MULE_PLUGIN).build());

    expectedException.expect(RuntimeException.class);
    mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                           asList(getDescriptor("connector-a"),
                                                                  getDescriptor("connector-b")));
  }

  @Test
  public void resolutionFailsWithIncompatibleVersions_1() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("2.0")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.0").classifier(MULE_PLUGIN).build());

    expectedException.expect(RuntimeException.class);
    mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                           asList(getDescriptor("connector-b"),
                                                                  getDescriptor("connector-a")));
  }

  @Test
  public void resolutionFailsWithIncompatibleVersions_2() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .classifier(MULE_PLUGIN)
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-c")
                                                                                                                     .version("2.0")
                                                                                                                     .classifier(MULE_PLUGIN)))
                                      .build(),
                                  artifact("connector-c").version("1.0").classifier(MULE_PLUGIN).build());

    expectedException.expect(RuntimeException.class);
    mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                           asList(getDescriptor("connector-a"),
                                                                  getDescriptor("connector-c")));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPluginsConsideringDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("1.1")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.0").classifier(MULE_PLUGIN).build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                               asList(getDescriptor("connector-b"),
                                                                      getDescriptor("connector-a")));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-b").and().version("1.1"))));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPluginsConsideringDependencies_1() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("1.0")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.1").classifier(MULE_PLUGIN).build());

    BundleDescriptor descriptorB = getDescriptor("connector-b", "1.1");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(asList(descriptorB, getDescriptor("connector-a")));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-b").and().version("1.1")),
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION))));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPluginsConsideringTransitiveDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .classifier(MULE_PLUGIN)
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-c")
                                                                                                                     .version("1.1")
                                                                                                                     .classifier(MULE_PLUGIN)))
                                      .build(),
                                  artifact("connector-c").version("1.0").classifier(MULE_PLUGIN).build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                               asList(getDescriptor("connector-a"),
                                                                      getDescriptor("connector-c")));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-b").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version("1.1"))));
  }

  @Test
  public void fromPluginsOnlyNestedPluginsShouldBeIncludedInDependencyGraph() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("connector-b")
                                                        .classifier(MULE_PLUGIN)
                                                        .dependencies(
                                                                      dependency("connector-c")
                                                                          .classifier(MULE_PLUGIN),
                                                                      dependency("third-party").version("1.0")),
                                                    dependency("internal-library").version("1.0")
                                                        .dependencies(
                                                                      dependency("third-party").version("1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-b").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("internal-library").and().version("1.0")),
                                            aBundleDependency(withArtifactId("third-party").and().version("1.0"))));
  }

  @Test
  public void pluginsDoNotParticipateOnMavenVersionResolution() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("connector-b")
                                                        .classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("third-party").version("2.0")),
                                                    dependency("internal-library").version("1.0")
                                                        .dependencies(dependency("third-party").version("1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-b").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("internal-library").and().version("1.0")),
                                            aBundleDependency(withArtifactId("third-party").and().version("1.0"))));
  }

  @Test
  public void pluginDependsOnJarThatDependsOnPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("connector-b").classifier(MULE_PLUGIN)
                                                            .dependencies(dependency("library-d").version("1.1"))),
                                                    dependency("library-b").version("1.0")
                                                        .dependencies(dependency("library-c").version("1.0")
                                                            .dependencies(dependency("library-d").version("1.0"))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-a").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-b").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("library-b").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("library-c").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("library-d").and().version(VERSION))));
  }

  @Test
  public void resolveBundleDependenciesFromJarFileShouldStillUsePomArtifact() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").version(VERSION).classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("library-a"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a", VERSION, "jar", MULE_PLUGIN));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies,
               contains(
                        aBundleDependency(withArtifactId("library-a").and().version(VERSION))));
  }

  @Test
  public void applicationThatDependsOnPluginThatDependsOnJarThatDependsOnPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(
                                                                      dependency("library-a").version("1.1")
                                                                          .dependencies(dependency("connector-b")
                                                                              .classifier(MULE_PLUGIN)
                                                                              .dependencies(
                                                                                            dependency("library-d").version("1.1")
                                                                                                .dependencies(
                                                                                                              dependency("library-e")
                                                                                                                  .version("1.1")
                                                                                                                  .dependencies(
                                                                                                                                dependency("connector-d")
                                                                                                                                    .classifier(MULE_PLUGIN))),
                                                                                            dependency("connector-c")
                                                                                                .classifier(MULE_PLUGIN)))),
                                                    dependency("library-b").version("1.0")
                                                        .dependencies(
                                                                      dependency("library-c").version("1.0")
                                                                          .dependencies(
                                                                                        dependency("library-d").version("1.0"))),
                                                    dependency("library-a").version("1.0"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-b").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-d").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("library-b").and().version("1.0")),
                                            aBundleDependency(withArtifactId("library-c").and().version("1.0")),
                                            aBundleDependency(withArtifactId("library-d").and().version("1.0")),
                                            aBundleDependency(withArtifactId("library-a").and().version("1.0"))));
  }

  @Test
  public void appWithPluginsWithSimpleCyclicDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)),
                                                    dependency("connector-c").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-a").classifier(MULE_PLUGIN)))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version(VERSION))));

  }

  @Test
  public void resolvePluginRootNodeShouldNotRemoveDirectDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a")
                                      .dependencies(
                                                    dependency("connector-b").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)),
                                                    dependency("library-1"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-b").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("library-1").and().version(VERSION))));
  }

  @Test
  public void appWithPluginsWithComplexCyclicDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .dependencies(
                                                                          dependency("connector-d").classifier(MULE_PLUGIN)
                                                                              .dependencies(dependency("connector-a")
                                                                                  .classifier(MULE_PLUGIN)))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-d").and().version(VERSION))));

  }

  @Test
  public void appWithMultipleVersionsOfPlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")),
                                                    dependency("connector-b").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-b").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version("1.1.0"))));
  }

  @Test
  public void appWithIncompatibleVersionsOfPlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")),
                                                    dependency("connector-b").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("2.1.0")))
                                      .build());

    expectedException.expect(IncompatibleMulePluginVersionResolutionException.class);
    expectedException
        .expectMessage(containsString("Incompatible Mule plugins versions among: org.mule:connector-c:jar:mule-plugin:2.1.0, org.mule:connector-c:jar:mule-plugin:1.0.2"));

    mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-1").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version("1.1.0"))));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependenciesDifferentOrder() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0")),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-1").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version("1.1.0")),
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION))));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependenciesTwoLevels() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.3.0"))),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.1.0"))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-1").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version("1.1.0")),
                                            aBundleDependency(withArtifactId("connector-d").and().version("2.3.0")),
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION))));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependenciesTwoLevelsAndRootDefinitionOrSameConnector() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.1.0"))),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.3.0"))),
                                                    dependency("connector-d").classifier(MULE_PLUGIN).version("2.2.0"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("library-1").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-c").and().version("1.1.0")),
                                            aBundleDependency(withArtifactId("connector-a").and().version(VERSION)),
                                            aBundleDependency(withArtifactId("connector-d").and().version("2.2.0"))));
  }

  @Test
  public void appSamePluginDefinedBothTimesMavenTakesLastDefinitionOfDependency() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-1").classifier(MULE_PLUGIN)
                                                        .version("1.1.0"),
                                                    dependency("connector-1").classifier(MULE_PLUGIN)
                                                        .version("1.0.2"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("connector-1").and().version("1.0.2"))));
  }

  @Test
  public void transitiveApiDependenciesDontOverrideAppDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("api-a").classifier(RAML).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("fragment")
                                                                                     .classifier(RAML_FRAGMENT)
                                                                                     .version("1.0.0"),
                                                                                 dependency("library").version("2.0.0")),
                                                               dependency("library").version("1.0.0"))
                                      .build());

    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, hasSize(3));

    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("api-a").and().version("1.0.0")),
                                            aBundleDependency(withArtifactId("fragment").and().version("1.0.0")),
                                            aBundleDependency(withArtifactId("library").and().version("1.0.0"))));
  }

  @Test
  public void ifPluginWinsAlsoItsDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("plugin").version("1.0.0").classifier(MULE_PLUGIN)
                                                                   .dependencies(
                                                                                 dependency("library-a").version("1.1.0"),
                                                                                 dependency("library-b").version("1.1.0")),
                                                               dependency("synthetic").version("1.0.0").dependencies(
                                                                                                                     dependency("plugin")
                                                                                                                         .version("1.1.0")
                                                                                                                         .classifier(MULE_PLUGIN)
                                                                                                                         .dependencies(
                                                                                                                                       dependency("library-a")
                                                                                                                                           .version("1.0.0"))))
                                      .build());
    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("synthetic").and().version("1.0.0")),
                                            aBundleDependency(withArtifactId("plugin").version("1.1.0").and()
                                                .transitiveDependenciesThat(
                                                                            contains(
                                                                                     aBundleDependency(withArtifactId("library-a")
                                                                                         .and().version("1.0.0")))))));
  }

  @Test
  public void sameDependenciesInDifferentPlugins() {
    final String version = "1.0.0";
    ArtifactCreator.DependencyBuilder libraryBDep = dependency("library-b").version(version);
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("plugin-a").classifier(MULE_PLUGIN).version(version)
                                                                   .dependencies(
                                                                                 dependency("library-a").version(version)
                                                                                     .dependencies(
                                                                                                   libraryBDep)),
                                                               dependency("plugin-b").classifier(MULE_PLUGIN).version(version)
                                                                   .dependencies(
                                                                                 dependency("library-a").version(version)
                                                                                     .dependencies(
                                                                                                   libraryBDep)
                                                                                     .exclusions(exclusion("library-b"))))
                                      .build());

    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("plugin-a").and()
                                                .transitiveDependenciesThat(contains(aBundleDependency(withArtifactId("library-a")
                                                    .and().transitiveDependenciesThat(hasSize(1)))))),
                                            aBundleDependency(withArtifactId("plugin-b").and()
                                                .transitiveDependenciesThat(contains(aBundleDependency(withArtifactId("library-a")
                                                    .and().transitiveDependenciesThat(hasSize(0))))))));
  }

  @Test
  public void sameDependenciesDifferentPluginsWithOtherWinner() {
    ArtifactCreator.DependencyBuilder artifactD = dependency("artifact-d").version("1.0.0").dependencies(
                                                                                                         dependency("artifact-a")
                                                                                                             .version("1.0.0"));

    ArtifactCreator.DependencyBuilder artifactDExcludingA = dependency("artifact-d").version("1.0.0").dependencies(
                                                                                                                   dependency("artifact-a")
                                                                                                                       .version("1.0.0"))
        .exclusions(exclusion("artifact-a"));

    ArtifactCreator.DependencyBuilder artifactC = dependency("artifact-c").version("1.0.0").dependencies(
                                                                                                         artifactDExcludingA);
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("plugin-a").classifier(MULE_PLUGIN).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("plugin-b").classifier(MULE_PLUGIN)
                                                                                     .version("1.0.0").dependencies(
                                                                                                                    artifactC),
                                                                                 artifactD),
                                                               dependency("plugin-c").classifier(MULE_PLUGIN).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("plugin-b").classifier(MULE_PLUGIN)
                                                                                     .version("1.1.0").dependencies(
                                                                                                                    artifactC)))
                                      .build());

    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("plugin-a").and().transitiveDependenciesThat(
                                                                                                                          contains(
                                                                                                                                   aBundleDependency(withArtifactId(null)),
                                                                                                                                   aBundleDependency(withArtifactId("artifact-d")
                                                                                                                                       .and()
                                                                                                                                       .transitiveDependenciesThat(
                                                                                                                                                                   contains(
                                                                                                                                                                            aBundleDependency(withArtifactId("artifact-a")))))))),
                                            aBundleDependency(withArtifactId("plugin-c")),
                                            aBundleDependency(withArtifactId("plugin-b").version("1.1.0").and()
                                                .transitiveDependenciesThat(
                                                                            contains(
                                                                                     aBundleDependency(withArtifactId("artifact-c")
                                                                                         .and()
                                                                                         .transitiveDependenciesThat(contains(
                                                                                                                              aBundleDependency(withArtifactId("artifact-d")
                                                                                                                                  .and()
                                                                                                                                  .transitiveDependenciesThat(hasSize(0)))))))))));
  }


  @Test
  public void sameApiDependencyInChainForRAML() {
    assertSameApiDependencyInChain(RAML, RAML);
    assertSameApiDependencyInChain(RAML, RAML_FRAGMENT);
  }

  @Test
  public void sameApiDependencyInChainForOAS() {
    assertSameApiDependencyInChain(OAS, OAS);
  }

  @Test
  public void sameApiDependencyInChainForWSDL() {
    assertSameApiDependencyInChain(WSDL, WSDL);
  }

  @Test
  public void multipleApiLevels() {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("api-a").version("1.0.0").classifier(RAML)
                                                                   .dependencies(
                                                                                 dependency("fragment").classifier(RAML_FRAGMENT)
                                                                                     .version("1.0.0"),
                                                                                 dependency("api-b").classifier(RAML)
                                                                                     .version("1.0.0").dependencies(
                                                                                                                    dependency("fragment")
                                                                                                                        .classifier(RAML_FRAGMENT)
                                                                                                                        .version("1.1.0"))))
                                      .build()


    );
    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("api-a").and().version("1.0.0")),
                                            aBundleDependency(withArtifactId("fragment").and().version("1.0.0")),
                                            aBundleDependency(withArtifactId("api-b").and().version("1.0.0")),
                                            aBundleDependency(withArtifactId("fragment").and().version("1.1.0"))));

  }

  private void assertSameApiDependencyInChain(String apiClassifier, String dependencyClassifier) {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("api-a").classifier(apiClassifier).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("fragment")
                                                                                     .classifier(dependencyClassifier)
                                                                                     .version("1.0.0")),
                                                               dependency("api-b").classifier(apiClassifier).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("fragment")
                                                                                     .classifier(dependencyClassifier)
                                                                                     .version("2.0.0")))
                                      .build());

    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, contains(
                                            aBundleDependency(withArtifactId("api-a").and().version("1.0.0")),
                                            aBundleDependency(withArtifactId("fragment").and().version("1.0.0")),
                                            aBundleDependency(withArtifactId("api-b").and().version("1.0.0")),
                                            aBundleDependency(withArtifactId("fragment").and().version("2.0.0"))));
  }

}
