/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test.dependency;

import static java.lang.String.format;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.BundleScope;

import java.util.function.BiPredicate;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeMatcher;

public class BundleDependencyMatcher extends TypeSafeMatcher<BundleDependency> {

  private String groupId;
  private String artifactId;
  private String version;
  private String classifier;
  private BundleScope scope;
  private String type;
  private Boolean hasUri;

  private Matcher transitiveDependenciesMatcher;

  private StringBuilder failure = new StringBuilder();

  public static BundleDependencyMatcher aBundleDependency(BundleDependencyMatcherBuilder bundleDependencyMatcherBuilder) {
    return bundleDependencyMatcherBuilder.dependencyMatcher;
  }

  private boolean check(Object expected, Object actual, String attribute) {
    return check(expected, actual, attribute, Object::equals);
  }

  private boolean check(Object expected, Object actual, String attribute, BiPredicate<Object, Object> predicate) {
    boolean result = predicate.test(expected, actual);
    if (!result) {
      failure.append(format("a different %s with value: %s, expected: %s, ", attribute, actual, expected));
    }
    return result;
  }

  @Override
  protected boolean matchesSafely(BundleDependency bundleDependency) {
    final BundleDescriptor bundleDescriptor = bundleDependency.getDescriptor();
    boolean matches = true;

    if (groupId != null) {
      matches &= check(groupId, bundleDescriptor.getGroupId(), "groupId");
    }
    if (artifactId != null) {
      matches &= check(artifactId, bundleDescriptor.getArtifactId(), "artifactId");
    }
    if (version != null) {
      matches &= check(version, bundleDescriptor.getVersion(), "version");
    }
    if (classifier != null) {
      matches &= check(classifier, bundleDescriptor.getClassifier().orElse(null), "classifier");;
    }
    if (type != null) {
      matches &= check(type, bundleDescriptor.getType(), "type");
    }


    if (hasUri != null) {
      if (hasUri) {
        matches &= check(null, bundleDependency.getBundleUri(), "bundleUri", (e, a) -> a != e);
      } else {
        matches &= check(null, bundleDependency.getBundleUri(), "bundleUri", (e, a) -> a == e);
      }
    }

    if (transitiveDependenciesMatcher != null) {
      matches &= check(null, bundleDependency.getTransitiveDependencies(), "transitiveDependencies",
                       (e, a) -> (transitiveDependenciesMatcher.matches(a)));
    }

    if (scope != null) {
      matches &= check(scope, bundleDependency.getScope(), "scope");
    }
    return matches;
  }

  @Override
  public void describeTo(Description description) {
    description.appendText(
                           format("a BundleDependency with " +
                               "artifactId: %s, " +
                               "groupId: %s, " +
                               "version: %s, " +
                               "classifier: %s, " +
                               "type: %s, " +
                               "scope: %s, " +
                               "with%s uri",
                                  artifactId == null ? "ANY" : artifactId,
                                  groupId == null ? "ANY" : groupId,
                                  version == null ? "ANY" : version,
                                  classifier == null ? "ANY" : classifier,
                                  type == null ? "ANY" : type,
                                  scope,
                                  hasUri == null ? " ANY" : (hasUri ? "" : "out")));

  }

  @Override
  protected void describeMismatchSafely(BundleDependency item, Description mismatchDescription) {
    String failureString = failure.toString();
    if (!failureString.isEmpty()) {
      mismatchDescription.appendText("had ");
      mismatchDescription.appendText(failureString);
    } else {
      super.describeMismatchSafely(item, mismatchDescription);
    }
  }

  public static class BundleDependencyMatcherBuilder {

    private BundleDependencyMatcher dependencyMatcher;

    private BundleDependencyMatcherBuilder() {
      dependencyMatcher = new BundleDependencyMatcher();
    }

    public static BundleDependencyMatcherBuilder withArtifactId(String expectedArtifactId) {
      BundleDependencyMatcherBuilder builder = new BundleDependencyMatcherBuilder();
      builder.dependencyMatcher.artifactId = expectedArtifactId;
      return builder;
    }

    public BundleDependencyMatcherBuilder groupId(String groupId) {
      dependencyMatcher.groupId = groupId;
      return this;
    }

    public BundleDependencyMatcherBuilder version(String version) {
      dependencyMatcher.version = version;
      return this;
    }

    public BundleDependencyMatcherBuilder classifier(String classifier) {
      dependencyMatcher.classifier = classifier;
      return this;
    }

    public BundleDependencyMatcherBuilder scope(BundleScope bundleScope) {
      dependencyMatcher.scope = bundleScope;
      return this;
    }

    public BundleDependencyMatcherBuilder type(String type) {
      dependencyMatcher.type = type;
      return this;
    }

    public BundleDependencyMatcherBuilder withUri() {
      dependencyMatcher.hasUri = true;
      return this;
    }

    public BundleDependencyMatcherBuilder withoutUri() {
      dependencyMatcher.hasUri = false;
      return this;
    }

    public BundleDependencyMatcherBuilder transitiveDependenciesThat(Matcher transitiveDependenciesMatcher) {
      dependencyMatcher.transitiveDependenciesMatcher = transitiveDependenciesMatcher;
      return this;
    }

    public BundleDependencyMatcherBuilder and() {
      return this;
    }
  }

}
