/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.hasItem;
import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.BundleScope.COMPILE;
import static org.mule.maven.client.api.model.BundleScope.PROVIDED;
import static org.mule.maven.client.api.model.BundleScope.RUNTIME;
import static org.mule.maven.client.api.model.BundleScope.SYSTEM;
import static org.mule.maven.client.api.model.BundleScope.TEST;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.exception.IncompatibleMulePluginVersionResolutionException;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.BundleScope;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.test.ArtifactCreator;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Arrays;
import java.util.List;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.junit.Test;

public abstract class AbstractOnlineMavenClientTestCase extends AbstractMavenClientTestCase {

  private static final String MULE_PLUGIN = "mule-plugin";
  private static final String MULE_DOMAIN = "mule-domain";

  @Override
  protected void beforeTest() throws Exception {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration());
  }

  protected abstract MavenConfiguration createMavenConfiguration() throws Exception;

  @Test
  public void resolveExistentBundleDescriptor() {
    String artifactId = "artifact";
    String type = "jar";
    repositoryFolder.addArtifacts(artifact(artifactId).classifier(MULE_PLUGIN).type(type).build());
    BundleDependency coreBundle = mavenClient.resolveBundleDescriptor(getDescriptor(artifactId, VERSION, type, MULE_PLUGIN));

    assertThat(coreBundle.getDescriptor().getGroupId(), is(GROUP_ID));
    assertThat(coreBundle.getDescriptor().getArtifactId(), is(artifactId));
    assertThat(coreBundle.getDescriptor().getVersion(), is(VERSION));
    assertThat(coreBundle.getDescriptor().getType(), is(type));
    assertThat(coreBundle.getDescriptor().getClassifier().isPresent(), is(true));
    assertThat(coreBundle.getDescriptor().getClassifier().get(), is(MULE_PLUGIN));
  }

  @Test
  public void resolveNotExistentBundleDescriptor() {
    expectedException.expect(BundleDependenciesResolutionException.class);
    mavenClient.resolveBundleDescriptor(getDescriptor("artifact-does-not-exist"));
  }

  @Test
  public void resolvePluginDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("mule-http-connector").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("mule-sockets-connector")
                                                        .classifier(MULE_PLUGIN))
                                      .build());
    BundleDescriptor socketsBundleDescriptor = getDescriptor("mule-sockets-connector");
    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(ImmutableList.<BundleDescriptor>builder()
            .add(httpBundleDescriptor)
            .build());
    List<BundleDescriptor> result = bundleDependencies.stream().map(BundleDependency::getDescriptor)
        .map(bundleDescriptor -> new BundleDescriptor.Builder()
            .setGroupId(bundleDescriptor.getGroupId())
            .setArtifactId(bundleDescriptor.getArtifactId())
            .setVersion(VERSION)
            .setType(bundleDescriptor.getType())
            .setClassifier(bundleDescriptor.getClassifier().orElse(null))
            .build())
        .collect(toList());
    assertThat(result.size(), is(2));
    assertThat(result, contains(httpBundleDescriptor, socketsBundleDescriptor));
  }

  @Test
  public void mulePluginConflictResolutionUseSemVer() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("mule-sockets-connector").version("1.1.5")
                                                            .classifier(MULE_PLUGIN),
                                                                      dependency("library-a")),
                                                    dependency("mule-sockets-connector").classifier(MULE_PLUGIN).version("1.1.0"))
                                      .build());

    BundleDescriptor socketsBundleDescriptor = getDescriptor("mule-sockets-connector", "1.1.5");
    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, true, getDescriptor("app"));

    assertThat(result.size(), is(2));
    BundleDependency httpBundleDependency = result.get(0);
    assertThat(httpBundleDependency.getDescriptor(), equalTo(httpBundleDescriptor));
    assertThat(httpBundleDependency.getScope(), equalTo(COMPILE));
    assertThat(httpBundleDependency.getBundleUri(), not(nullValue()));
    assertThat(httpBundleDependency.getTransitiveDependencies().size(), is(2));

    BundleDependency socketsBundleDependency = httpBundleDependency.getTransitiveDependencies().get(0);
    assertThat(socketsBundleDependency.getDescriptor(), equalTo(socketsBundleDescriptor));
    assertThat(socketsBundleDependency.getScope(), equalTo(COMPILE));
    assertThat(socketsBundleDependency.getBundleUri(), not(nullValue()));

    socketsBundleDependency = result.get(1);
    assertThat(socketsBundleDependency.getDescriptor(), equalTo(socketsBundleDescriptor));
    assertThat(socketsBundleDependency.getScope(), equalTo(COMPILE));
    assertThat(socketsBundleDependency.getBundleUri(), not(nullValue()));
  }

  @Test
  public void compileTransitiveDependencyFromTestIsWinner() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("artifact-a")
                                                        .dependencies(dependency("artifact-b")
                                                            .dependencies(dependency("artifact-c").version("1.0"))),
                                                    dependency("test-artifact-a").scope("test")
                                                        .dependencies(dependency("artifact-c").version("2.0")))
                                      .build());

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));

    assertThat(result.size(), is(3));

    List<String> artifactIds = result.stream()
        .map(dep -> dep.getDescriptor().getArtifactId() + ":" + dep.getDescriptor().getVersion()).collect(toList());
    assertThat(artifactIds, hasItems("artifact-a:1.0", "artifact-b:1.0", "artifact-c:2.0"));

    BundleDependency artifactA = result.get(0);
    assertThat(artifactA.getDescriptor().getArtifactId(), equalTo("artifact-a"));
    assertThat(artifactA.getScope(), equalTo(COMPILE));
    assertThat(artifactA.getBundleUri(), not(nullValue()));
    assertThat(artifactA.getTransitiveDependencies().size(), is(1));

    BundleDependency transitiveArtifactB = artifactA.getTransitiveDependencies().get(0);
    assertThat(transitiveArtifactB.getDescriptor().getArtifactId(), equalTo("artifact-b"));
    assertThat(transitiveArtifactB.getScope(), equalTo(COMPILE));
    assertThat(transitiveArtifactB.getBundleUri(), not(nullValue()));
    assertThat(transitiveArtifactB.getTransitiveDependencies().size(), is(1));

    BundleDependency transitiveArtifactC = transitiveArtifactB.getTransitiveDependencies().get(0);
    assertThat(transitiveArtifactC.getDescriptor().getArtifactId(), equalTo("artifact-c"));
    // should be 2.0 artifact-c version as the winner comes from the test dependency
    assertThat(transitiveArtifactC.getDescriptor().getVersion(), equalTo("2.0"));
    assertThat(transitiveArtifactC.getScope(), equalTo(COMPILE));
    assertThat(transitiveArtifactC.getBundleUri(), not(nullValue()));
    assertThat(transitiveArtifactC.getTransitiveDependencies().isEmpty(), is(true));

    BundleDependency artifactB = result.get(1);
    assertThat(artifactB.getDescriptor().getArtifactId(), equalTo("artifact-b"));
    assertThat(artifactB.getScope(), equalTo(COMPILE));
    assertThat(artifactB.getBundleUri(), not(nullValue()));
    assertThat(artifactB.getTransitiveDependencies().size(), is(1));

    transitiveArtifactC = artifactB.getTransitiveDependencies().get(0);
    assertThat(transitiveArtifactC.getDescriptor().getArtifactId(), equalTo("artifact-c"));
    // should be 2.0 artifact-c version as the winner comes from the test dependency
    assertThat(transitiveArtifactC.getDescriptor().getVersion(), equalTo("2.0"));
    assertThat(transitiveArtifactC.getScope(), equalTo(COMPILE));
    assertThat(transitiveArtifactC.getBundleUri(), not(nullValue()));
    assertThat(transitiveArtifactC.getTransitiveDependencies().isEmpty(), is(true));

    BundleDependency artifactC = result.get(2);
    assertThat(artifactC.getDescriptor().getArtifactId(), equalTo("artifact-c"));
    assertThat(artifactC.getDescriptor().getVersion(), equalTo("2.0"));
    assertThat(artifactC.getScope(), equalTo(COMPILE));
    assertThat(artifactC.getBundleUri(), not(nullValue()));
    assertThat(artifactC.getTransitiveDependencies().isEmpty(), is(true));
  }

  @Test
  public void providedDependencyScopeShouldOverrideTransitiveDependenciesScopes() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("mule-sockets-connector")
                                                            .classifier(MULE_PLUGIN),
                                                                      dependency("library-a")),
                                                    dependency("mule-sockets-connector").classifier(MULE_PLUGIN)
                                                        .scope("provided"),
                                                    dependency("library-a").scope("provided"))
                                      .build());

    BundleDescriptor socketsBundleDescriptor = getDescriptor("mule-sockets-connector");
    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, true, getDescriptor("app"));

    assertThat(result.size(), is(3));
    BundleDependency httpBundleDependency = result.get(0);
    assertThat(httpBundleDependency.getDescriptor(), equalTo(httpBundleDescriptor));
    assertThat(httpBundleDependency.getScope(), equalTo(COMPILE));
    assertThat(httpBundleDependency.getBundleUri(), not(nullValue()));
    assertThat(httpBundleDependency.getTransitiveDependencies().size(), is(1));

    // Scopes are propagated inside mule-plugin resolution, therefore the library-a dependency remains as compile
    BundleDependency libraryAFromHttpPlugin = httpBundleDependency.getTransitiveDependencies().get(0);
    assertThat(libraryAFromHttpPlugin.getDescriptor().getArtifactId(), equalTo("library-a"));
    assertThat(libraryAFromHttpPlugin.getScope(), equalTo(COMPILE));
    assertThat(libraryAFromHttpPlugin.getBundleUri(), not(nullValue()));

    BundleDependency socketsBundleDependency = result.get(1);
    assertThat(socketsBundleDependency.getDescriptor(), equalTo(socketsBundleDescriptor));
    assertThat(socketsBundleDependency.getScope(), equalTo(PROVIDED));
    assertThat(socketsBundleDependency.getBundleUri(), nullValue());

    BundleDependency libraryAFromApplication = result.get(2);
    assertThat(libraryAFromApplication.getDescriptor().getArtifactId(), equalTo("library-a"));
    assertThat(libraryAFromApplication.getScope(), equalTo(PROVIDED));
    assertThat(libraryAFromApplication.getBundleUri(), nullValue());
  }

  @Test
  public void providedDependencyScopeOverrideNotIncludedInTransitiveDependency() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("mule-sockets-connector")
                                                            .classifier(MULE_PLUGIN),
                                                                      dependency("library-a")),
                                                    dependency("mule-sockets-connector").classifier(MULE_PLUGIN)
                                                        .scope("provided"),
                                                    dependency("library-a").scope("provided"))
                                      .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));

    assertThat(result.size(), is(1));
    BundleDependency httpBundleDependency = result.get(0);
    assertThat(httpBundleDependency.getDescriptor(), equalTo(httpBundleDescriptor));
    assertThat(httpBundleDependency.getScope(), equalTo(COMPILE));
    assertThat(httpBundleDependency.getBundleUri(), not(nullValue()));
    assertThat(httpBundleDependency.getTransitiveDependencies().size(), is(1));

    // Scopes are propagated inside mule-plugin resolution, therefore the library-a dependency remains as compile
    BundleDependency libraryAFromHttpPlugin = httpBundleDependency.getTransitiveDependencies().get(0);
    assertThat(libraryAFromHttpPlugin.getDescriptor().getArtifactId(), equalTo("library-a"));
    assertThat(libraryAFromHttpPlugin.getScope(), equalTo(COMPILE));
    assertThat(libraryAFromHttpPlugin.getBundleUri(), not(nullValue()));
  }

  @Test
  public void exclusionOfMulePlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("mule-sockets-connector")
                                                            .classifier(MULE_PLUGIN))
                                                        .exclusions(exclusion("mule-sockets-connector")))
                                      .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));

    assertThat(result.size(), is(1));
    BundleDependency httpBundleDependency = result.get(0);
    assertThat(httpBundleDependency.getDescriptor(), equalTo(httpBundleDescriptor));
    assertThat(httpBundleDependency.getScope(), equalTo(COMPILE));
    assertThat(httpBundleDependency.getBundleUri(), not(nullValue()));
    assertThat(httpBundleDependency.getTransitiveDependencies().size(), is(0));
  }

  @Test
  public void exclusionOfArtifactsFromMulePlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("mule-http-connector").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("library-a"),
                                                                      dependency("library-b"))
                                                        .exclusions(exclusion("library-a")))
                                      .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("mule-http-connector");

    List<BundleDependency> result =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("app"));

    assertThat(result.size(), is(1));
    BundleDependency httpBundleDependency = result.get(0);
    assertThat(httpBundleDependency.getDescriptor(), equalTo(httpBundleDescriptor));
    assertThat(httpBundleDependency.getScope(), equalTo(COMPILE));
    assertThat(httpBundleDependency.getBundleUri(), not(nullValue()));
    // library-a is excluded from the mule-plugin dependencies resolution
    assertThat(httpBundleDependency.getTransitiveDependencies().size(), is(1));
    assertThat(httpBundleDependency.getTransitiveDependencies().get(0).getDescriptor().getArtifactId(), is("library-b"));
  }

  @Test
  public void resolveZipDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("zip-dependency-1").type("zip")
                                                        .dependencies(dependency("zip-dependency-2").type("zip")
                                                            .dependencies(dependency("library-jar-1")).type("jar")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, hasItems("zip-dependency-1", "zip-dependency-2", "library-jar-1"));
  }

  @Test
  public void resolvePluginDependenciesFromNonPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("library")
                                      .dependencies(
                                                    dependency("connector").classifier(MULE_PLUGIN))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, getDescriptor("library"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, hasItem("connector"));
  }

  @Test
  public void resolveArtifactFromListOfDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("library-a")
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-b")
                                                                                                                     .classifier(MULE_PLUGIN)
                                                                                                                     .dependencies(
                                                                                                                                   dependency("library-b"))))
                                      .build());

    BundleDescriptor descriptor = getDescriptor("connector-a");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(Lists.newArrayList(descriptor),
                                                of(mavenClient.getMavenConfiguration().getLocalMavenRepositoryLocation()),
                                                empty());
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, contains("connector-a", "library-a", "connector-b"));
    assertThat(result, not(hasItem("library-b")));
  }

  @Test
  public void resolvePluginWithDependencyThatDependsOnPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("library-a")
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-b")
                                                                                                                     .classifier(MULE_PLUGIN)
                                                                                                                     .dependencies(
                                                                                                                                   dependency("library-b"))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, getDescriptor("connector-a"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, contains("library-a", "connector-b"));
    assertThat(result, not(hasItem("library-b")));
  }

  @Test
  public void resolvePluginWithDependencyThatDependsOnPluginProvided() {
    repositoryFolder.addArtifacts(artifact("connector-a").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("library-a"),
                      dependency("library-b").scope("provided"),
                      dependency("library-c").dependencies(dependency("library-c1").scope("provided")),
                      dependency("library-d").scope("provided").dependencies(dependency("library-d1")),
                      dependency("library-e").scope("test").dependencies(dependency("test-library-e1").scope("test")),
                      dependency("library-f").scope("test").dependencies(dependency("test-library-f1")))
        .build());

    List<BundleDependency> runtimeClasspath =
        mavenClient.resolveBundleDescriptorDependencies(false, false, getDescriptor("connector-a"));
    assertThat(runtimeClasspath, hasSize(2));
    assertThat(runtimeClasspath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(runtimeClasspath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-c", COMPILE, true));

    List<BundleDependency> testClasspath =
        mavenClient.resolveBundleDescriptorDependencies(true, false, getDescriptor("connector-a"));
    assertThat(testClasspath, hasSize(5));
    assertThat(testClasspath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(testClasspath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-c", COMPILE, true));
    assertThat(testClasspath.get(2), BundleDependencyMatcher.bundleDependencyMatcher("library-e", TEST, true));
    assertThat(testClasspath.get(3), BundleDependencyMatcher.bundleDependencyMatcher("library-f", TEST, true));
    assertThat(testClasspath.get(4), BundleDependencyMatcher.bundleDependencyMatcher("test-library-f1", TEST, true));

    List<BundleDependency> providedClasspath =
        mavenClient.resolveBundleDescriptorDependencies(false, true, getDescriptor("connector-a"));
    assertThat(providedClasspath, hasSize(5));
    assertThat(providedClasspath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(providedClasspath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-b", PROVIDED, false));
    assertThat(providedClasspath.get(2), BundleDependencyMatcher.bundleDependencyMatcher("library-c", COMPILE, true));
    assertThat(providedClasspath.get(3), BundleDependencyMatcher.bundleDependencyMatcher("library-d", PROVIDED, false));
    assertThat(providedClasspath.get(4), BundleDependencyMatcher.bundleDependencyMatcher("library-d1", PROVIDED, false));
  }

  @Test
  public void scopeResolutions() throws IOException {
    repositoryFolder.addArtifacts(artifact("artifact").dependencies(
                                                                    dependency("library-a").scope("compile"),
                                                                    dependency("library-b").scope("provided"),
                                                                    dependency("library-c").scope("runtime"),
                                                                    dependency("library-d").scope("system")
                                                                        .systemPath(temporaryFolder.newFile()),
                                                                    dependency("test-library-a").scope("test"))
        .build());

    BundleDescriptor artifact = getDescriptor("artifact");
    List<BundleDependency> testClasspathPlusProvided =
        mavenClient.resolveBundleDescriptorDependencies(true, true, artifact);

    assertThat(testClasspathPlusProvided, hasSize(5));

    assertThat(testClasspathPlusProvided.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(testClasspathPlusProvided.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-b", PROVIDED, false));
    assertThat(testClasspathPlusProvided.get(2), BundleDependencyMatcher.bundleDependencyMatcher("library-c", RUNTIME, true));
    assertThat(testClasspathPlusProvided.get(3), BundleDependencyMatcher.bundleDependencyMatcher("library-d", SYSTEM, true));
    assertThat(testClasspathPlusProvided.get(4), BundleDependencyMatcher.bundleDependencyMatcher("test-library-a", TEST, true));

    List<BundleDependency> testClasspath =
        mavenClient.resolveBundleDescriptorDependencies(true, false, artifact);

    assertThat(testClasspath, hasSize(4));

    assertThat(testClasspath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(testClasspath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-c", RUNTIME, true));
    assertThat(testClasspath.get(2), BundleDependencyMatcher.bundleDependencyMatcher("library-d", SYSTEM, true));
    assertThat(testClasspath.get(3), BundleDependencyMatcher.bundleDependencyMatcher("test-library-a", TEST, true));

    List<BundleDependency> providedClasspath =
        mavenClient.resolveBundleDescriptorDependencies(false, true, artifact);

    assertThat(providedClasspath, hasSize(4));

    assertThat(providedClasspath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(providedClasspath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-b", PROVIDED, false));
    assertThat(providedClasspath.get(2), BundleDependencyMatcher.bundleDependencyMatcher("library-c", RUNTIME, true));
    assertThat(providedClasspath.get(3), BundleDependencyMatcher.bundleDependencyMatcher("library-d", SYSTEM, true));

    List<BundleDependency> runtimeClasspath =
        mavenClient.resolveBundleDescriptorDependencies(false, false, artifact);

    assertThat(runtimeClasspath, hasSize(3));

    assertThat(runtimeClasspath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(runtimeClasspath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-c", RUNTIME, true));
    assertThat(runtimeClasspath.get(2), BundleDependencyMatcher.bundleDependencyMatcher("library-d", SYSTEM, true));
  }

  @Test
  public void resolveArtifactButNotPluginDependencies() {
    repositoryFolder.addArtifacts(artifact("dependency-a").dependencies(
                                                                        dependency("library-a").scope("provided"),
                                                                        dependency("library-b").scope("provided").dependencies(
                                                                                                                               dependency("library-b1")
                                                                                                                                   .scope("provided"),
                                                                                                                               dependency("library-b2")),
                                                                        dependency("connector-c").classifier(MULE_PLUGIN)
                                                                            .dependencies(
                                                                                          dependency("library-c1")),
                                                                        dependency("connector-d").classifier(MULE_PLUGIN),
                                                                        dependency("library-e").scope("test").dependencies(
                                                                                                                           dependency("library-e1")))
        .build());

    List<BundleDependency> classpath =
        mavenClient.resolveBundleDescriptorDependencies(true, true, getDescriptor("dependency-a"));

    assertThat(classpath, hasSize(7));

    assertThat(classpath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", PROVIDED, false));
    assertThat(classpath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-b", PROVIDED, false));
    assertThat(classpath.get(2), BundleDependencyMatcher.bundleDependencyMatcher("library-b2", PROVIDED, false));
    assertThat(classpath.get(3), BundleDependencyMatcher.bundleDependencyMatcher("connector-c", COMPILE, true));
    assertThat(classpath.get(3).getDescriptor().getClassifier().isPresent(), is(true));
    assertThat(classpath.get(3).getDescriptor().getClassifier().get(), is(MULE_PLUGIN));
    assertThat(classpath.get(4), BundleDependencyMatcher.bundleDependencyMatcher("connector-d", COMPILE, true));
    assertThat(classpath.get(4).getDescriptor().getClassifier().isPresent(), is(true));
    assertThat(classpath.get(4).getDescriptor().getClassifier().get(), is(MULE_PLUGIN));
    assertThat(classpath.get(5), BundleDependencyMatcher.bundleDependencyMatcher("library-e", TEST, true));
    assertThat(classpath.get(6), BundleDependencyMatcher.bundleDependencyMatcher("library-e1", TEST, true));
  }

  @Test
  public void resolveArtifactButNotIncludeOptionals() {
    repositoryFolder.addArtifacts(artifact("dependency-a").dependencies(
                                                                        dependency("library-a").optional("true"),
                                                                        dependency("library-b")
                                                                            .dependencies(dependency("library-b1")
                                                                                .optional("true"), dependency("library-b2")),
                                                                        dependency("connector-c").classifier(MULE_PLUGIN),
                                                                        dependency("library-d").scope("test")
                                                                            .dependencies(dependency("library-e1")
                                                                                .optional("true")),
                                                                        dependency("library-e")
                                                                            .dependencies(dependency("library-e1")
                                                                                .dependencies(dependency("library-e2"))))
        .build());

    List<BundleDependency> classpath =
        mavenClient.resolveBundleDescriptorDependencies(true, true, getDescriptor("dependency-a"));

    assertThat(classpath, hasSize(8));

    assertThat(classpath.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(classpath.get(1), BundleDependencyMatcher.bundleDependencyMatcher("library-b", COMPILE, true));
    assertThat(classpath.get(2), BundleDependencyMatcher.bundleDependencyMatcher("library-b2", COMPILE, true));

    assertThat(classpath.get(3), BundleDependencyMatcher.bundleDependencyMatcher("connector-c", COMPILE, true));
    assertThat(classpath.get(3).getDescriptor().getClassifier().isPresent(), is(true));
    assertThat(classpath.get(3).getDescriptor().getClassifier().get(), is(MULE_PLUGIN));

    assertThat(classpath.get(4), BundleDependencyMatcher.bundleDependencyMatcher("library-d", TEST, true));
    assertThat(classpath.get(5), BundleDependencyMatcher.bundleDependencyMatcher("library-e", COMPILE, true));
    assertThat(classpath.get(6), BundleDependencyMatcher.bundleDependencyMatcher("library-e1", COMPILE, true));
    assertThat(classpath.get(7), BundleDependencyMatcher.bundleDependencyMatcher("library-e2", COMPILE, true));
  }

  @Test
  public void resolveChainOfPluginDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("artifact").dependencies(
                                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                                        .dependencies(
                                                                                      dependency("connector-b")
                                                                                          .classifier(MULE_PLUGIN).dependencies(
                                                                                                                                dependency("connector-c")
                                                                                                                                    .classifier(MULE_PLUGIN))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, getDescriptor("artifact"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, contains("connector-a", "connector-b", "connector-c"));
  }

  @Test
  public void resolveArtifactDependenciesFromPomFile() throws MalformedURLException {
    repositoryFolder.addArtifacts(
                                  artifact("artifact").dependencies(
                                                                    dependency("connector-a").classifier(MULE_PLUGIN))
                                      .build());

    BundleDependency bundleDependency = mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId("artifact")
        .setVersion(VERSION)
        .setType("pom")
        .build());
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(toFile(bundleDependency.getBundleUri().toURL()),
                                                false,
                                                false,
                                                of(repositoryFolder.getRoot()),
                                                empty());

    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());
    assertThat(result, contains("connector-a"));
  }

  @Test
  public void resolvePluginDependenciesDoesNotIncludeTransitiveDependenciesOfOtherPlugins() {
    repositoryFolder.addArtifacts(artifact("connector-a").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("connector-b")
                          .classifier(MULE_PLUGIN)
                          .dependencies(
                                        dependency("library")))
        .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector-a");
    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(true, httpBundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat(result, hasItem("connector-b"));
    assertThat(result, not(hasItem("library")));
  }

  @Test
  public void resolveNonPluginDependenciesWithoutTestDependencies() {
    repositoryFolder.addArtifacts(artifact("connector").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("library-a")
                          .dependencies(dependency("library-a-dep")),
                      dependency("test-library")
                          .scope("test"))
        .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, httpBundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat("Direct non-plugin dependencies are not returned", result, hasItem("library-a"));
    assertThat("Direct non-plugin dependencies' dependencies are not returned", result,
               hasItem("library-a-dep"));
    assertThat("Test dependencies are returned (they shouldn't)", result, not(hasItem("test-library")));
  }

  @Test
  public void resolveNonPluginDependenciesWithRuntimeDependencies() {
    repositoryFolder.addArtifacts(artifact("connector").classifier(MULE_PLUGIN)
        .dependencies(
                      dependency("library-a")
                          .dependencies(dependency("library-a-dep")),
                      dependency("runtime-library")
                          .scope("runtime"))
        .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, httpBundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    assertThat("Direct non-plugin dependencies are not returned", result, hasItem("library-a"));
    assertThat("Direct non-plugin dependencies' dependencies are not returned", result,
               hasItem("library-a-dep"));
    assertThat("Runtime dependencies are not returned", result, hasItem("runtime-library"));
  }

  @Test
  public void resolveNonPluginDependenciesWithTestDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-a-dep")),
                                                    dependency("test-library")
                                                        .scope("test"))
                                      .build());

    BundleDescriptor httpBundleDescriptor = getDescriptor("connector");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, httpBundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    // direct dependencies
    assertThat("Direct non-plugin dependencies are not returned", result, hasItem("library-a"));
    assertThat("Direct non-plugin dependencies' dependencies are not returned", result,
               hasItem("library-a-dep"));
    assertThat("Test dependencies are not returned", result, hasItem("test-library"));
  }

  @Test
  public void pluginsWithLibraryDependenciesWithTransitiveDependencies() throws Exception {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                                                              dependency("library-a")
                                                                                                                                  .dependencies(
                                                                                                                                                dependency("transitive-library")
                                                                                                                                                    .version("1.0.0"))),
                                                               dependency("connector-b").classifier(MULE_PLUGIN).dependencies(
                                                                                                                              dependency("library-b")
                                                                                                                                  .dependencies(
                                                                                                                                                dependency("transitive-library")
                                                                                                                                                    .version("2.0.0"))))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);

    assertThat(bundleDependencies, hasSize(2));

    BundleDependency connectorADependency = bundleDependencies.get(0);
    BundleDependency connectorBDependency = bundleDependencies.get(1);

    assertThat(connectorADependency.getDescriptor().getArtifactId(), is(equalTo("connector-a")));
    assertThat(connectorADependency.getTransitiveDependencies().get(0).getDescriptor().getArtifactId(), is(equalTo("library-a")));
    assertThat(connectorADependency.getTransitiveDependencies().get(0).getTransitiveDependencies().get(0).getDescriptor()
        .getArtifactId(), is(equalTo("transitive-library")));
    assertThat(connectorADependency.getTransitiveDependencies().get(0).getTransitiveDependencies().get(0).getDescriptor()
        .getVersion(), is(equalTo("1.0.0")));

    assertThat(connectorBDependency.getDescriptor().getArtifactId(), is(equalTo("connector-b")));
    assertThat(connectorBDependency.getTransitiveDependencies().get(0).getDescriptor().getArtifactId(), is(equalTo("library-b")));
    assertThat(connectorBDependency.getTransitiveDependencies().get(0).getTransitiveDependencies().get(0).getDescriptor()
        .getArtifactId(), is(equalTo("transitive-library")));
    assertThat(connectorBDependency.getTransitiveDependencies().get(0).getTransitiveDependencies().get(0).getDescriptor()
        .getVersion(), is(equalTo("2.0.0")));
  }

  @Test
  public void resolveTransitiveDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-a-dep")),
                                                    dependency("library-b")
                                                        .dependencies(dependency("library-b-dep1"),
                                                                      dependency("library-b-dep2")
                                                                          .dependencies(dependency("library-b-dep2-dep"))))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, bundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    // direct dependencies
    assertThat("Not all dependencies are listed", result,
               containsInAnyOrder("library-a", "library-a-dep", "library-b", "library-b-dep1", "library-b-dep2",
                                  "library-b-dep2-dep"));
    assertTransitiveDependencies(bundleDependencies, "Library a does not have all transitive dependencies", "library-a",
                                 "library-a-dep");

    assertTransitiveDependencies(bundleDependencies, "Library b does not have all transitive dependencies", "library-b",
                                 "library-b-dep1", "library-b-dep2");

    assertTransitiveDependencies(bundleDependencies, "Library b-dep2 does not have all transitive dependencies", "library-b-dep2",
                                 "library-b-dep2-dep");
  }

  @Test
  public void resolveTransitiveDependenciesWithResolvedLibraryFromNonSharedBranch() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-dep").version("1.0")),
                                                    dependency("library-b")
                                                        .dependencies(dependency("library-dep").version("2.0")))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, bundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    // direct dependencies
    assertThat("Not all dependencies are listed", result,
               containsInAnyOrder("library-a", "library-dep", "library-b"));

    assertTransitiveDependencies(bundleDependencies, "Library a does not have all transitive dependencies", "library-a",
                                 "library-dep");

    assertTransitiveDependencies(bundleDependencies, "Library b does not have all transitive dependencies", "library-b",
                                 "library-dep");
  }

  @Test
  public void resolveTransitiveDependenciesWithCycles() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-b")),
                                                    dependency("library-b")
                                                        .dependencies(dependency("library-a")))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, bundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    // direct dependencies
    assertThat("Not all dependencies are listed", result,
               containsInAnyOrder("library-a", "library-b"));

    // weird behaviour by Aether but not relevant for the current supported cases.
    assertTransitiveDependencies(bundleDependencies, "Library a should not have transitive dependencies", "library-a");
    assertTransitiveDependencies(bundleDependencies, "Library b should have transitive dependencies", "library-b",
                                 "library-a");
  }

  @Test
  public void resolveTransitiveDependenciesWithCyclesOnNestedDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("library-c")
                                                            .dependencies(dependency("library-d"))),
                                                    dependency("library-b")
                                                        .dependencies(dependency("library-d")
                                                            .dependencies(dependency("library-c"))))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(true, bundleDescriptor);
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());

    // direct dependencies
    assertThat("Not all dependencies are listed", result,
               containsInAnyOrder("library-a", "library-b", "library-c", "library-d"));

    assertTransitiveDependencies(bundleDependencies, "Library a does not have all transitive dependencies", "library-a",
                                 "library-c");

    assertTransitiveDependencies(bundleDependencies, "Library b does not have all transitive dependencies", "library-b",
                                 "library-d");

    // weird behaviour by Aether but not relevant for the current supported cases.
    assertTransitiveDependencies(bundleDependencies, "Library c should not have transitive dependencies", "library-c");
    assertTransitiveDependencies(bundleDependencies, "Library d should have transitive dependencies", "library-d", "library-c");
  }

  private void assertTransitiveDependencies(List<BundleDependency> bundleDependencies, String failureMessage,
                                            String libraryContainingDependencies, String... expectedDependencies) {
    assertThat(failureMessage, bundleDependencies.stream()
        .filter(bundleDependency -> bundleDependency.getDescriptor().getArtifactId().equals(libraryContainingDependencies))
        .map(bundleDependency -> bundleDependency.getTransitiveDependencies())
        .flatMap(List::stream)
        .map(bundleDependency -> bundleDependency.getDescriptor().getArtifactId())
        .collect(toList()), containsInAnyOrder(expectedDependencies));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("connector").version("1.1").classifier(MULE_PLUGIN).build(),
                                  artifact("connector").version("1.0").classifier(MULE_PLUGIN).build());

    BundleDescriptor connector10 = getDescriptor("connector"); // 1.0
    BundleDescriptor connector11 = getDescriptor("connector", "1.1");

    List<String> result =
        mavenClient.resolvePluginBundleDescriptorsDependencies(Arrays.asList(connector11, connector10)).stream()
            .map(dep -> dep.getDescriptor())
            .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());
    assertThat(result.size(), is(1));
    assertThat(result, hasItem("connector:1.1"));

    result = mavenClient.resolvePluginBundleDescriptorsDependencies(Arrays.asList(connector10, connector11)).stream()
        .map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());
    assertThat(result.size(), is(1));
    assertThat(result, hasItem("connector:1.1"));
  }

  @Test
  public void shouldAvoidResolvingDependenciesForPluginDiscardedAfterByDependencyResolutionConflictResolver() {
    repositoryFolder.addArtifacts(artifact("connector-a").version("1.1")
        .classifier(MULE_PLUGIN)
        .dependencies(dependency("connector-c")
            .version("2.0")
            .classifier(MULE_PLUGIN))
        .build(),
                                  artifact("connector-b").version("1.0")
                                      .classifier(MULE_PLUGIN)
                                      .dependencies(dependency("connector-c")
                                          .version("2.1")
                                          .classifier(MULE_PLUGIN))
                                      .build());

    // Should not resolve connector-c:2.0 mule-plugin.jar file as the conflict resolver defines the bigger version (semver) as
    // winner.
    repositoryFolder.removeArtifact(artifact("connector-c").version("2.0").classifier(MULE_PLUGIN).type("jar").build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                               Arrays.asList(getDescriptor("connector-a", "1.1"),
                                                                             getDescriptor("connector-b", "1.0")));

    List<String> result = bundleDependencies.stream()
        .map(dep -> dep.getDescriptor().getArtifactId() + ":" + dep.getDescriptor().getVersion())
        .collect(toList());

    // direct dependencies
    assertThat(result, hasItems("connector-a:1.1", "connector-b:1.0", "connector-c:2.1"));
  }

  @Test
  public void shouldAvoidCollectDomainDependency() throws MalformedURLException {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-a"),
                                                    dependency("domain").classifier(MULE_DOMAIN).scope("provided"))
                                      .build());

    BundleDescriptor bundleDescriptor = getDescriptor("app");
    File artifactFile =
        toFile(mavenClient.resolveBundleDescriptor(bundleDescriptor).getBundleUri().toURL());

    repositoryFolder.removeArtifact(artifact("domain").type("pom").build());
    repositoryFolder.removeArtifact(artifact("domain").classifier(MULE_DOMAIN).type("jar").build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(artifactFile, false, false, of(repositoryFolder.getRoot()), empty());

    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor().getArtifactId()).collect(toList());
    assertThat(result.size(), is(1));
    assertThat(result, contains("library-a"));

    bundleDependencies =
        mavenClient.resolveArtifactDependencies(artifactFile, false, true, of(repositoryFolder.getRoot()), empty());

    assertThat(bundleDependencies, hasSize(2));
    assertThat(bundleDependencies.get(0), BundleDependencyMatcher.bundleDependencyMatcher("library-a", COMPILE, true));
    assertThat(bundleDependencies.get(1), BundleDependencyMatcher.bundleDependencyMatcher("domain", PROVIDED, false));
  }

  @Test
  public void resolutionFailsWithIncompatibleVersions() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("2.0")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.0").classifier(MULE_PLUGIN).build());

    expectedException.expect(RuntimeException.class);
    mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                           Arrays.asList(getDescriptor("connector-a"),
                                                                         getDescriptor("connector-b")));
  }

  @Test
  public void resolutionFailsWithIncompatibleVersions_1() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("2.0")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.0").classifier(MULE_PLUGIN).build());

    expectedException.expect(RuntimeException.class);
    mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                           Arrays.asList(getDescriptor("connector-b"),
                                                                         getDescriptor("connector-a")));
  }

  @Test
  public void resolutionFailsWithIncompatibleVersions_2() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .classifier(MULE_PLUGIN)
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-c")
                                                                                                                     .version("2.0")
                                                                                                                     .classifier(MULE_PLUGIN)))
                                      .build(),
                                  artifact("connector-c").version("1.0").classifier(MULE_PLUGIN).build());

    expectedException.expect(RuntimeException.class);
    mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                           Arrays.asList(getDescriptor("connector-a"),
                                                                         getDescriptor("connector-c")));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPluginsConsideringDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("1.1")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.0").classifier(MULE_PLUGIN).build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                               Arrays.asList(getDescriptor("connector-b"),
                                                                             getDescriptor("connector-a")));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION, "connector-b:1.1"));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPluginsConsideringDependencies_1() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .version("1.0")
                                                                                                   .classifier(MULE_PLUGIN))
                                      .build(),
                                  artifact("connector-b").version("1.1").classifier(MULE_PLUGIN).build());

    BundleDescriptor descriptorB = getDescriptor("connector-b", "1.1");
    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(Arrays.asList(descriptorB, getDescriptor("connector-a")));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:1.1", "connector-a:" + VERSION));
  }

  @Test
  public void resolutionReturnsCompatibleVersionsOfPluginsConsideringTransitiveDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN).dependencies(
                                                                                               dependency("connector-b")
                                                                                                   .classifier(MULE_PLUGIN)
                                                                                                   .dependencies(
                                                                                                                 dependency("connector-c")
                                                                                                                     .version("1.1")
                                                                                                                     .classifier(MULE_PLUGIN)))
                                      .build(),
                                  artifact("connector-c").version("1.0").classifier(MULE_PLUGIN).build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolvePluginBundleDescriptorsDependencies(
                                                               Arrays.asList(getDescriptor("connector-a"),
                                                                             getDescriptor("connector-c")));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION, "connector-b:" + VERSION, "connector-c:1.1"));
  }

  @Test
  public void fromPluginsOnlyNestedPluginsShouldBeIncludedInDependencyGraph() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("connector-b")
                                                        .classifier(MULE_PLUGIN)
                                                        .dependencies(
                                                                      dependency("connector-c")
                                                                          .classifier(MULE_PLUGIN),
                                                                      dependency("third-party").version("1.0")),
                                                    dependency("internal-library").version("1.0")
                                                        .dependencies(
                                                                      dependency("third-party").version("1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:" + VERSION,
                                "connector-c:" + VERSION,
                                "internal-library:1.0",
                                "third-party:1.0"));
  }

  @Test
  public void pluginsDoNotParticipateOnMavenVersionResolution() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("connector-b")
                                                        .classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("third-party").version("2.0")),
                                                    dependency("internal-library").version("1.0")
                                                        .dependencies(dependency("third-party").version("1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:" + VERSION,
                                "internal-library:1.0",
                                "third-party:1.0"));
  }

  @Test
  public void pluginDependsOnJarThatDependsOnPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("library-a")
                                                        .dependencies(dependency("connector-b").classifier(MULE_PLUGIN)
                                                            .dependencies(dependency("library-d").version("1.1"))),
                                                    dependency("library-b").version("1.0")
                                                        .dependencies(dependency("library-c").version("1.0")
                                                            .dependencies(dependency("library-d").version("1.0"))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-a:" + VERSION, "connector-b:" + VERSION,
                                "library-b:" + VERSION,
                                "library-c:" + VERSION,
                                "library-d:" + VERSION));
  }

  @Test
  public void resolveBundleDependenciesFromJarFileShouldStillUsePomArtifact() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a").version(VERSION).classifier(MULE_PLUGIN)
                                      .dependencies(
                                                    dependency("library-a"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a", VERSION, "jar", MULE_PLUGIN));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-a:" + VERSION));
  }

  @Test
  public void applicationThatDependsOnPluginThatDependsOnJarThatDependsOnPlugin() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(
                                                                      dependency("library-a").version("1.1")
                                                                          .dependencies(dependency("connector-b")
                                                                              .classifier(MULE_PLUGIN)
                                                                              .dependencies(
                                                                                            dependency("library-d").version("1.1")
                                                                                                .dependencies(
                                                                                                              dependency("library-e")
                                                                                                                  .version("1.1")
                                                                                                                  .dependencies(
                                                                                                                                dependency("connector-d")
                                                                                                                                    .classifier(MULE_PLUGIN))),
                                                                                            dependency("connector-c")
                                                                                                .classifier(MULE_PLUGIN)))),
                                                    dependency("library-b").version("1.0")
                                                        .dependencies(
                                                                      dependency("library-c").version("1.0")
                                                                          .dependencies(
                                                                                        dependency("library-d").version("1.0"))),
                                                    dependency("library-a").version("1.0"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION,
                                "connector-b:" + VERSION,
                                "connector-d:" + VERSION,
                                "connector-c:" + VERSION,
                                "library-b:1.0",
                                "library-c:1.0",
                                "library-d:1.0",
                                "library-a:1.0"));
  }

  @Test
  public void appWithPluginsWithSimpleCyclicDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)),
                                                    dependency("connector-c").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-a").classifier(MULE_PLUGIN)))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION,
                                "connector-c:" + VERSION));
  }

  @Test
  public void resolvePluginRootNodeShouldNotRemoveDirectDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("connector-a")
                                      .dependencies(
                                                    dependency("connector-b").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)),
                                                    dependency("library-1"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("connector-a"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-b:" + VERSION,
                                "connector-c:" + VERSION,
                                "library-1:" + VERSION));
  }

  @Test
  public void appWithPluginsWithComplexCyclicDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .dependencies(
                                                                          dependency("connector-d").classifier(MULE_PLUGIN)
                                                                              .dependencies(dependency("connector-a")
                                                                                  .classifier(MULE_PLUGIN)))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION,
                                "connector-c:" + VERSION,
                                "connector-d:" + VERSION));
  }

  @Test
  public void appWithMultipleVersionsOfPlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")),
                                                    dependency("connector-b").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-a:" + VERSION,
                                "connector-b:" + VERSION,
                                "connector-c:1.1.0"));
  }

  @Test
  public void appWithIncompatibleVersionsOfPlugins() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")),
                                                    dependency("connector-b").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("2.1.0")))
                                      .build());

    expectedException.expect(IncompatibleMulePluginVersionResolutionException.class);
    expectedException
        .expectMessage(containsString("Incompatible Mule plugins versions among: org.mule:connector-c:jar:mule-plugin:2.1.0, org.mule:connector-c:jar:mule-plugin:1.0.2"));

    mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-1:" + VERSION,
                                "connector-a:" + VERSION,
                                "connector-c:1.1.0"));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependenciesDifferentOrder() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0")),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2")))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-1:" + VERSION,
                                "connector-c:1.1.0",
                                "connector-a:" + VERSION));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependenciesTwoLevels() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.3.0"))),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.1.0"))))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-1:" + VERSION,
                                "connector-c:1.1.0",
                                "connector-d:2.3.0",
                                "connector-a:" + VERSION));
  }

  @Test
  public void appWithMultipleVersionsOfPluginsByTransitiveDependenciesTwoLevelsAndRootDefinitionOrSameConnector() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("library-1")
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.1.0").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.1.0"))),
                                                    dependency("connector-a").classifier(MULE_PLUGIN)
                                                        .dependencies(dependency("connector-c").classifier(MULE_PLUGIN)
                                                            .version("1.0.2").dependencies(dependency("connector-d")
                                                                .classifier(MULE_PLUGIN).version("2.3.0"))),
                                                    dependency("connector-d").classifier(MULE_PLUGIN).version("2.2.0"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("library-1:" + VERSION,
                                "connector-c:1.1.0",
                                "connector-a:" + VERSION,
                                "connector-d:2.2.0"));
  }

  @Test
  public void appSamePluginDefinedBothTimesMavenTakesLastDefinitionOfDependency() {
    repositoryFolder.addArtifacts(
                                  artifact("app")
                                      .dependencies(
                                                    dependency("connector-1").classifier(MULE_PLUGIN)
                                                        .version("1.1.0"),
                                                    dependency("connector-1").classifier(MULE_PLUGIN)
                                                        .version("1.0.2"))
                                      .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, contains("connector-1:1.0.2"));
  }

  @Test
  public void transitiveApiDependenciesDontOverrideAppDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("api-a").classifier(RAML).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("fragment")
                                                                                     .classifier(RAML_FRAGMENT)
                                                                                     .version("1.0.0"),
                                                                                 dependency("library").version("2.0.0")),
                                                               dependency("library").version("1.0.0"))
                                      .build());

    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));

    assertThat(bundleDependencies, hasSize(3));

    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, hasItems(
                                "api-a:1.0.0",
                                "fragment:1.0.0",
                                "library:1.0.0"));
  }

  @Test
  public void ifPluginWinsAlsoItsDependencies() {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("plugin").version("1.0.0").classifier(MULE_PLUGIN)
                                                                   .dependencies(
                                                                                 dependency("library-a").version("1.1.0"),
                                                                                 dependency("library-b").version("1.1.0")),
                                                               dependency("synthetic").version("1.0.0").dependencies(
                                                                                                                     dependency("plugin")
                                                                                                                         .version("1.1.0")
                                                                                                                         .classifier(MULE_PLUGIN)
                                                                                                                         .dependencies(
                                                                                                                                       dependency("library-a")
                                                                                                                                           .version("1.0.0"))))
                                      .build());
    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));
    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, hasSize(2));
    assertThat(result, hasItems(
                                "plugin:1.1.0",
                                "synthetic:1.0.0"));

    BundleDependency pluginDep =
        bundleDependencies.stream().filter(d -> d.getDescriptor().getArtifactId().equals("plugin")).findFirst().get();
    assertThat(pluginDep.getTransitiveDependencies(), hasSize(1));
    assertThat(pluginDep.getTransitiveDependencies().get(0).getDescriptor().getArtifactId(), equalTo("library-a"));
    assertThat(pluginDep.getTransitiveDependencies().get(0).getDescriptor().getVersion(), equalTo("1.0.0"));

  }

  @Test
  public void sameDependenciesInDifferentPlugins() {
    final String version = "1.0.0";
    ArtifactCreator.DependencyBuilder libraryBDep = dependency("library-b").version(version);
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("plugin-a").classifier(MULE_PLUGIN).version(version)
                                                                   .dependencies(
                                                                                 dependency("library-a").version(version)
                                                                                     .dependencies(
                                                                                                   libraryBDep)),
                                                               dependency("plugin-b").classifier(MULE_PLUGIN).version(version)
                                                                   .dependencies(
                                                                                 dependency("library-a").version(version)
                                                                                     .dependencies(
                                                                                                   libraryBDep)
                                                                                     .exclusions(exclusion("library-b"))))
                                      .build());

    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));

    assertThat(bundleDependencies, hasSize(2));
    BundleDependency pluginA = bundleDependencies.get(0);
    BundleDependency pluginB = bundleDependencies.get(1);

    assertThat(pluginA.getDescriptor().getArtifactId(), equalTo("plugin-a"));
    assertThat(pluginB.getDescriptor().getArtifactId(), equalTo("plugin-b"));

    assertThat(pluginA.getTransitiveDependencies().get(0).getTransitiveDependencies(), hasSize(1));
    assertThat(pluginB.getTransitiveDependencies().get(0).getTransitiveDependencies(), hasSize(0));
  }

  @Test
  public void sameDependenciesDifferentPluginsWithOtherWinner() {
    ArtifactCreator.DependencyBuilder artifactD = dependency("artifact-d").version("1.0.0").dependencies(
                                                                                                         dependency("artifact-a")
                                                                                                             .version("1.0.0"));

    ArtifactCreator.DependencyBuilder artifactDExcludingA = dependency("artifact-d").version("1.0.0").dependencies(
                                                                                                                   dependency("artifact-a")
                                                                                                                       .version("1.0.0"))
        .exclusions(exclusion("artifact-a"));

    ArtifactCreator.DependencyBuilder artifactC = dependency("artifact-c").version("1.0.0").dependencies(
                                                                                                         artifactDExcludingA);
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("plugin-a").classifier(MULE_PLUGIN).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("plugin-b").classifier(MULE_PLUGIN)
                                                                                     .version("1.0.0").dependencies(
                                                                                                                    artifactC),
                                                                                 artifactD),
                                                               dependency("plugin-c").classifier(MULE_PLUGIN).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("plugin-b").classifier(MULE_PLUGIN)
                                                                                     .version("1.1.0").dependencies(
                                                                                                                    artifactC)))
                                      .build());

    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));

    assertThat(bundleDependencies, hasSize(3));

    BundleDependency pluginA = bundleDependencies.get(0);
    assertThat(pluginA.getDescriptor().getArtifactId(), equalTo("plugin-a"));

    BundleDependency dInA = pluginA.getTransitiveDependencies().get(1);
    assertThat(dInA.getDescriptor().getArtifactId(), equalTo("artifact-d"));
    assertThat(dInA.getTransitiveDependencies().get(0).getDescriptor().getArtifactId(), equalTo("artifact-a"));

    BundleDependency pluginB = bundleDependencies.get(2);
    assertThat(pluginB.getDescriptor().getArtifactId(), equalTo("plugin-b"));
    assertThat(pluginB.getDescriptor().getVersion(), equalTo("1.1.0"));

    BundleDependency dInB = pluginB.getTransitiveDependencies().get(0).getTransitiveDependencies().get(0);
    assertThat(dInB.getDescriptor().getArtifactId(), equalTo("artifact-d"));
    assertThat(dInB.getTransitiveDependencies(), hasSize(0));
  }


  @Test
  public void sameApiDependencyInChainForRAML() {
    assertSameApiDependencyInChain(RAML, RAML);
    assertSameApiDependencyInChain(RAML, RAML_FRAGMENT);
  }

  @Test
  public void sameApiDependencyInChainForOAS() {
    assertSameApiDependencyInChain(OAS, OAS);
  }

  @Test
  public void sameApiDependencyInChainForWSDL() {
    assertSameApiDependencyInChain(WSDL, WSDL);
  }

  @Test
  public void multipleApiLevels() {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("api-a").version("1.0.0").classifier(RAML)
                                                                   .dependencies(
                                                                                 dependency("fragment").classifier(RAML_FRAGMENT)
                                                                                     .version("1.0.0"),
                                                                                 dependency("api-b").classifier(RAML)
                                                                                     .version("1.0.0").dependencies(
                                                                                                                    dependency("fragment")
                                                                                                                        .classifier(RAML_FRAGMENT)
                                                                                                                        .version("1.1.0"))))
                                      .build()


    );
    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));

    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, hasSize(4));

    assertThat(result, hasItems(
                                "api-a:1.0.0",
                                "fragment:1.0.0",
                                "api-b:1.0.0",
                                "fragment:1.1.0"));

  }

  private void assertSameApiDependencyInChain(String apiClassifier, String dependencyClassifier) {
    repositoryFolder.addArtifacts(
                                  artifact("app").dependencies(
                                                               dependency("api-a").classifier(apiClassifier).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("fragment")
                                                                                     .classifier(dependencyClassifier)
                                                                                     .version("1.0.0")),
                                                               dependency("api-b").classifier(apiClassifier).version("1.0.0")
                                                                   .dependencies(
                                                                                 dependency("fragment")
                                                                                     .classifier(dependencyClassifier)
                                                                                     .version("2.0.0")))
                                      .build());

    List<BundleDependency> bundleDependencies = mavenClient.resolveBundleDescriptorDependencies(false, getDescriptor("app"));

    assertThat(bundleDependencies, hasSize(4));

    List<String> result = bundleDependencies.stream().map(dep -> dep.getDescriptor())
        .map(desc -> desc.getArtifactId() + ":" + desc.getVersion()).collect(toList());

    assertThat(result, hasItems(
                                "api-a:1.0.0",
                                "api-b:1.0.0",
                                "fragment:1.0.0",
                                "fragment:2.0.0"));
  }

  private static class BundleDependencyMatcher extends BaseMatcher<BundleDependency> {

    private String groupId;
    private String artifactId;
    private String version;
    private BundleScope scope;
    private boolean hasUri;

    static BundleDependencyMatcher bundleDependencyMatcher(String artifactId, BundleScope scope, boolean hasUri) {
      return new BundleDependencyMatcher(null, artifactId, null, scope, hasUri);
    }

    private BundleDependencyMatcher(String groupId, String artifactId, String version,
                                    BundleScope scope, boolean hasUri) {
      this.groupId = groupId;
      this.artifactId = artifactId;
      this.version = version;
      this.scope = scope;
      this.hasUri = hasUri;
    }

    @Override
    public boolean matches(Object o) {
      BundleDependency bundleDependency = (BundleDependency) o;
      final BundleDescriptor bundleDescriptor = bundleDependency.getDescriptor();
      boolean matches = true;
      if (groupId != null) {
        matches &= bundleDescriptor.getGroupId().equals(groupId);
      }
      if (artifactId != null) {
        matches &= bundleDescriptor.getArtifactId().equals(artifactId);
      }
      if (version != null) {
        matches &= bundleDescriptor.getVersion().equals(version);
      }
      if (hasUri) {
        matches &= bundleDependency.getBundleUri() != null;
      } else {
        matches &= bundleDependency.getBundleUri() == null;
      }
      return matches && bundleDependency.getScope().equals(scope);
    }

    @Override
    public void describeTo(Description description) {
      description.appendText("BundleDependency does not match.");

    }
  }
}
